<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele De Nobili
 * @author Daniele Termini
 */

namespace Pongho\Updates;

use Exception;
use Symfony\Component\Process\Process;

final class Updater
{
    private string $site_path;
    private string $site_url;
    private string $site_dir;

    private string $old_path;
    private string $new_path;

    public function __construct(string $site_path, string $site_url, string $site_dir)
    {
        $this->site_path = rtrim($site_path, '/');
        $this->site_url = $site_url;
        $this->site_dir = $site_dir;

        $this->old_path = $this->site_path . '/_old_' . date('Ymd');
        $this->new_path = $this->site_path . '/_new_' . date('Ymd');
    }

    private function isMaintenanceMode(): bool
    {
        return file_exists($this->old_path . '/.htaccess');
    }

    private function isReadyForUpgrade(): bool
    {
        return file_exists($this->new_path . '/pongho');
    }

    /**
     * @throws Exception
     */
    public function activateMaintenance(): void
    {
        if ($this->isMaintenanceMode()) {
            return;
        }

        if (!file_exists($this->old_path)) {
            mkdir($this->old_path);
        }

        /**
         * Crea un backup del file .htaccess
         */
        if (false === copy($this->site_path . '/.htaccess', $this->old_path . '/.htaccess')) {
            throw new Exception('Non è stato possibile fare un backup del file <code>.htaccess</code>.');
        }

        /**
         * Crea il file maintenance.php
         */
        $maintenance_path = $this->site_path . '/maintenance.php';
        $maintenance_code = file_get_contents(__DIR__ . '/../resources/maintenance.php');
        if (false === file_put_contents($maintenance_path, $maintenance_code)) {
            throw new Exception('Non è stato possibile creare il file <code>maintenance.php</code>. Controlla i permessi di scrittura della root del sito.');
        }

        /**
         * Crea il file .htaccess con le regole di redirect al maintenance.php
         */
        $base = empty($this->site_dir) ? '/' : "/$this->site_dir/";
        $site_url = $this->site_url . $this->site_dir;

        $htaccess_path = $this->site_path . '/.htaccess';
        $htaccess_code = <<<HTACCESS
<ifmodule mod_rewrite.c>
	RewriteEngine On

	RewriteBase $base

	RewriteCond %{REMOTE_ADDR} !^185\.56\.156\.220
	RewriteCond %{REQUEST_URI} !^$this->site_dir/maintenance\.php$
	RewriteCond %{REQUEST_URI} !^$this->site_dir/updates/(.*)$
	RewriteRule ^(.*)$ $site_url/maintenance.php [R=307,L]

	RewriteRule ^pongho/$ index.php/pongho/$1 [PT,L]
	RewriteCond %{REQUEST_FILENAME} !-f
	RewriteCond %{REQUEST_FILENAME} !-d
	RewriteRule ^(.*)$ index.php/$1 [L,PT]
</ifmodule>
HTACCESS;

        if (false === file_put_contents($htaccess_path, $htaccess_code)) {
            throw new Exception('Non è stato possibile creare il file <code>.htaccess</code>. Controlla i permessi di scrittura della root del sito.');
        }
    }

    /**
     * @throws Exception
     */
    public function deactivateMaintenance(): void
    {
        if (!$this->isMaintenanceMode()) {
            return;
        }

        /**
         * Riscrivo il maintenance.php in modo che reindirizzi alla home del sito.
         */
        $maintenance_path = $this->site_path . '/maintenance.php';
        $maintenance_code = <<<PHP
<?php

header('HTTP/1.0 303 See Other');
header('Location: $this->site_url');

?>
PHP;
        if (false === file_put_contents($maintenance_path, $maintenance_code)) {
            throw new Exception('Non è stato possibile ripristinare il file <code>maintenance.php</code>. Controlla i permessi di scrittura della root del sito.');
        }

        /**
         * Ripristino il file .htaccess
         */
        $htaccess_backup_path = $this->old_path . '/.htaccess';
        $htaccess_path = $this->site_path . '/.htaccess';
        if (false === copy($htaccess_backup_path, $htaccess_path)) {
            throw new Exception("Non è stato possibile ripristinare il file <code>.htaccess</code>. La modalità di manutenzione deve essere disattivata manualmente copiando il file dalla cartella <code>$this->old_path</code>");
        }

        /**
         * Elimina il backup del file .htaccess per dichiarare disattivata la manutenzione.
         */
        unlink($this->old_path . '/.htaccess');
    }

    /**
     * @throws Exception
     */
    public function update(): void
    {
        if (!$this->isMaintenanceMode()) {
            throw new Exception('Il sito NON è in modalità di manutenzione!');
        }

        if (!$this->isReadyForUpgrade()) {
            throw new Exception('Non ci sono i file di Pongho nella cartella <code>_new_' . date('Ymd') . '</code>!');
        }

        if (!file_exists($this->old_path . '/content')) {
            mkdir($this->old_path . '/content', 0o777, true);;
        }

        /**
         * Crea un backup dei file
         */
        $paths = [
            // Backup del vecchio
            $this->site_path . '/pongho'          => $this->old_path . '/pongho',
            $this->site_path . '/content/Plugins' => $this->old_path . '/content/Plugins',
            $this->site_path . '/content/themes'  => $this->old_path . '/content/themes',

            // Copio i nuovi file
            $this->new_path . '/pongho'           => $this->site_path . '/pongho',
            $this->new_path . '/content/Plugins'  => $this->site_path . '/content/Plugins',
            $this->new_path . '/content/themes'   => $this->site_path . '/content/themes',
        ];

        foreach ($paths as $from => $to) {
            if (!file_exists($from)) {
                continue;
            }

            $process = Process::fromShellCommandline('mv "${:FROM}" "${:TO}"');
            $process->run(null, ['FROM' => $from, 'TO' => $to]);

            if (!$process->isSuccessful()) {
                throw new Exception($process->getErrorOutput());
            }
        }
    }

    public function showHome(): void
    {
        if ($this->isMaintenanceMode()) {
            if ($this->isReadyForUpgrade()) {
                $siteMode = '<p class="alert alert-primary">Il sito è in modalità di manutenzione e pronto per l’aggiornamento.</p>';
            } else {
                $siteMode = '<p class="alert alert-warning">Il sito è in modalità di manutenzione, ma manca la cartella <code>_new_' . date('Ymd') . '</code>.</p>';
            }
        } else {
            $siteMode = '<p class="alert alert-light">Il sito è in modalità normale.</p>';
        }

        $manualOperation = '';
        if ($this->isMaintenanceMode()) {
            if ($this->isReadyForUpgrade()) {
                $manualOperation .= '<a href="index.php?action=update" class="btn btn-primary">Aggiorna il sito</a>';
            }

            $manualOperation .= '<a href="update-db.php" class="btn btn-primary" target="_blank">Aggiorna il database</a>';
            $manualOperation .= '<a href="index.php?action=deactivate-maintenance" class="btn btn-primary">Disattiva la manutenzione</a>';
        } else {
            $manualOperation .= '<a href="index.php?action=activate-maintenance" class="btn btn-primary">Attiva la manutenzione</a>';
        }

        $home = <<<HTML
        <div class="mb-5">$siteMode</div>
        <div class="mb-5">
            <h3>Aggiornamento manuale</h3>
            <div class="list-group list-group-horizontal mb-3">
                <div class="list-group-item w-75 p-3">
                    <p>L’aggiornamento manuale è una procedura in 3 passaggi:</p>
                    <ol>
                        <li>Attiva la modalità di manutenzione</li>
                        <li>Aggiorna i file</li>
                        <li>Disattiva la modalità di manutenzione</li>
                    </ol>
                    <p>In questo modo è possibile caricare altri file (ad esempio template e plugin), aggiornare il database o effettuare dei controlli.</p>
                </div>
                <div class="list-group-item d-flex align-items-center w-25 p-3 bg-body-tertiary">
                    <div style="display: grid;gap: 0.5rem;flex-direction: column;justify-items: start;">$manualOperation</div>
                </div>
            </div>
        </div>

        <div class="mb-5">
            <h3>Aggiornamento automatico</h3>
            <div class="list-group list-group-horizontal mb-3">
                <div class="list-group-item w-75 p-3">
                    <p>L’aggiornamento automatico riduce il tempo di manutenzione al minimo, ma non è possibile effettuare altre operazioni.</p>
                </div>
                <div class="list-group-item d-flex align-items-center w-25 p-3 bg-body-tertiary">
                    <a href="index.php?action=fast-update" class="btn btn-secondary">Aggiorna subito</a>
                </div>
            </div>
        </div>
HTML;

        echo $this->renderPage($home);
    }

    public function showError(string $message): void
    {
        $html = <<<HTML
<div class="alert alert-danger mb-5" role="alert">
    <p>$message</p>
    <p class="m-0"><a href="index.php" class="btn btn-secondary">Torna indietro</a></p>
</div>
HTML;

        echo $this->renderPage($html);
    }

    private function renderPage(string $code): string
    {
        return <<<HTML
<!doctype html>
<html lang="it">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Pongho Updater</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet" integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH" crossorigin="anonymous">
</head>
<body>
    <header class="p-3 mb-3 text-bg-dark">
        <div class="container">
            <h1>Pongho Updater</h1>
            <p>Strumento di aiuto per l’aggiornamento dei siti in Pongho.</p>
        </div>
    </header>
    <div class="container mt-5">
        <main class="mb-5">$code</main>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js" integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz" crossorigin="anonymous"></script>
</body>
</html>
HTML;
    }
}
