<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Utilities;

use Application\Core\Utilities\Mailer;
use Application\Showcase\Model\Order;
use Pongho\Core\LocalizationInterface;
use Pongho\Template\View;
use Psr\Log\LoggerInterface;

/**
 * OrderNotification
 */
class OrderNotification
{
    /**
     * @var LocalizationInterface
     */
    protected $lang;

    /**
     * @var \Application\Core\Utilities\Mailer
     */
    protected $mailer;

    /**
     * Elenco delle variabili da passare al template.
     *
     * Di fatto, il servizio "global_view_vars".
     *
     * @var array
     */
    protected $view_vars;

    /**
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * @param LocalizationInterface $lang
     * @param Mailer                $mailer
     * @param array                 $view_vars
     * @param LoggerInterface       $logger
     */
    public function __construct(LocalizationInterface $lang, Mailer $mailer, array $view_vars, LoggerInterface $logger)
    {
        $this->lang = $lang;
        $this->mailer = $mailer;
        $this->view_vars = $view_vars;
        $this->logger = $logger;
    }

    /**
     * @param Order $order
     */
    public function send(Order $order)
    {
        $site = $order->site;

        $user_view = new View($site->getThemesPath('email/shop_order_to_user.php'));
        $user_view
            ->assignVars($this->view_vars)
            ->assignVars([
                'order'          => $order,
                'order_user_url' => $site->getPermalink() . 'shop/order/' . $order->id . '/',
            ]);

        $admin_view = new View($site->getThemesPath('email/shop_order_to_admin.php'));
        $admin_view
            ->assignVars($this->view_vars)
            ->assignVars([
                'order'           => $order,
                'order_admin_url' => pongho_url('/shop-orders/view/' . $order->id . '/?site=' . $site->id),
                'user_admin_url'  => pongho_url('/users/edit/' . $order->customer_id . '/?site=' . $site->id),
            ]);

        $orders_email = $site->getOption('orders_email', $site->getOption('company_email'));

        try {
            // Mail al cliente
            $this->mailer->send(
                [$orders_email => $site->getOption('company_name')],
                [$order->getCustomerEmail() => $order->getCustomerName()],
                sprintf($this->lang->get('email_shop_order_to_user_subject'), $site->getName(), $order->id),
                $this->mailer->content(
                    $site->getThemesPath('email/base.php'),
                    [
                        'CONTENT' => $user_view->render(),
                    ]
                )
            );

            $this->logger->debug(
                '[SEND ORDER EMAIL] Sent email to user.',
                [
                    'order' => $order->getId(),
                    'from'  => $orders_email,
                    'to'    => $order->getCustomerEmail(),
                ]
            );
        } catch (\Exception $e) {
            $this->logger->error(
                sprintf(
                    '[SEND ORDER EMAIL] Could not be send the order confirmation email to user: "%s"',
                    $e->getMessage()
                ),
                [
                    'order'     => $order->getId(),
                    'from'      => $orders_email,
                    'to'        => $order->getCustomerEmail(),
                    'exception' => $e,
                ]
            );
        }

        try {
            // Mail all'amministratore
            $this->mailer->send(
                [$orders_email => $site->getOption('company_name')],
                [$orders_email => $site->getOption('company_name')],
                sprintf($this->lang->get('email_shop_order_to_admin_subject'), $site->getName(), $order->id),
                $this->mailer->content(
                    $site->getThemesPath('email/base.php'),
                    [
                        'CONTENT' => $admin_view->render(),
                    ]
                ),
                [$order->getCustomerEmail() => $order->getCustomerName()]
            );

            $this->logger->debug(
                '[SEND ORDER EMAIL] Sent email to admin.',
                [
                    'order' => $order->getId(),
                    'from'  => $orders_email,
                    'to'    => $orders_email,
                ]
            );
        } catch (\Exception $e) {
            $this->logger->error(
                sprintf(
                    '[SEND ORDER EMAIL] Could not be send the order confirmation email to admin: "%s"',
                    $e->getMessage()
                ),
                [
                    'order'     => $order->getId(),
                    'from'      => $orders_email,
                    'to'        => $order->getCustomerEmail(),
                    'exception' => $e,
                ]
            );
        }
    }
}
