<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Utilities;

use Application\Showcase\Model\Order;
use Pongho\Core\LocalizationInterface;
use Pongho\Http\Request;
use Pongho\Menu\Item;
use Pongho\Menu\Menu;

/**
 * CheckoutBreadcrumbBuilder.
 */
class CheckoutBreadcrumbBuilder
{
    const ACTION_CART = 'cart';
    const ACTION_LOGIN = 'login';
    const ACTION_ADDRESSES = 'addresses';
    const ACTION_CHECKOUT = 'checkout';
    const ACTION_TICKET = 'ticket';

    /**
     * @var array
     */
    protected $stepAnonymousOptions = [
        self::ACTION_CART     => ['active' => 'cart', 'link' => []],
        self::ACTION_LOGIN    => ['active' => 'login', 'link' => ['cart']],
        self::ACTION_CHECKOUT => ['active' => 'checkout', 'link' => []],
        self::ACTION_TICKET   => ['active' => 'ticket', 'link' => []],
    ];

    /**
     * @var array
     */
    protected $stepLoggedOptions = [
        self::ACTION_CART      => ['active' => 'cart', 'link' => []],
        self::ACTION_ADDRESSES => ['active' => 'addresses', 'link' => ['cart']],
        self::ACTION_CHECKOUT  => ['active' => 'checkout', 'link' => ['cart', 'addresses']],
        self::ACTION_TICKET    => ['active' => 'ticket', 'link' => []],
    ];

    /**
     * @var array
     */
    protected $stepGuestOptions = [
        self::ACTION_CART     => ['active' => 'cart', 'link' => []],
        self::ACTION_LOGIN    => ['active' => 'login', 'link' => ['cart']],
        self::ACTION_CHECKOUT => ['active' => 'checkout', 'link' => ['cart', 'login']],
        self::ACTION_TICKET   => ['active' => 'ticket', 'link' => []],
    ];

    /**
     * @var LocalizationInterface
     */
    protected $language;

    /**
     * @var Request
     */
    protected $request;

    /**
     * @param LocalizationInterface $language
     * @param Request               $request
     */
    function __construct(LocalizationInterface $language, Request $request)
    {
        $this->language = $language;
        $this->request = $request;
    }

    /**
     * build
     *
     * @param Order $order
     * @return Menu
     * @throws \LogicException
     */
    public function build(Order $order)
    {
        $requestUri = $this->request->getRequestUri();
        if (($pos = strpos($requestUri, '?')) !== false) {
            $requestUri = substr($requestUri, 0, $pos);
        }

        $segments = explode('/', trim($requestUri, '/'));
        $currentAction = array_pop($segments);

        if (is_numeric($currentAction)) {
            $currentAction = array_pop($segments);
        }

        if ($order->customerIsLogged()) {
            $options = $this->stepLoggedOptions;
        } else {
            if ($order->customerIsGuest()) {
                $options = $this->stepGuestOptions;
            } else {
                $options = $this->stepAnonymousOptions;
            }
        }

        if (!array_key_exists($currentAction, $options)) {
            throw new \LogicException(sprintf('Action "%s" is not valid.', $currentAction));
        }

        $basePath = '/' . implode('/', $segments) . '/';

        $stepOptions = $options[$currentAction];

        $menu = new Menu('checkout-breadcrumb');
        $menu->setOptions(['item_template' => [$this, 'itemTemplate']]);

        $step = 1;
        $old = true;
        foreach (array_keys($options) as $stepAction) {
            $title = $this->language->get('checkout_breadcrumb_' . $stepAction);
            $label = sprintf('<span class="number">%d</span><span class="step">%s</span>', $step, $title);
            $url = null;

            if (in_array($stepAction, $stepOptions['link'])) {
                $url = $basePath . $stepAction . '/';
            }

            $item = new Item($label, $url);

            if ($stepAction === $currentAction) {
                $item->addClass('active');
            }

            if ($url) {
                $item->setAttribute('title', $title);
            }

            if ($stepAction === $currentAction) {
                $old = false;
            }

            if ($old) {
                $item->addClass('checked');
            }

            $menu->add($item);

            $step++;
        }

        return $menu;
    }

    /**
     * itemTemplate
     *
     * @ignore
     */
    public function itemTemplate(Item $item)
    {
        $url = $item->getUrl();

        if ($url === null) {
            return $item->getLabel();
        } else {
            return '<a' . $item->getAttributesString() . '>' . $item->getLabel() . '</a>';
        }
    }
}
