<?php

/**
 * Questo file è parte di pongho-better-checkout.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Tests\Utilities;

use Application\Showcase\Utilities\CheckoutBreadcrumbBuilder;
use Pongho\Core\Tests\Mock\Localization;

class CheckoutBreadcrumbBuilderTest extends \PHPUnit_Framework_TestCase
{
    /**
     * @dataProvider casesProvider
     */
    public function testAllCases($url, $is_logged, $is_guest, $expected)
    {
        $builder = $this->getBuilder($url);
        $order = $this->getOrderMock($is_logged, $is_guest);
        $expected = preg_replace("#[\r|\n]#", '', $expected);

        $this->assertEquals($expected, (string) $builder->build($order));
    }

    public function casesProvider()
    {
        return [
            [
                '/shop/cart/',
                false,
                false,
                <<<HTML
<li class="active"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></li>
<li><span class="number">2</span><span class="step">checkout_breadcrumb_login</span></li>
<li><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
            [
                '/shop/cart/',
                false,
                true,
                <<<HTML
<li class="active"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></li>
<li><span class="number">2</span><span class="step">checkout_breadcrumb_login</span></li>
<li><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
            [
                '/foo/shop/cart/',
                false,
                false,
                <<<HTML
<li class="active"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></li>
<li><span class="number">2</span><span class="step">checkout_breadcrumb_login</span></li>
<li><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
            [
                '/shop/cart/',
                true,
                false,
                <<<HTML
<li class="active"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></li>
<li><span class="number">2</span><span class="step">checkout_breadcrumb_addresses</span></li>
<li><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
            [
                '/shop/cart/',
                true,
                true,
                <<<HTML
<li class="active"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></li>
<li><span class="number">2</span><span class="step">checkout_breadcrumb_addresses</span></li>
<li><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
            [
                '/foo/shop/cart/',
                true,
                false,
                <<<HTML
<li class="active"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></li>
<li><span class="number">2</span><span class="step">checkout_breadcrumb_addresses</span></li>
<li><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
            [
                '/shop/login/',
                false,
                false,
                <<<HTML
<li class="checked"><a title="checkout_breadcrumb_cart" href="/shop/cart/"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></a></li>
<li class="active"><span class="number">2</span><span class="step">checkout_breadcrumb_login</span></li>
<li><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
            [
                '/shop/login/',
                false,
                true,
                <<<HTML
<li class="checked"><a title="checkout_breadcrumb_cart" href="/shop/cart/"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></a></li>
<li class="active"><span class="number">2</span><span class="step">checkout_breadcrumb_login</span></li>
<li><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
            [
                '/foo/shop/login/',
                false,
                false,
                <<<HTML
<li class="checked"><a title="checkout_breadcrumb_cart" href="/foo/shop/cart/"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></a></li>
<li class="active"><span class="number">2</span><span class="step">checkout_breadcrumb_login</span></li>
<li><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
            [
                '/shop/addresses/',
                true,
                false,
                <<<HTML
<li class="checked"><a title="checkout_breadcrumb_cart" href="/shop/cart/"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></a></li>
<li class="active"><span class="number">2</span><span class="step">checkout_breadcrumb_addresses</span></li>
<li><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
            [
                '/foo/shop/addresses/',
                true,
                false,
                <<<HTML
<li class="checked"><a title="checkout_breadcrumb_cart" href="/foo/shop/cart/"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></a></li>
<li class="active"><span class="number">2</span><span class="step">checkout_breadcrumb_addresses</span></li>
<li><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
            [
                '/shop/checkout/',
                true,
                false,
                <<<HTML
<li class="checked"><a title="checkout_breadcrumb_cart" href="/shop/cart/"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></a></li>
<li class="checked"><a title="checkout_breadcrumb_addresses" href="/shop/addresses/"><span class="number">2</span><span class="step">checkout_breadcrumb_addresses</span></a></li>
<li class="active"><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
            [
                '/shop/checkout/',
                false,
                true,
                <<<HTML
<li class="checked"><a title="checkout_breadcrumb_cart" href="/shop/cart/"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></a></li>
<li class="checked"><a title="checkout_breadcrumb_login" href="/shop/login/"><span class="number">2</span><span class="step">checkout_breadcrumb_login</span></a></li>
<li class="active"><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
            [
                '/foo/shop/checkout/',
                true,
                false,
                <<<HTML
<li class="checked"><a title="checkout_breadcrumb_cart" href="/foo/shop/cart/"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></a></li>
<li class="checked"><a title="checkout_breadcrumb_addresses" href="/foo/shop/addresses/"><span class="number">2</span><span class="step">checkout_breadcrumb_addresses</span></a></li>
<li class="active"><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
            [
                '/shop/ticket/123/?hash=456foo789bar123baz',
                true,
                false,
                <<<HTML
<li class="checked"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></li>
<li class="checked"><span class="number">2</span><span class="step">checkout_breadcrumb_addresses</span></li>
<li class="checked"><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li class="active"><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
            [
                '/shop/ticket/123/?hash=456foo789bar123baz',
                false,
                true,
                <<<HTML
<li class="checked"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></li>
<li class="checked"><span class="number">2</span><span class="step">checkout_breadcrumb_login</span></li>
<li class="checked"><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li class="active"><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
            [
                '/foo/shop/ticket/123/?hash=456foo789bar123baz',
                true,
                false,
                <<<HTML
<li class="checked"><span class="number">1</span><span class="step">checkout_breadcrumb_cart</span></li>
<li class="checked"><span class="number">2</span><span class="step">checkout_breadcrumb_addresses</span></li>
<li class="checked"><span class="number">3</span><span class="step">checkout_breadcrumb_checkout</span></li>
<li class="active"><span class="number">4</span><span class="step">checkout_breadcrumb_ticket</span></li>
HTML
            ],
        ];
    }

    /**
     * @expectedException \LogicException
     */
    public function testNotValidPath()
    {
        $builder = $this->getBuilder('/foo/bar/');
        $builder->build($this->getOrderMock());
    }

    protected function getBuilder($request_uri)
    {
        // @todo Sostituire questo mock con Request::create() quando passeremo al componente Http di Symfony.
        $request = $this->getMockBuilder('Pongho\Http\Request')
            ->disableOriginalConstructor()
            ->setMethods(['getRequestUri'])
            ->getMock();

        $request
            ->expects($this->any())
            ->method('getRequestUri')
            ->will($this->returnValue($request_uri));

        /**
         * @var \Pongho\Http\Request $request
         */

        return new CheckoutBreadcrumbBuilder(new Localization(), $request);
    }

    /**
     * getOrderMock
     *
     * @param bool $is_logged
     * @param bool $is_guest
     * @return \Application\Showcase\Model\Order
     */
    protected function getOrderMock($is_logged = false, $is_guest = false)
    {
        $order = $this->getMockBuilder('Application\\Showcase\\Model\\Order')
            ->disableOriginalConstructor()
            ->setMethods(['customerIsLogged', 'customerIsGuest', '__wakeup'])
            ->getMock();

        $order
            ->expects($this->any())
            ->method('customerIsLogged')
            ->will($this->returnValue($is_logged));

        $order
            ->expects($this->any())
            ->method('customerIsGuest')
            ->will($this->returnValue($is_guest));

        return $order;
    }
}
