<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Tests\ShoppingPoints;

use Application\Showcase\Model\Order;
use Application\Showcase\ShoppingPoints\Scheme;

class SchemeTest extends \PHPUnit_Framework_TestCase
{
    public function testInitialize()
    {
        $scheme = new Scheme(20, 1, 10, 50);

        $this->assertEquals(20, $scheme->getPointCost());
        $this->assertEquals(1, $scheme->getPointValue());
        $this->assertEquals(10, $scheme->getMinPoints());
        $this->assertEquals(50, $scheme->getMinOrderTotal());
    }

    public function testStringParameters()
    {
        $scheme = new Scheme('30', '2', '5', '40');

        $this->assertEquals(30, $scheme->getPointCost());
        $this->assertEquals(2, $scheme->getPointValue());
        $this->assertEquals(5, $scheme->getMinPoints());
        $this->assertEquals(40, $scheme->getMinOrderTotal());
    }

    /**
     * @dataProvider calculateDataProvider
     *
     * @param float $pointCost
     * @param float $amount
     * @param int   $points
     */
    public function testCalculate($pointCost, $amount, $points)
    {
        $scheme = new Scheme($pointCost, 1, 10, 10);

        $this->assertEquals($points, $scheme->calculate($amount));
    }

    /**
     * @dataProvider calculateDataProvider
     *
     * @param float $pointCost
     * @param float $amount
     * @param int   $points
     */
    public function testCalculateFromOrder($pointCost, $amount, $points)
    {
        $scheme = new Scheme($pointCost, 1, 10, 10);
        $order = $this->getOrderMock(false, 0, $amount);

        $this->assertEquals($points, $scheme->calculateFromOrder($order));
    }

    public function calculateDataProvider()
    {
        return [
            [20, 50, 2],
            [10, 50, 5],
            [20, 39, 1],
            [19.99, 39.98, 2],
        ];
    }

    public function testIsValidOrder()
    {
        $minPoints = 10;
        $minOrderTotal = 50;

        $scheme = new Scheme(20, 1, $minPoints, $minOrderTotal);

        $this->assertTrue(
            $scheme->isValidOrder(
                $this->getOrderMock(false, $minPoints, $minOrderTotal)
            )
        );
    }

    public function testGuestOrderIsNotApplicable()
    {
        $scheme = new Scheme(20, 1, 10, 50);

        $this->assertFalse(
            $scheme->isValidOrder(
                $this->getOrderMock(true)
            )
        );
    }

    public function testMinPointsIsNotApplicable()
    {
        $scheme = new Scheme(20, 1, 10, 50);

        $this->assertFalse(
            $scheme->isValidOrder(
                $this->getOrderMock(false, 9, 50)
            )
        );
    }

    public function testMinTotalOrderIsNotApplicable()
    {
        $scheme = new Scheme(20, 1, 10, 50);

        $this->assertFalse(
            $scheme->isValidOrder(
                $this->getOrderMock(false, 10, 49)
            )
        );
    }

    /**
     * @dataProvider calculateMaxUsableOrderPointsProvider
     *
     * @param int   $expected
     * @param int   $customerPoints
     * @param float $orderTotal
     * @param float $pointCost
     */
    public function testCalculateMaxUsableOrderPoints($expected, $customerPoints, $orderTotal, $pointCost)
    {
        $scheme = new Scheme(20, $pointCost, 10, 10);
        $order = $this->getOrderMock(false, $customerPoints, $orderTotal);

        $this->assertEquals($expected, $scheme->calculateMaxUsableOrderPoints($order));
    }

    public function calculateMaxUsableOrderPointsProvider()
    {
        return [
            [49, 49, 50, 1],
            [10, 10, 50, 2],
            [50, 100, 50, 1],
            [25, 100, 50, 2],
            [16, 100, 50, 3],
            [10, 100, 50, 5],
            [5, 100, 50, 10],
            [2, 100, 50, 20],
            [1, 100, 50, 50],
            [0, 100, 49.99, 50],
        ];
    }

    /**
     * @dataProvider arePointsApplicableToOrderProvider
     *
     * @param bool $expected
     * @param int  $usedPoints   Il numero di punti che il cliente vuole utilizzare
     * @param bool $isValidOrder Se l’ordine soddisfa i requisiti del programma
     * @param int  $maxPoints    Il numero massimo di punti utilizzabili per l’ordine
     * @param int  $minPoints    Il numero minimo di punti per poterli usare
     */
    public function testArePointsApplicableToOrder($expected, $usedPoints, $isValidOrder, $maxPoints, $minPoints)
    {
        $scheme = $this->getMockBuilder(Scheme::class)
            ->setMethods(['isValidOrder', 'calculateMaxUsableOrderPoints'])
            ->setConstructorArgs([10, 1, $minPoints, 10])
            ->getMock();

        $scheme->method('isValidOrder')->willReturn($isValidOrder);
        $scheme->method('calculateMaxUsableOrderPoints')->willReturn($maxPoints);

        $order = $this->getMockBuilder(Order::class)
            ->disableOriginalConstructor()
            ->getMock();

        /**
         * @var Scheme $scheme
         * @var Order  $order
         */

        $this->assertEquals($expected, $scheme->arePointsApplicableToOrder($order, $usedPoints));
    }

    public function arePointsApplicableToOrderProvider()
    {
        return [
            // Caso standard
            'standard'               => [
                'expected'       => true,
                'used_points'    => 15,
                'is_valid_order' => true,
                'max_points'     => 50,
                'min_points'     => 10,
            ],

            // Il cliente utilizza tutti i punti permessi dall’ordine e non paga nulla
            'gratis'                 => [
                'expected'       => true,
                'used_points'    => 50,
                'is_valid_order' => true,
                'max_points'     => 50,
                'min_points'     => 10,
            ],

            // Ordine non valido
            'order_not_valid'        => [
                'expected'       => false,
                'used_points'    => 15,
                'is_valid_order' => false,
                'max_points'     => 50,
                'min_points'     => 10,
            ],

            // Il cliente vuole usare più punti di quelli che potrebbe
            'exceed_customer_points' => [
                'expected'       => false,
                'used_points'    => 21,
                'is_valid_order' => true,
                'max_points'     => 20,
                'min_points'     => 10,
            ],

            // Il cliente vuole usare meno punti del minimo utilizzabile
            'not_enough'             => [
                'expected'       => false,
                'used_points'    => 9,
                'is_valid_order' => true,
                'max_points'     => 50,
                'min_points'     => 10,
            ],
        ];
    }

    /**
     * @param bool  $isGuest
     * @param int   $customerPoints
     * @param float $orderTotal
     * @return Order
     */
    protected function getOrderMock($isGuest = false, $customerPoints = 20, $orderTotal = 50.0)
    {
        $order = $this->getMockBuilder(Order::class)
            ->disableOriginalConstructor()
            ->getMock();

        $order->method('customerIsGuest')->willReturn($isGuest);
        $order->method('getAvailableCustomerShoppingPoints')->willReturn($customerPoints);
        $order->method('subtotal')->willReturn($orderTotal);

        return $order;
    }
}
