<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Stats;

use Application\Admin\Widgets\Stats\StatWidget;
use Application\Admin\Widgets\Stats\View\GoogleChart\ChartFactory;
use Application\Core\I18n\Translation\Translator;
use Application\Showcase\Stats\Report\SoldReport;
use Application\Showcase\Stats\Report\TopSoldProductsReport;
use Pongho\Utilities\DateTime;

/**
 * StatsFactory
 */
class StatsFactory
{
    /**
     * @var Translator
     */
    protected $translator;

    /**
     * @var ChartFactory
     */
    protected $chartFactory;

    /**
     * @param Translator   $translator
     * @param ChartFactory $chartFactory
     */
    public function __construct(Translator $translator, ChartFactory $chartFactory)
    {
        $this->translator = $translator;
        $this->chartFactory = $chartFactory;
    }

    /**
     * @return StatWidget
     */
    public function createLastMonthSold()
    {
        $report = new SoldReport(
            new DateTime('-30 days'),
            new DateTime('today'),
            SoldReport::INTERVAL_DAY
        );

        return new StatWidget(
            'sold_last_month',
            $this->translator->trans('Last month'),
            2,
            $this->chartFactory->createComboChart(
                $this->getSoldHeadersLabel(),
                $this->getSoldOptions($report)
            ),
            $report
        );
    }

    /**
     * @return StatWidget
     */
    public function createLastQuarterSold()
    {
        $report = new SoldReport(
            new DateTime('-3 months'),
            new DateTime('today'),
            SoldReport::INTERVAL_WEEK
        );

        return new StatWidget(
            'sold_last_quarter',
            $this->translator->trans('Last quarter'),
            2,
            $this->chartFactory->createComboChart(
                $this->getSoldHeadersLabel(),
                $this->getSoldOptions($report)
            ),
            $report
        );
    }

    /**
     * @return StatWidget
     */
    public function createLastHalfYearSold()
    {
        $report = new SoldReport(
            new DateTime('-6 months'),
            new DateTime('today'),
            SoldReport::INTERVAL_WEEK
        );

        return new StatWidget(
            'sold_last_half_year',
            $this->translator->trans('Last half year'),
            2,
            $this->chartFactory->createComboChart(
                $this->getSoldHeadersLabel(),
                $this->getSoldOptions($report)
            ),
            $report
        );
    }

    /**
     * @return StatWidget
     */
    public function createLastYearSold()
    {
        $report = new SoldReport(
            new DateTime('-1 year'),
            new DateTime('tomorrow midnight'),
            SoldReport::INTERVAL_MONTH
        );

        return new StatWidget(
            'sold_last_year',
            $this->translator->trans('Last year'),
            2,
            $this->chartFactory->createComboChart(
                $this->getSoldHeadersLabel(),
                $this->getSoldOptions($report)
            ),
            $report
        );
    }

    /**
     * @param int $language_id
     * @return StatWidget
     */
    public function createLastMonthTopSoldProducts($language_id)
    {
        $report = new TopSoldProductsReport(
            $language_id,
            new DateTime('-30 days'),
            new DateTime('today')
        );

        return new StatWidget(
            'top_sold_products_last_month',
            $this->translator->trans('Top sold products last month'),
            1,
            $this->chartFactory->createBarChart(
                $this->getTopSoldProductsHeadersLabel(),
                $this->getTopSoldProductsOptions()
            ),
            $report
        );
    }

    /**
     * @param int $language_id
     * @return StatWidget
     */
    public function createLastYearTopSoldProducts($language_id)
    {
        $report = new TopSoldProductsReport(
            $language_id,
            new DateTime('-180 days'),
            new DateTime('today')
        );

        return new StatWidget(
            'top_sold_products_last_year',
            $this->translator->trans('Top sold products last year'),
            1,
            $this->chartFactory->createBarChart(
                $this->getTopSoldProductsHeadersLabel(),
                $this->getTopSoldProductsOptions()
            ),
            $report
        );
    }

    /**
     * @return StatWidget
     */
    public function createLastMonthOrdered()
    {
        $report = new SoldReport(
            new DateTime('-30 days'),
            new DateTime('today'),
            SoldReport::INTERVAL_DAY,
            SoldReport::TYPE_ORDERED
        );

        return new StatWidget(
            'ordered_last_month',
            $this->translator->trans('Last month'),
            2,
            $this->chartFactory->createComboChart(
                $this->getOrderedHeadersLabel(),
                $this->getOrderedOptions($report)
            ),
            $report
        );
    }

    /**
     * @return StatWidget
     */
    public function createLastQuarterOrdered()
    {
        $report = new SoldReport(
            new DateTime('-3 months'),
            new DateTime('today'),
            SoldReport::INTERVAL_WEEK,
            SoldReport::TYPE_ORDERED
        );

        return new StatWidget(
            'ordered_last_quarter',
            $this->translator->trans('Last quarter'),
            2,
            $this->chartFactory->createComboChart(
                $this->getOrderedHeadersLabel(),
                $this->getOrderedOptions($report)
            ),
            $report
        );
    }

    /**
     * @return StatWidget
     */
    public function createLastHalfYearOrdered()
    {
        $report = new SoldReport(
            new DateTime('-6 months'),
            new DateTime('today'),
            SoldReport::INTERVAL_WEEK,
            SoldReport::TYPE_ORDERED
        );

        return new StatWidget(
            'ordered_last_half_year',
            $this->translator->trans('Last half year'),
            2,
            $this->chartFactory->createComboChart(
                $this->getOrderedHeadersLabel(),
                $this->getOrderedOptions($report)
            ),
            $report
        );
    }

    /**
     * @return StatWidget
     */
    public function createLastYearOrdered()
    {
        $report = new SoldReport(
            new DateTime('-1 year'),
            new DateTime('today'),
            SoldReport::INTERVAL_MONTH,
            SoldReport::TYPE_ORDERED
        );

        return new StatWidget(
            'ordered_last_year',
            $this->translator->trans('Last year'),
            2,
            $this->chartFactory->createComboChart(
                $this->getOrderedHeadersLabel(),
                $this->getOrderedOptions($report)
            ),
            $report
        );
    }

    /**
     * @param int $language_id
     * @return StatWidget
     */
    public function createLastMonthTopOrderedProducts($language_id)
    {
        $report = new TopSoldProductsReport(
            $language_id,
            new DateTime('-30 days'),
            new DateTime('today'),
            TopSoldProductsReport::TYPE_ORDERED
        );

        return new StatWidget(
            'top_ordered_products_last_month',
            $this->translator->trans('Top ordered products last month'),
            1,
            $this->chartFactory->createBarChart(
                $this->getTopOrderedProductsHeadersLabel(),
                $this->getTopOrderedProductsOptions()
            ),
            $report
        );
    }

    /**
     * @param int $language_id
     * @return StatWidget
     */
    public function createLastYearTopOrderedProducts($language_id)
    {
        $report = new TopSoldProductsReport(
            $language_id,
            new DateTime('-180 days'),
            new DateTime('today'),
            TopSoldProductsReport::TYPE_ORDERED
        );

        return new StatWidget(
            'top_ordered_products_last_year',
            $this->translator->trans('Top ordered products last year'),
            1,
            $this->chartFactory->createBarChart(
                $this->getTopOrderedProductsHeadersLabel(),
                $this->getTopOrderedProductsOptions()
            ),
            $report
        );
    }

    /**
     * @return array
     */
    protected function getSoldHeadersLabel()
    {
        return [
            '',
            $this->translator->trans('Amount'),
            $this->translator->trans('Products'),
        ];
    }

    /**
     * @param SoldReport $report
     * @return array
     */
    protected function getSoldOptions(SoldReport $report)
    {
        return [
            'hAxis'      => $this->getHAxisOptions($report),
            'seriesType' => 'line',
            'series'     => [
                1 => [
                    'type'            => 'steppedArea',
                    'targetAxisIndex' => 1,
                ]
            ],
            'vAxes'      => [
                [
                    'title'     => $this->translator->trans('Amount sold'),
                    'textStyle' => [
                        'color' => '#0000CC',
                    ]
                ],
                [
                    'title'     => $this->translator->trans('Number of products sold'),
                    'textStyle' => [
                        'color' => '#CC0000',
                    ]
                ],
            ],
            'vAxis'      => [
                'gridlines' => [
                    'count' => 4,
                ],
            ],
        ];
    }

    /**
     * @return array
     */
    protected function getTopSoldProductsHeadersLabel()
    {
        return [
            '',
            $this->translator->trans('Products'),
        ];
    }

    /**
     * @return array
     */
    protected function getTopSoldProductsOptions()
    {
        return [
            'hAxis'     => [
                'title' => $this->translator->trans('Number of products sold'),
            ],
            'legend'    => 'none',
            'chartArea' => [
                'height' => '75%',
                'left'   => '45%',
                'top'    => 0,
                'width'  => '50%',
            ],
            'height'    => 250,
        ];
    }

    /**
     * @return array
     */
    protected function getOrderedHeadersLabel()
    {
        return [
            '',
            $this->translator->trans('Amount'),
            $this->translator->trans('Products'),
        ];
    }

    /**
     * @param SoldReport $report
     * @return array
     */
    protected function getOrderedOptions(SoldReport $report)
    {
        return [
            'hAxis'      => $this->getHAxisOptions($report),
            'seriesType' => 'line',
            'series'     => [
                1 => [
                    'type'            => 'steppedArea',
                    'targetAxisIndex' => 1,
                ]
            ],
            'vAxes'      => [
                [
                    'title'     => $this->translator->trans('Amount ordered'),
                    'textStyle' => [
                        'color' => '#0000CC',
                    ]
                ],
                [
                    'title'     => $this->translator->trans('Number of products ordered'),
                    'textStyle' => [
                        'color' => '#CC0000',
                    ]
                ],
            ],
            'vAxis'      => [
                'gridlines' => [
                    'count' => 4,
                ],
            ],
        ];
    }

    /**
     * @return array
     */
    protected function getTopOrderedProductsHeadersLabel()
    {
        return [
            '',
            $this->translator->trans('Products'),
        ];
    }

    /**
     * @return array
     */
    protected function getTopOrderedProductsOptions()
    {
        return [
            'hAxis'     => [
                'title' => $this->translator->trans('Number of products ordered'),
            ],
            'legend'    => 'none',
            'chartArea' => [
                'height' => '75%',
                'left'   => '45%',
                'top'    => 0,
                'width'  => '50%',
            ],
            'height'    => 250,
        ];
    }

    /**
     * @param SoldReport $report
     * @return array
     */
    protected function getHAxisOptions(SoldReport $report)
    {
        switch ($report->getInterval()) {
            case SoldReport::INTERVAL_DAY:
                return [
                    'title'         => $this->translator->trans('Days'),
                    'showTextEvery' => 4,
                ];

            case SoldReport::INTERVAL_WEEK:
                return [
                    'title'         => $this->translator->trans('Weeks'),
                    'showTextEvery' => 3,
                ];

            case SoldReport::INTERVAL_MONTH:
                return [
                    'title'         => $this->translator->trans('Months'),
                    'showTextEvery' => 2,
                ];

            case SoldReport::INTERVAL_YEAR:
                return [
                    'title' => $this->translator->trans('Years'),
                ];
        }

        return [];
    }
}
