<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Stats\Report;

use Application\Admin\Widgets\Stats\ReportInterface;
use Application\Showcase\Model\Order;
use Pongho\Utilities\DateTime;

/**
 * SoldReport
 */
class SoldReport implements ReportInterface
{
    const INTERVAL_DAY = 'day';
    const INTERVAL_WEEK = 'week';
    const INTERVAL_MONTH = 'month';
    const INTERVAL_YEAR = 'year';

    const TYPE_SOLD = 'sold';
    const TYPE_ORDERED = 'ordered';

    /**
     * @var DateTime
     */
    protected $start_date;

    /**
     * @var DateTime
     */
    protected $end_date;

    /**
     * @var string
     */
    protected $interval = self::INTERVAL_DAY;

    /**
     * @var string
     */
    protected $type = self::TYPE_SOLD;

    /**
     * @param DateTime $start_date
     * @param DateTime $end_date
     * @param string   $interval Una delle costanti INTERVAL_
     * @param string   $type
     */
    public function __construct(DateTime $start_date, DateTime $end_date, $interval = self::INTERVAL_DAY, $type = self::TYPE_SOLD)
    {
        if (!in_array($type, [self::TYPE_SOLD, self::TYPE_ORDERED])) {
            throw new \InvalidArgumentException(
                sprintf(
                    'Type "%s" is not supported. You must use only these: %s',
                    $type,
                    implode(', ', [self::TYPE_SOLD, self::TYPE_ORDERED])
                )
            );
        }

        $this->start_date = $start_date;
        $this->end_date = $end_date;
        $this->interval = $interval;
        $this->type = $type;
    }

    /**
     * @return \DateTime
     */
    public function getStartDate()
    {
        return $this->start_date;
    }

    /**
     * @return \DateTime
     */
    public function getEndDate()
    {
        return $this->end_date;
    }

    /**
     * @return string
     */
    public function getInterval()
    {
        return $this->interval;
    }

    /**
     * {@inheritdoc}
     */
    public function getData()
    {
        $stats = [];
        foreach (Order::all($this->getQueryOptions()) as $stat) {
            $stats[$stat->time_interval] = [
                'time_interval' => null,
                'total'         => (float) $stat->total,
                'products'      => (int) $stat->products,
            ];
        }

        $period = new \DatePeriod($this->start_date, $this->getDateInterval(), $this->end_date);
        $key_format = $this->getDateFormatForKey();
        $time_interval_format = $this->getDateFormatForTimeInterval();

        $data = [];

        /** @var DateTime $date */
        foreach ($period as $date) {
            $key = $date->format($key_format);
            $time_interval = $date->format($time_interval_format);

            if (isset($stats[$key])) {
                $data[] = [
                    'x' => [$time_interval],
                    'y' => [$stats[$key]['total'], $stats[$key]['products']],
                ];
            } else {
                $data[] = [
                    'x' => [$time_interval],
                    'y' => [0, 0]
                ];
            }
        }

        return $data;
    }

    /**
     * @return array
     */
    protected function getQueryOptions()
    {
        $group = $this->getIntervalGroupStatement();

        if ($this->type === self::TYPE_ORDERED) {
            $date_field = 'ordered_at';
            $status = Order::$ORDER_STATUSES;
        } else {
            $date_field = 'paid_at';
            $status = [Order::STATUS_PROCESSED];
        }

        return [
            'select'     => $group . ' AS time_interval, SUM(products_total) AS total, SUM(products_count) AS products',
            'conditions' => [
                "({$date_field} BETWEEN :start AND :end) AND status IN ('" . implode("', '", $status) . "')",
                'start'  => $this->start_date,
                'end'    => $this->end_date,
            ],
            'group'      => $group,
            'order'      => $group,
        ];
    }

    /**
     * @return string
     */
    protected function getIntervalGroupStatement()
    {
        $field = $this->type === self::TYPE_ORDERED ? 'ordered_at' : 'paid_at';

        switch ($this->interval) {
            case self::INTERVAL_DAY:
                return "DATE_FORMAT({$field}, '%Y-%m-%d')";

            case self::INTERVAL_WEEK:
                return "DATE_FORMAT({$field}, '%x-%v')";

            case self::INTERVAL_MONTH:
                return "DATE_FORMAT({$field}, '%Y-%m')";

            case self::INTERVAL_YEAR:
                return "YEAR({$field})";
        }

        throw new \LogicException(sprintf('Interval "%s" is not supported.', $this->interval));
    }

    /**
     * @return \DateInterval
     */
    protected function getDateInterval()
    {
        switch ($this->interval) {
            case self::INTERVAL_DAY:
                return new \DateInterval('P1D');

            case self::INTERVAL_WEEK:
                return new \DateInterval('P7D');

            case self::INTERVAL_MONTH:
                return new \DateInterval('P1M');

            case self::INTERVAL_YEAR:
                return new \DateInterval('P1Y');
        }

        throw new \LogicException(sprintf('Interval "%s" is not supported.', $this->interval));
    }

    /**
     * @return string
     */
    protected function getDateFormatForKey()
    {
        switch ($this->interval) {
            case self::INTERVAL_DAY:
                return 'Y-m-d';

            case self::INTERVAL_WEEK:
                return 'Y-W';

            case self::INTERVAL_MONTH:
                return 'Y-m';

            case self::INTERVAL_YEAR:
                return 'Y';
        }

        throw new \LogicException(sprintf('Interval "%s" is not supported.', $this->interval));
    }

    /**
     * @return string
     */
    protected function getDateFormatForTimeInterval()
    {
        switch ($this->interval) {
            case self::INTERVAL_DAY:
                return 'Y-m-d';

            case self::INTERVAL_WEEK:
                return 'Y-m-d';

            case self::INTERVAL_MONTH:
                return 'Y-m';

            case self::INTERVAL_YEAR:
                return 'Y';
        }

        throw new \LogicException(sprintf('Interval "%s" is not supported.', $this->interval));
    }
}
