<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\ShoppingPoints;

use Application\Showcase\Model\Order;

/**
 * Scheme
 */
class Scheme
{
    /**
     * @var float
     */
    protected $point_cost;

    /**
     * @var float
     */
    protected $point_value;

    /**
     * @var int
     */
    protected $min_points;

    /**
     * @var float
     */
    protected $min_order_total;

    /**
     * @var bool
     */
    protected $is_enabled_points_per_product;

    /**
     * @var bool
     */
    protected $calculate_on_discounts;

    /**
     * @param float $point_cost
     * @param float $point_value
     * @param int   $min_points
     * @param float $min_order_total
     * @param bool  $is_enabled_points_per_product
     * @param bool  $calculate_on_discounts
     */
    public function __construct(
        $point_cost,
        $point_value,
        $min_points,
        $min_order_total,
        $is_enabled_points_per_product = false,
        $calculate_on_discounts = true
    ) {
        $this->point_cost = (float) $point_cost;
        $this->point_value = (float) $point_value;
        $this->min_points = (int) $min_points;
        $this->min_order_total = (float) $min_order_total;
        $this->is_enabled_points_per_product = (bool) $is_enabled_points_per_product;
        $this->calculate_on_discounts = (bool) $calculate_on_discounts;
    }

    /**
     * @return float
     */
    public function getPointCost()
    {
        return $this->point_cost;
    }

    /**
     * @return string
     *
     * @todo Test
     */
    public function getFormatPointCost()
    {
        return format_price($this->point_cost);
    }

    /**
     * @return float
     */
    public function getPointValue()
    {
        return $this->point_value;
    }

    /**
     * @return string
     *
     * @todo Test
     */
    public function getFormatPointValue()
    {
        return format_price($this->point_value);
    }

    /**
     * @return int
     */
    public function getMinPoints()
    {
        return $this->min_points;
    }

    /**
     * @return float
     */
    public function getMinOrderTotal()
    {
        return $this->min_order_total;
    }

    /**
     * @return string
     *
     * @todo Test
     */
    public function getFormatMinOrderTotal()
    {
        return format_price($this->min_order_total);
    }

    /**
     * @param float $amount
     * @return int
     *
     * @internal
     */
    public function calculate($amount)
    {
        if ($this->getPointCost()) {
            return floor($amount / $this->getPointCost());
        }

        return 0;
    }

    /**
     * @param Order $order
     * @return int
     */
    public function calculateFromOrder(Order $order)
    {
        $points = $this->calculate($this->getOrderTotal($order));

        if ($this->is_enabled_points_per_product) {
            if ($order->isCart()) {
                foreach ($order->getRows() as $row) {
                    if ($row->size()) {
                        $points += $row->quantity() * $row->size()->getShoppingPoints();
                    } else {
                        $points += $row->quantity() * $row->getProduct()->getShoppingPoints();
                    }
                }
            } else {
                foreach ($order->getRows() as $row) {
                    $points += $row->getShoppingPoints();
                }
            }
        }

        return $points;
    }

    /**
     * @param Order $order
     * @return bool
     */
    public function isValidOrder(Order $order)
    {
        if ($order->customerIsGuest()) {
            return false;
        }

        if ($this->getOrderTotal($order) < $this->getMinOrderTotal()) {
            return false;
        }

        $customer_points = $order->getAvailableCustomerShoppingPoints();

        if ($customer_points === 0 || $customer_points < $this->getMinPoints()) {
            return false;
        }

        return true;
    }

    /**
     * @param Order $order
     * @return int
     */
    public function calculateMaxUsableOrderPoints(Order $order)
    {
        if (!$this->point_value) {
            return 0;
        }

        $max_order = (int) floor($this->getOrderTotal($order) / $this->point_value);
        $max_customer = $order->getAvailableCustomerShoppingPoints();

        return $max_order < $max_customer ? $max_order : $max_customer;
    }

    /**
     * @param Order $order
     * @param int   $points
     * @return bool
     */
    public function arePointsApplicableToOrder(Order $order, $points)
    {
        if (!$this->isValidOrder($order)) {
            return false;
        }

        if ($points < $this->min_points) {
            return false;
        }

        if ($points > $this->calculateMaxUsableOrderPoints($order)) {
            return false;
        }

        return true;
    }

    /**
     * @param Order $order
     * @return float
     */
    private function getOrderTotal(Order $order)
    {
        if ($this->calculate_on_discounts) {
            return $order->subtotal();
        } else {
            return $order->total();
        }
    }
}
