/**
 * specProduct.js
 *
 * @author Daniele De Nobili
 */

/*jslint nomen: true, plusplus: true, passfail: true, browser: true, devel: true */
/*global $, jasmine, describe, it, xdescribe, xit, beforeEach, afterEach, expect, loadFixtures, spyOnEvent */
/*global Pongho */

jasmine.getFixtures().fixturesPath = 'spec/fixtures';
jasmine.getStyleFixtures().fixturesPath = 'spec/fixtures';

// Potrei usare toHaveBeenPrevented, ma crea un loop con le form.
// @todo Integrare queste funzioni in jasmine o inviare un pull request al progetto https://github.com/velesin/jasmine-jquery
function testSubmitToHaveBeenPrevented(form) {
    var sent = false;

    form.on('submit', function (e) {
        if (!e.isDefaultPrevented()) {
            sent = true;
        }

        e.preventDefault();
    });

    form.trigger('submit');

    expect(sent).toBe(true);
}

function testSubmitNotToHaveBeenPrevented(form) {
    var sent = false;

    form.on('submit', function (e) {
        if (!e.isDefaultPrevented()) {
            sent = true;
        }

        e.preventDefault();
    });

    form.trigger('submit');

    expect(sent).toBe(false);
}

describe('Scheda prodotto', function () {
    'use strict';

    describe('Situazione iniziale', function () {
        beforeEach(function () {
            loadFixtures('product-single.html');

            Pongho.Shop.productPriceManager();

            this.$price = $('.pongho-product-price');
            this.$size = $('[name="size"]');
            this.$quantity = $('[name="quantity"]');
            this.$addToCart = $('.pongho-product-add-to-cart');
            this.$form = this.$size.closest('form');
        });

        it('La quantità non dovrebbe essere diversa da quella impostata', function () {
            expect(this.$quantity.val()).toBe('1');
        });

        it('La quantità massima dovrebbe essere pari alla disponibilità', function () {
            expect(this.$quantity.attr('max')).toBe('3');
        });

        it('Le taglie senza disponibilità dovrebbero essere disabilitate', function () {
            expect(this.$size.find('[data-availabilities="0"]')).toBeDisabled();
        });

        it('Non avendo la prima opzione vuota nella select delle taglie, dovrei poter acquistare fin da subito', function () {
            expect(this.$addToCart).toHaveClass('enabled');
            expect(this.$form).toHaveClass('enabled');

            testSubmitToHaveBeenPrevented(this.$form);
        });
    });

    describe('Eventi', function () {
        beforeEach(function () {
            loadFixtures('product-single.html');

            Pongho.Shop.productPriceManager();

            this.$price = $('.pongho-product-price');
            this.$size = $('[name="size"]');
            this.$quantity = $('[name="quantity"]');
        });

        it('Quando cambio taglia dovrebbe aggiornare prezzo e quantità massima', function () {
            this.$size.val(3).change();

            expect(this.$price.text()).toBe('€ 22,00');
            expect(this.$quantity.attr('max')).toBe('1');
        });

        it('Quando cambio la quantità dovrebbe aggiornare il prezzo', function () {
            this.$quantity.val(2).change();

            expect(this.$price.text()).toBe('€ 36,00');
        });

        it('Quando cambio taglia e quantità dovrebbe aggiornare il prezzo', function () {
            this.$size.val(4).change();
            this.$quantity.val(3).change();

            expect(this.$price.text()).toBe('€ 69,00');
        });

        it('Non dovrei poter impostare una quantità maggiore delle disponibilità', function () {
            this.$size.val(3).change();
            this.$quantity.val(2).change();

            expect(this.$quantity.attr('max')).toBe('1');
            expect(this.$quantity.val()).toBe('1');
        });

        it('Ma la quantità minima è comunque 1 (uno)', function () {
            this.$quantity.val(0).change();

            expect(this.$quantity.val()).toBe('1');
        });

        it('Dovrebbe essere possibile inserire solo numeri nel campo della quantità', function () {
            this.$quantity.val('foo').change();

            expect(this.$quantity.val()).toBe('1');
        });
    });

    describe('Select delle taglie senza disponibilità', function () {
        beforeEach(function () {
            loadFixtures('product-without-availabilities.html');

            Pongho.Shop.productPriceManager();

            this.$form = $('form');
            this.$addToCart = $('.pongho-product-add-to-cart');
            this.$price = $('.pongho-product-price');
            this.$size = $('[name="size"]');
            this.$quantity = $('[name="quantity"]');
        });

        it('Situazione iniziale', function () {
            expect(this.$price.text()).toBe('');
            expect(this.$quantity.val()).toBe('1');
            expect(this.$quantity.is('[max]')).toBe(false);

            expect(this.$addToCart).not.toHaveClass('enabled');
            expect(this.$form).not.toHaveClass('enabled');
            testSubmitNotToHaveBeenPrevented(this.$form);
        });

        it('Posso inserire qualsiasi quantità', function () {
            this.$size.val(3).change();
            this.$quantity.val(200).change();

            expect(this.$price.text()).toBe('€ 4.400,00');
            expect(this.$quantity.val()).toBe('200');
            expect(this.$quantity.is('[max]')).toBe(false);

            expect(this.$addToCart).toHaveClass('enabled');
            expect(this.$form).toHaveClass('enabled');
            testSubmitToHaveBeenPrevented(this.$form);
        });

        it('Ma la quantità minima è comunque 1 (uno)', function () {
            this.$size.val(3).change();
            this.$quantity.val(0).change();

            expect(this.$price.text()).toBe('€ 22,00');
            expect(this.$quantity.val()).toBe('1');
            expect(this.$quantity.is('[max]')).toBe(false);

            expect(this.$addToCart).toHaveClass('enabled');
            expect(this.$form).toHaveClass('enabled');
            testSubmitToHaveBeenPrevented(this.$form);
        });
    });

    describe('Select delle taglie con la prima opzione vuota', function () {
        beforeEach(function () {
            loadFixtures('product-select-a-size.html');

            Pongho.Shop.productPriceManager();
        });

        it('Non posso acquistare finché non seleziono una taglia', function () {
            var form = $('form');

            expect($('.pongho-product-add-to-cart')).not.toHaveClass('enabled');
            expect(form).not.toHaveClass('enabled');
            testSubmitNotToHaveBeenPrevented(form);
        });
    });

    describe('Selettore delle taglie', function () {

        beforeEach(function () {
            loadFixtures('product-size-selector.html');

            Pongho.Shop.productPriceManager();

            this.$price = $('.pongho-product-price');
            this.$size = $('[name="size"]');
            this.$quantity = $('[name="quantity"]');
            this.$sizeSelector = $('.pongho-size-selector');
            this.$addToCart = $('.pongho-product-add-to-cart');
            this.$form = this.$size.closest('form');
        });

        it('Non posso acquistare finché non seleziono una taglia', function () {
            expect(this.$addToCart).not.toHaveClass('enabled');
            expect(this.$form).not.toHaveClass('enabled');

            testSubmitNotToHaveBeenPrevented(this.$form);
        });

        it('Posso acquistare se seleziono una taglia', function () {
            this.$size.val(1).change();

            expect(this.$addToCart).toHaveClass('enabled');
            expect(this.$form).toHaveClass('enabled');

            testSubmitToHaveBeenPrevented(this.$form);
        });

        it('Non posso acquistare se seleziono una taglia senza disponibilità', function () {
            this.$size.val(5).change();

            expect(this.$addToCart).not.toHaveClass('enabled');
            expect(this.$form).not.toHaveClass('enabled');

            testSubmitNotToHaveBeenPrevented(this.$form);
        });

        it('Dovrebbe creare automaticamente i selettori', function () {
            expect(this.$sizeSelector).toExist();
            expect(this.$sizeSelector.find('ul')).toExist();
            expect(this.$sizeSelector.find('li')).toHaveLength(4); // in tutto sono 5, ma una è disabilitata
            expect(this.$sizeSelector.find('li').first()).toContainText('XS');
        });

        it('Cliccando un selettore dovrebbe cambiare la select delle taglie e quindi il prezzo', function () {
            var
                second = this.$sizeSelector.find('a').eq(1),
                third = this.$sizeSelector.find('a').eq(2),
                spyEvent = spyOnEvent(this.$size, 'change');

            second.click();
            expect(spyEvent).toHaveBeenTriggered();
            expect(this.$price).toContainText('€ 20,00');
            expect(second.closest('li')).toHaveClass('active');

            third.click();
            expect(spyEvent).toHaveBeenTriggered();
            expect(this.$price).toContainText('€ 22,00');
            expect(third.closest('li')).toHaveClass('active');
            expect(second.closest('li')).not.toHaveClass('active');
        });
    });

    describe('Selettore delle taglie con taglia unica', function () {

        beforeEach(function () {
            loadFixtures('product-unique-size-selector.html');

            Pongho.Shop.productPriceManager();

            this.$size = $('[name="size"]');
            this.$addToCart = $('.pongho-product-add-to-cart');
            this.$form = this.$size.closest('form');
        });

        it('Dovrebbe selezionare la taglia in automatico', function () {
            expect(this.$addToCart).toHaveClass('enabled');
            expect(this.$form).toHaveClass('enabled');

            testSubmitToHaveBeenPrevented(this.$form);
        });
    });
});

describe('Archivio prodotti', function () {
    'use strict';

    beforeEach(function () {
        loadFixtures('product-archive.html');

        Pongho.Shop.productPriceManager();

        this.firstProduct = $('#product-1');
        this.secondProduct = $('#product-2');
        this.thirdProduct = $('#product-3');
    });

    it('Quando cambio taglia dovrebbe aggiornare il prezzo solo per il prodotto interessato', function () {
        this.firstProduct.find('[name="size"]').val(2).change();

        expect(this.firstProduct.find('.pongho-product-price').text()).toBe('€ 20,00');
        expect(this.secondProduct.find('.pongho-product-price').text()).toBe('');
        expect(this.thirdProduct.find('.pongho-product-price').text()).toBe('');

        this.secondProduct.find('[name="size"]').val(3).change();

        expect(this.firstProduct.find('.pongho-product-price').text()).toBe('€ 20,00');
        expect(this.secondProduct.find('.pongho-product-price').text()).toBe('€ 10,00');
        expect(this.thirdProduct.find('.pongho-product-price').text()).toBe('');

        this.thirdProduct.find('[name="size"]').val(1).change();

        expect(this.firstProduct.find('.pongho-product-price').text()).toBe('€ 20,00');
        expect(this.secondProduct.find('.pongho-product-price').text()).toBe('€ 10,00');
        expect(this.thirdProduct.find('.pongho-product-price').text()).toBe('€ 12,00');
    });

    it('Quando cambio la quantità dovrebbe aggiornare il prezzo solo per il prodotto interessato', function () {
        this.secondProduct.find('[name="quantity"]').val(3).change();

        expect(this.firstProduct.find('.pongho-product-price').text()).toBe('');
        expect(this.secondProduct.find('.pongho-product-price').text()).toBe('€ 30,00');
        expect(this.thirdProduct.find('.pongho-product-price').text()).toBe('');

        this.firstProduct.find('[name="quantity"]').val(2).change();

        expect(this.firstProduct.find('.pongho-product-price').text()).toBe('€ 36,00');
        expect(this.secondProduct.find('.pongho-product-price').text()).toBe('€ 30,00');
        expect(this.thirdProduct.find('.pongho-product-price').text()).toBe('');

        this.thirdProduct.find('[name="quantity"]').val(4).change();

        expect(this.firstProduct.find('.pongho-product-price').text()).toBe('€ 36,00');
        expect(this.secondProduct.find('.pongho-product-price').text()).toBe('€ 30,00');
        expect(this.thirdProduct.find('.pongho-product-price').text()).toBe('€ 48,00');
    });

    it('Quando cambio taglia e quantità dovrebbe aggiornare il prezzo solo per il prodotto interessato', function () {
        this.thirdProduct.find('[name="size"]').val(3).change();
        this.thirdProduct.find('[name="quantity"]').val(2).change();

        expect(this.firstProduct.find('.pongho-product-price').text()).toBe('');
        expect(this.secondProduct.find('.pongho-product-price').text()).toBe('');
        expect(this.thirdProduct.find('.pongho-product-price').text()).toBe('€ 28,00');

        this.firstProduct.find('[name="size"]').val(3).change();
        this.firstProduct.find('[name="quantity"]').val(2).change();

        expect(this.firstProduct.find('.pongho-product-price').text()).toBe('€ 44,00');
        expect(this.secondProduct.find('.pongho-product-price').text()).toBe('');
        expect(this.thirdProduct.find('.pongho-product-price').text()).toBe('€ 28,00');

        this.secondProduct.find('[name="size"]').val(4).change();
        this.secondProduct.find('[name="quantity"]').val(3).change();

        expect(this.firstProduct.find('.pongho-product-price').text()).toBe('€ 44,00');
        expect(this.secondProduct.find('.pongho-product-price').text()).toBe('€ 30,00');
        expect(this.thirdProduct.find('.pongho-product-price').text()).toBe('€ 28,00');
    });
});
