/**
 * JS per le funzionalità dello shop.
 *
 * @dependency jQuery v1.7+
 * @author Daniele De Nobili
 * @copyright 2012 Web Agency Meta Line S.r.l. (www.metaline.it)
 */

/*jslint plusplus: true, browser: true, devel: true, indent: 4, maxerr: 50 */
/*global $ */

var Pongho;

if (Pongho === undefined) {
    Pongho = {};
}

// https://github.com/kvz/locutus/blob/master/src/php/strings/number_format.js
Pongho.formatNumber = function(number, decimals, decPoint, thousandsSep) {
    number = (number + '').replace(/[^0-9+\-Ee.]/g, '');

    var n = !isFinite(+number) ? 0 : +number,
        prec = !isFinite(+decimals) ? 0 : Math.abs(decimals),
        sep = (typeof thousandsSep === 'undefined') ? ',' : thousandsSep,
        dec = (typeof decPoint === 'undefined') ? '.' : decPoint,
        s;

    var toFixedFix = function (n, prec) {
        var k = Math.pow(10, prec);
        return '' + (Math.round(n * k) / k).toFixed(prec);
    };

    s = (prec ? toFixedFix(n, prec) : '' + Math.round(n)).split('.');

    if (s[0].length > 3) {
        s[0] = s[0].replace(/\B(?=(?:\d{3})+(?!\d))/g, sep);
    }

    if ((s[1] || '').length < prec) {
        s[1] = s[1] || '';
        s[1] += new Array(prec - s[1].length + 1).join('0');
    }

    return s.join(dec);
};

Pongho.Shop = (function () {
    'use strict';

    var pub = {};

    pub.formatPrice = function (n) {
        return "€ " + Pongho.formatNumber(n, 2, ',', '.');
    };

    pub.productPriceManager = function () {
        $('.pongho-product').each(function () {
            var
                $el = $(this),
                $realPrice = $el.find('.pongho-product-real-price'),
                $price = $el.find('.pongho-product-price'),
                $size = $el.find('[name="size"]'),
                $quantity = $el.find('[name="quantity"]'),
                $sizeSelector = $el.find('.pongho-size-selector'),
                $addToCart = $el.find('.pongho-product-add-to-cart'),
                $form = $size.closest('form');

            function updateQuantity() {
                var max = $size.find(':selected').data('availabilities');

                if (!$.isNumeric($quantity.val())) {
                    $quantity.val(1);
                }

                $quantity.attr('max', max);

                if ($quantity.val() > max) {
                    $quantity.val(max);
                } else if ($quantity.val() < 1) {
                    $quantity.val(1);
                }
            }

            function update() {
                updateQuantity();

                var size = $size.find(':selected');

                if (size.data('price') !== null && size.data('availabilities') !== 0) {
                    var sizePrice = parseFloat(size.data('price')) * parseInt($quantity.val(), 10);
                    var sizeRealPrice = parseFloat(size.data('real-price')) * parseInt($quantity.val(), 10);

                    $price.text(
                        Pongho.Shop.formatPrice(
                            sizePrice
                        )
                    );

                    if (sizeRealPrice !== sizePrice) {
                        $realPrice.show().text(
                            Pongho.Shop.formatPrice(
                                sizeRealPrice
                            )
                        );
                    }

                    $addToCart.addClass('enabled');
                    $form.addClass('enabled');
                } else {
                    $addToCart.removeClass('enabled');
                    $form.removeClass('enabled');
                }
            }

            function prepare() {
                updateQuantity();

                $size.on('change', update);
                $quantity.on('change', update);

                $form.on('submit', function (event) {
                    if (!$form.hasClass('enabled')) {
                        event.preventDefault();
                    }
                });

                $size.find('[data-availabilities="0"]').attr('disabled', 'disabled');

                if ($sizeSelector.length) {
                    var selectors = '',
                        $uniqueSize;

                    $size.find('option[value]').not(':disabled').each(function () {
                        var option = $(this);
                        selectors += '<li><a data-size="' + option.attr('value') + '">' + option.text() + '</a></li>';
                    });

                    $sizeSelector
                        .html('<ul>' + selectors + '</ul>')
                        .on('click', 'a', function (event) {
                            event.preventDefault();

                            var selector = $(this);

                            $size.val(selector.data('size')).change();

                            $sizeSelector.find('li').removeClass('active');
                            selector.closest('li').addClass('active');
                        });

                    $uniqueSize = $sizeSelector.find('a');

                    if ($uniqueSize.length === 1) {
                        $uniqueSize.click();
                    }
                } else {
                    if ($size.find(':selected').data('price') !== null) {
                        $addToCart.addClass('enabled');
                        $form.addClass('enabled');
                    }
                }
            }

            prepare();
        });
    };

    pub.cartManager = function () {
        var $table = $('#cart-table'),
            $productsTotalRow = $('#cart-products-total-row'),
            $productsTotal = $('#cart-products-total'),
            $discountRowTpl = $('#js-tpl-discount-row').text(),
            $subtotal = $('#cart-subtotal'),
            $form = $table.closest('form').data('in_ajax', false);

        function updateDiscount(cart) {
            var code = '';

            if ($.isArray(cart.discounts) && cart.discounts.length) {
                $.map(cart.discounts, function (discount) {
                    code += $discountRowTpl
                        .replace('{{label}}', discount.name)
                        .replace('{{value}}', discount.formatValue);
                });

                $productsTotalRow.show();
                $productsTotal.text(Pongho.Shop.formatPrice(cart.productsTotal));
            } else {
                $productsTotalRow.hide();
            }

            $('.js-discount-row').remove();
            $productsTotalRow.after(code);

            $subtotal.text(Pongho.Shop.formatPrice(cart.subtotal));

            $table.trigger('update_total', [cart]);
        }

        function updateTotal(response) {
            var cartSubTotal = 0;

            $table.find('.row').each(function () {
                var row = $(this),
                    rowTotal = row.find('.quantity-field').val() * row.find('.price').data('value');

                cartSubTotal += rowTotal;

                row.find('.sale-total').text(Pongho.Shop.formatPrice(rowTotal));
            });

            if (response.cart.productsTotal !== cartSubTotal) {
                // Se il totale calcolato è diverso da quello restituito dalla risposta AJAX,
                // forzo il refresh della pagina.
                window.location.reload();
            }

            updateDiscount(response.cart);
        }

        function increase(field, increment) {
            if ($form.data('in_ajax')) {
                return;
            }

            field.val(parseInt(field.val(), 10) + increment);
            field.change();
        }

        if ($table.length) {
            updateDiscount(Pongho.ShopOptions.cart);

            $table.find('.row').each(function () {
                var $row = $(this),
                    $quantityField = $row.find('.quantity-field'),
                    $wrapper = $quantityField.closest('.quantity-field-wrapper'),
                    $body = $('body'),
                    originalValue = parseInt($quantityField.val(), 10),
                    minValue = parseInt(($quantityField.data('min') || $quantityField.attr('min')), 10),
                    maxValue = parseInt(($quantityField.data('max') || $quantityField.attr('max')), 10),
                    valValue,
                    newValue,
                    timer;

                $quantityField.on('change keypress keyup blur input', function (event) {
                    if (timer) {
                        clearTimeout(timer);
                    }

                    if ($form.data('in_ajax')) {
                        return;
                    }

                    if (event.type === 'keypress') {
                        var key = event.which;

                        // event.which = 8: Back
                        // event.which = 13: Enter
                        // event.which = 48: 0
                        // event.which = 57: 9

                        if ((key < 48 || key > 57) && key !== 8 && key !== 13) {
                            return;
                        }
                    }

                    valValue = $quantityField.val();
                    newValue = parseInt(valValue, 10);

                    if (valValue != newValue) {
                        if (isNaN(newValue)) {
                            newValue = 1;
                        }

                        $quantityField.val(newValue);
                    }

                    if (originalValue !== newValue) {
                        if (newValue <= minValue) {
                            newValue = minValue;
                        }

                        if (newValue >= maxValue) {
                            newValue = maxValue;
                        }

                        $quantityField.val(newValue);

                        // Questa parte del codice non riesco a testarla. E purtroppo fa fallire numerosi test.
                        // Probabilmente sto sbagliando approccio o forse non ho abbastanza conoscenze...
                        // Per eseguire i test bisogna commentare la parte con timer = setTimeout() e togliere
                        // il commento dalle due righe successive.

                        timer = setTimeout(function () {
                            $form.data('in_ajax', true);
                            $wrapper.addClass('updating');
                            $body.addClass('cart-updating');

                            $.post(
                                $form.attr('action'),
                                $form.serialize(),
                                function (response) {
                                    if (response.error) {
                                        alert(response.message);
                                    } else {
                                        originalValue = newValue;
                                        updateTotal(response);
                                    }

                                    $form.data('in_ajax', false);
                                    $wrapper.removeClass('updating');
                                    $body.removeClass('cart-updating');
                                },
                                'json'
                            );
                        }, 300);

//                        originalValue = newValue;
//                        updateTotal();
                    }
                });

                $row.on('click', '.quantity-field-increase', function (event) {
                    event.preventDefault();
                    increase($quantityField, 1);
                });

                $row.on('click', '.quantity-field-decrease', function (event) {
                    event.preventDefault();
                    increase($quantityField, -1);
                });
            });
        }
    };

    pub.loginManager = function () {
        var $signInType = $('[name="sign-in[type]"]'),
            $signInLogin = $('#shop-signin-login'),
            $signInSubscribe = $('#shop-signin-subscribe'),
            $invoiceAddressMode = $('[name="sign-in[invoice_address_mode]"]'),
            $invoiceAddressBox = $('#shop-signin-invoice-address'),
            $checkoutMode = $('[name="sign-in[checkout_mode]"]'),
            $accountBox = $('#shop-signin-account-fields');

        function toggleSignInType() {
            if ($signInType.filter(':checked').val() === 'subscribe') {
                $signInLogin.hide();
                $signInSubscribe.show();
            } else {
                $signInLogin.show();
                $signInSubscribe.hide();
            }
        }

        function toggleInvoiceAddress() {
            if ($invoiceAddressMode.filter(':checked').val() === 'same') {
                $invoiceAddressBox.hide();
            } else {
                $invoiceAddressBox.show();
            }
        }

        function toggleCheckoutMode() {
            if ($checkoutMode.filter(':checked').val() === 'guest') {
                $accountBox.hide();
            } else {
                $accountBox.show();
            }
        }

        $signInType.on('change', toggleSignInType);
        $invoiceAddressMode.on('change', toggleInvoiceAddress);
        $checkoutMode.on('change', toggleCheckoutMode);

        toggleSignInType();
        toggleInvoiceAddress();
        toggleCheckoutMode();
    };

    pub.addressBookManager = function () {
        var $list = $('#shop-addresses-list'),
            $addresses,
            $contents,
            $addAddressButton,
            $addAddressWrapper;

        function getRealHeight(el) {
            var $el = $(el),
                $clone = $el.clone(),
                height;

            $clone.show().css('height', 'auto').insertAfter($el);
            height = $clone.outerHeight();
            $clone.remove();

            return height;
        }

        if ($list.length) {
            $addresses = $list.find('.shop-address-item');
            $contents = $addresses.find('.shop-address-content');
            $addAddressButton = $('#shop-add-address');
            $addAddressWrapper = $('#shop-add-address-form');

            $list.on('click', '.shop-edit-address', function (event) {
                event.preventDefault();

                var $address = $(this).closest('.shop-address-item'),
                    $content = $address.find('.shop-address-content');

                $content.addClass('loading');
                $addresses.removeClass('open');
                $addAddressButton.removeClass('disabled');
                $address.addClass('open');

                $.getJSON($(this).data('href'), function (response) {
                    $contents.removeAttr('style');

                    $content.html(response.html);
                    $content.removeClass('loading');
                    $content.css('height', getRealHeight($content));
                });
            });

            $list.on('submit', '.shop-address-form', function (event) {
                event.preventDefault();

                var $form = $(this),
                    $content = $form.closest('.shop-address-content');

                $content.addClass('loading');

                $.post(this.action, $form.serialize(), function (response) {
                    if (response.redirect) {
                        window.location.href = response.redirect;

                        return;
                    }

                    $content.html(response.html);
                    $content.removeClass('loading');
                    $content.height(getRealHeight($content));
                }, 'json');
            });

            $addAddressButton.on('click', function (event) {
                event.preventDefault();

                var $content = $addAddressWrapper.find('.shop-address-content');

                // Chiudo gli altri indirizzi
                $addresses.removeClass('open');
                $contents.removeAttr('style');

                // Apro l’indirizzo '#add-address-form'
                $addAddressWrapper.addClass('open');
                $content.css('height', getRealHeight($content));

                $addAddressButton.addClass('disabled');
            });

            if ($addresses.filter(':visible').length) {
                $list.find('.shop-edit-address').filter('[data-open="1"]').click();
            } else {
                $list.find('#shop-add-address').click();
            }
        }
    };

    pub.onDocumentReady = function () {
        pub.productPriceManager();
        pub.cartManager();
        pub.loginManager();
        pub.addressBookManager();
    };

    pub.init = function () {
        $(pub.onDocumentReady);
    };

    return pub;
}());

Pongho.Shop.init();

// Aggiungo il supporto ai nuovi eventi JS di HTML5
// http://jsfiddle.net/rwaldron/GJeez/
(function () {
    "use strict";

    $.each(["input", "contextmenu", "invalid"], function (i, name) {

        $.fn[name] = function (data, fn) {

            if (fn === null) {
                fn = data;
                data = null;
            }

            if (data || fn) {
                return this.bind(name, data, fn);
            }

            return this.trigger(name);
        };
    });

}());


$(document).ready(function () {
    "use strict";


    /**
     * Variabili
     */
    var $table = $('#shop-table'),
        $rows = $('.row', $table),
        $subtotal = $('#cart-subtotal'),
        $taxable = $('#taxable'),
        $total = $('#cart-total'),
        $payments = $('input[name="order[payment_id]"]'),
        $shippings = $('input[name="order[shipping_id]"]'),
        cart,
        o;


    $.ajaxSetup({
        error: function (jqXHR, textStatus, errorThrown) {
            console.log(jqXHR);
            console.log(textStatus); // timeout | error | abort | parsererror
            console.log(errorThrown);
        }
    });


    // Opzioni
    o = $.extend(true, {}, {
        subtotal: 0,
        tax: {
            enabled: false,
            value: 0
        }
    }, Pongho.ShopOptions);


    function priceFormat(n) {
        return Pongho.Shop.formatPrice(n);
    }


    /**
     * Carrello
     */
    function Cart() {
    }

    cart = new Cart();

    // Aggiorna i prezzi nel carrello.
    Cart.prototype.update = function () {

        var subtotal = 0,
            $payment,
            $shipping,
            payment_cost,
            shipping_cost,
            taxable,
            tax_cost,
            total;

        if ($payments.length || $shippings.length) {

            $payment = $payments.filter(':checked');
            $shipping = $shippings.filter(':checked');

            subtotal = o.subtotal || parseFloat($subtotal.data('value'));

            payment_cost = parseFloat($payment.data('cost'));
            shipping_cost = parseFloat($shipping.data('cost'));

            $('#payment-description').text($payment.attr('title'));
            $('#shipping-description').text($shipping.attr('title'));

            $('#payment-cost').text(priceFormat(payment_cost));
            $('#shipping-cost').text(priceFormat(shipping_cost));

            taxable = subtotal + payment_cost + shipping_cost;

            // Aggiungo le tasse se sono abilitate
            if (o.tax.enabled === true) {
                tax_cost = o.tax.value * taxable / 100;
                $('#tax-cost').text(priceFormat(tax_cost));
                total = taxable + tax_cost;

                $taxable.text(priceFormat(taxable));
            } else {
                total = taxable;
            }

            $total.text(priceFormat(total));
        }
    };


    /**
     * Lancia una chiamata ajax per eliminare le singole righe dal carrello.
     * Nel caso ci sia solo una riga in carrello, non fa nulla in quanto deve
     * aggiornarsi la pagina per visualizzare il messaggio del carrello vuoto.
     */
    $table.on('click', '.cart-remove', function (event) {

        if ($rows.length <= 1) {
            // Se è l’unico prodotto in carrello faccio l’eliminazione senza ajax
            // in modo da aggiornare la tabella con il messaggio "carello vuoto"
            return;
        }

        event.preventDefault();

        var btn = $(this),
            row = btn.closest('.row');

        $.post(this.href, function (data) {
            if (data.error) {
                alert(data.message);
            } else {
                row.fadeOut(function () {
                    row.remove();
                    $rows = $('.row', $table);

                    cart.update();
                });
            }
        }, 'json');
    });


    // Gestione checkout
    $payments.change(function () {
        cart.update();
    });

    $shippings.change(function () {
        cart.update();
    });

    cart.update();

});
