<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Payment\Volksbank;

use Application\Admin\Form\FormConfig;
use Application\Core\I18n\Translation\Translator;
use Application\Showcase\Model\Order;
use Application\Showcase\Payment\BasePayment;
use Application\Showcase\Payment\PaymentOptions;
use Application\Showcase\Payment\TicketResponse;
use Pongho\Core\TemplateResponse;
use Pongho\Form\Field\TextField;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\RedirectResponse;
use Pongho\Http\Request;

/**
 * Gestisce il pagamento tramite il gateway della banca Volksbank (Consorzio Triveneto S.p.a.).
 */
class Payment extends BasePayment
{
    /**
     * Azioni possibili.
     */
    const ACTION_PURCHASE = 1;
    const ACTION_AUTHORIZATION = 4;

    /**
     * {@inheritdoc}
     */
    public function preparePayment()
    {
        $this->order->convertCartToOrder(Order::STATUS_PAYMENT);

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function handlePayment(PaymentOptions $options)
    {
        $options = $options->get($this->order);

        return new RedirectResponse($this->getPaymentUrl($options));
    }

    /**
     * {@inheritdoc}
     *
     * @todo Traduzione dei messaggi (potrebbero essere inseriti nel pannello di controllo del metodo di pagamento).
     */
    public function handleCancel($redirect_url)
    {
        if ($this->order->isPaid()) {
            throw new HttpNotFoundException();
        }

        $lang = $this->getContainer()->get('language');
        $view = $this->getContainer()->get('theme_view');

        if (isset($_GET['error'])) {
            $details = unserialize($this->order->payment_details);

            if (isset($details['Error']) && $details['Error'] === $_GET['error']) {
                $this->order->status = Order::STATUS_CART;

                // Il messaggio sarà comunque riferito al caso in cui il pagamento è fallito.
                $view
                    ->setTemplate('message.php')
                    ->assignVars(
                        array(
                            'message_type' => 'alert',
                            'title'        => 'Attenzione',
                            'message'      => 'Si è verificato un errore durante il pagamento.' . (isset($details['ErrorText']) ? ('<br><br><span style="color: red;">' . $details['ErrorText'] . '</span>') : '') . '<br><br><a href="' . url(
                                    $redirect_url
                                ) . '" class="btn">Torna al carrello</a>',
                        )
                    );
            }
        } elseif (in_array($this->order->status, array(Order::STATUS_CART, Order::STATUS_PAYMENT_FAILED))) {
            // Controllo lo stato 'payment_failed' per quando il pagamento fallisce e lo stato 'cart' nel caso sia stata aggiornata la pagina.
            if ($this->order->status === Order::STATUS_CART) {
                // Mi assicuro di essere arrivato qui dopo il processo di pagamento.
                $details = unserialize($this->order->payment_details);

                if (!is_array($details) || !isset($details['processed']) || !$details['processed']) {
                    $this->order->status = Order::STATUS_CART;

                    // Come diavolo sei arrivato qua? Torna al carrello!
                    return new RedirectResponse($redirect_url);
                }
            } else {
                // Riporto l’ordine allo stato di carrello, in modo che l’utente possa riprovare il pagamento.
                $this->order->status = Order::STATUS_CART;
            }

            // Il messaggio sarà comunque riferito al caso in cui il pagamento è fallito.
            $view
                ->setTemplate('message.php')
                ->assignVars(
                    array(
                        'message_type' => 'alert',
                        'title'        => 'Attenzione',
                        'message'      => 'Il pagamento non è andato a buon fine. Attendi una e-mail con l’esito della transazione prima di riprovare.<br><br><a href="' . url(
                                $redirect_url
                            ) . '" class="btn">Torna al carrello</a>',
                    )
                );
        } else {
            // C'è un qualche errore che ha impedito il corretto procedimento del pagamento.
            // Invio la notifica con i dettagli.
            $details = unserialize($this->order->payment_details);
            $post = print_r($_POST, true);
            $get = print_r($_GET, true);
            $server = print_r($_SERVER, true);

            $message = <<<EOT

Si è verificato un possibile errore durante il pagamento tramite banca Volksbank. Di seguito i dettagli della richiesta:

ORDER DETAILS = {$details}
GET = {$get}
POST = {$post}
SERVER = {$server}

EOT;

            mail('log@metaline.it', 'Errore con pagamento Volksbank.', trim($message), 'From: no-reply@metaline.it');

            // Potrei riportare l'ordine allo stato di carrello, nel caso di errore, ma in realtà non ho modo di sapere a cosa è dovuto il problema.
            // Vedi nella documentazione la sezione Capitolo 3 -> Messaggio NotificationMessage -> ErrorURL.

            // Mostro un messaggio all’utente.
            $view
                ->setTemplate('message.php')
                ->assignVars(
                    array(
                        'message_type' => 'alert',
                        'title'        => 'Attenzione',
                        'message'      => 'Si è verificato un errore inatteso durante il pagamento, ma la transazione potrebbe essere andata comunque a buon fine. Ti consigliamo di controllare la tua e-mail per ulteriori dettagli sull’esito della transazione. I nostri tecnici sono già stati informati del problema.',
                    )
                );
        }

        return new TemplateResponse($view);
    }

    /**
     * {@inheritdoc}
     */
    public function handleListener(Request $request, PaymentOptions $options)
    {
        if ($request->getMethod() !== 'POST') {
            throw new \RuntimeException(sprintf('HTTP Method %s not allowed.', $request->getMethod()));
        }

        $order_id = isset($_GET['o']) ? intval($_GET['o']) : 0;

        /** @var Order $order */
        if (($order = Order::find($order_id)) === null) {
            throw new \RuntimeException('Order not found.');
        }

        if ($order->payment->handler_class !== __CLASS__) {
            throw new \RuntimeException(sprintf('Order not found with %s handler.', __CLASS__));
        }

        if ($order->isPaid()) {
            throw new \RuntimeException('Order paid.');
        }

        $this->setOrder($order);
        $listener = new Listener($order, $this->isDebug());

        return $listener->handle($request, $options);
    }

    /**
     * {@inheritdoc}
     */
    public function handleTicket()
    {
        return new TicketResponse(true);
    }

    /**
     * {@inheritdoc}
     */
    public static function getFormFieldsConfig(FormConfig $config, Translator $translator, array $options = [])
    {
        return [
            'user'     => [
                'class' => TextField::class,
                'label' => $translator->trans('User'),
            ],
            'password' => [
                'class' => TextField::class,
                'label' => $translator->trans('Password'),
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public static function getDefaultOptions()
    {
        return array(
            'user'     => '',
            'password' => '',
        );
    }

    /**
     * Restituisce l’indirizzo per il pagamento dell’ordine.
     *
     * Il metodo fa una richiesta (chiamata `PaymentInit`) tramite cURL al Payment Gateway di Consorzio Triveneto S.p.a.
     * In risposta riceve l’URL della Hosted Payment Page (HPP) e il PaymentID.
     *
     * @param array $options Elenco delle opzioni.
     * @throws \RuntimeException
     * @throws \OutOfBoundsException
     * @return string L’indirizzo del carrello per il pagamento.
     *
     * @todo   Traduzione del titolo dell’ordine
     * @todo   Internazionalizzare il parametro `langid`.
     */
    protected function getPaymentUrl(array $options = array())
    {
        $order = $this->getOrder();

        if ($options['user'] == '') {
            throw new \OutOfBoundsException('You must set "Utente" option for proceed to UniCredit payment!');
        }

        if ($options['password'] == '') {
            throw new \OutOfBoundsException('You must set "Password" option for proceed to UniCredit payment!');
        }

        $parameters = array(
            'id'           => $options['user'],
            'password'     => $options['password'],
            'action'       => self::ACTION_PURCHASE,
            'amt'          => $this->formatPrice($order->total()),
            'currencycode' => 978, // Euro
            'langid'       => 'ITA',
            'responseURL'  => $options['listener_url'],
            'errorURL'     => $options['cancel_url'],
            'trackid'      => $this->isDebug() ? sprintf('%d-%d', time(), $order->id) : $order->id,
            'udf1'         => $order->payment_hash,
            'udf2'         => '',
            'udf3'         => 'EMAILADDR:' . $order->getCustomerEmail(),
            'udf4'         => '',
            'udf5'         => '',
        );

        $channel = curl_init($this->getPaymentInitUrl());

        curl_setopt($channel, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($channel, CURLOPT_POST, 1);
        curl_setopt($channel, CURLOPT_POSTFIELDS, http_build_query($parameters, '', '&'));
        curl_setopt($channel, CURLOPT_RETURNTRANSFER, 1);

        $response = curl_exec($channel);
        curl_close($channel);

        if (substr($response, 0, 7) === '!ERROR!') {
            throw new \RuntimeException($response);
        } else {
            $pos = strpos($response, ':http');
            $payment_id = substr($response, 0, $pos);
            $redirect_url = substr($response, $pos + 1);

            $details = array('paymentid' => $payment_id);
            $order->payment_details = serialize($details);
            $order->save();

            return $redirect_url . '?PaymentID=' . $payment_id;
        }
    }

    /**
     * Formatta il prezzo secondo le specifiche del Consorzio Triveneto S.p.a.
     *
     * @param float $price Il prezzo da formattare.
     * @return string Il prezzo formattato.
     */
    protected function formatPrice($price)
    {
        return number_format($price, 2, '.', '');
    }

    /**
     * Restituisce l’indirizzo per la richiesta PaymentInit.
     *
     * @return string
     */
    protected function getPaymentInitUrl()
    {
        return $this->isDebug(
        ) ? 'https://test4.constriv.com/cg301/servlet/PaymentInitHTTPServlet' : 'https://www.constriv.com/cg/servlet/PaymentInitHTTPServlet';
    }
}
