<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Payment\Unicredit;

use Application\Showcase\Model\Order;
use Application\Showcase\Payment\BaseListener;
use Application\Showcase\Payment\ListenerResponse;
use Application\Showcase\Payment\PaymentOptions;
use Pongho\Http\Request;
use Pongho\Utilities\DateTime;

/**
 * Gestisce le notifiche di pagamento provenienti da PagOnline.
 */
class Listener extends BaseListener
{
    /**
     * {@inheritdoc}
     */
    public function handle(Request $request, PaymentOptions $options)
    {
        $options = $options->get($this->order);
        $order_confirmed = false;

        if ($this->isValidTransaction($request, $options) && isset($_GET['statoattuale'])) {
            switch ($_GET['statoattuale']) {
                case 'ON': // Nuovo ordine (accesso al pannello di Unicredit)
                case 'RO': // Richiesto (appena dopo ON)
                case 'IN': // Ordine inserito (appena dopo RO)
                    $this->order->status = Order::STATUS_PAYMENT;
                    break;

                case 'AB': // Ordine abbandonato dal pannello oppure chiudendo il browser
                    if ($this->order->status === Order::STATUS_PAYMENT) {
                        $this->order->status = Order::STATUS_CART;
                    }
                    break;

                case 'OK': // Ordine autorizzato, che deve essere confermato dal merchant
                    $this->order->status = Order::STATUS_PAYMENT_AUTHORIZED;
                    $this->order->paid_at = new DateTime();
                    $order_confirmed = true;
                    break;

                case 'IC': // Ordine confermato, ma non ancora contabilizzato
                    if ($this->order->status === Order::STATUS_PAYMENT || $this->order->status === Order::STATUS_PAYMENT_AUTHORIZED) {
                        $this->order->status = Order::STATUS_ORDER;

                        if (!$this->order->isPaid()) {
                            $this->order->paid_at = new DateTime();
                            $order_confirmed = true;
                        }
                    }

                    break;

                case 'CO': // Ordine contabilizzato
                    // Non fa niente
                    break;

                case 'ST': // Ordine stornato
                    $this->order->status = Order::STATUS_INVOICE_CANCELED;
                    break;

                case 'EX': // Scaduto, causa scadenza bancaria
                case 'KO': // Autorizzazione fallita
                    $this->order->status = Order::STATUS_PAYMENT_FAILED;
                    break;
            }
        }

        return new ListenerResponse($order_confirmed);
    }

    /**
     * Verifica la validità della transazione.
     *
     * @param \Pongho\Http\Request $request
     * @param array                $options
     * @return bool
     */
    protected function isValidTransaction(Request $request, array $options)
    {
        return ($this->isVerifiedMac($request, $options) && $this->isValidMerchant($request, $options));
    }

    /**
     * Controlla la validità del MAC.
     *
     * @param \Pongho\Http\Request $request
     * @param array                $options
     * @return bool
     * @throws \RuntimeException
     */
    protected function isVerifiedMac(Request $request, array $options)
    {
        if (Mac::validate(urldecode($request->server->get('QUERY_STRING')), $options['secret_string'])) {
            return true;
        }

        throw new \RuntimeException('MAC validation fail.');
    }

    /**
     * Controlla il numero del commerciante.
     *
     * @param \Pongho\Http\Request $request
     * @param array                $options
     * @return bool
     * @throws \RuntimeException
     */
    protected function isValidMerchant(Request $request, array $options)
    {
        if (isset($_GET['numeroCommerciante']) && $_GET['numeroCommerciante'] === $options['numeroCommerciante']) {
            return true;
        }

        throw new \RuntimeException('Merchant ID is not valid.');
    }
}
