<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Payment\PayPal;

use Application\Admin\Form\FormConfig;
use Application\Core\I18n\Translation\Translator;
use Application\Core\Model\Country;
use Application\Core\Model\Province;
use Application\Showcase\Model\Order;
use Application\Showcase\Payment\BasePayment;
use Application\Showcase\Payment\PaymentFailResponse;
use Application\Showcase\Payment\PaymentOptions;
use Application\Showcase\Payment\PaymentResponse;
use Application\Showcase\Payment\TicketResponse;
use Pongho\Form\Field\TextField;
use Pongho\Http\RedirectResponse;
use Pongho\Http\Request;
use Pongho\Sdk\PayPal\PaymentsStandard\Address;
use Pongho\Sdk\PayPal\PaymentsStandard\CartUpload\AggregateCartUploadButton;

/**
 * Gestisce il pagamento tramite il carrello semplice di PayPal.
 */
class Payment extends BasePayment
{
    /**
     * {@inheritdoc}
     */
    public function preparePayment()
    {
        $this->order->convertCartToOrder(Order::STATUS_PAYMENT);

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function handlePayment(PaymentOptions $options)
    {
        $options = $options->get($this->order);

        return new PaymentResponse(new RedirectResponse($this->getPaymentUrl($options)));
    }

    /**
     * {@inheritdoc}
     */
    public function handleCancel($redirect_url)
    {
        $this->order->status = Order::STATUS_CART;

        return new PaymentFailResponse(new RedirectResponse($redirect_url));
    }

    /**
     * {@inheritdoc}
     */
    public function handleListener(Request $request, PaymentOptions $options)
    {
        if ($request->getMethod() !== 'POST') {
            throw new \RuntimeException(sprintf('HTTP Method %s not allowed.', $request->getMethod()));
        }

        if (!isset($_POST['custom'])) {
            throw new \RuntimeException('POST field "custom" not exists.');
        }

        $custom = unserialize($_POST['custom']);

        /** @var Order $order */
        if (($order = Order::find($custom['order'])) === null) {
            throw new \RuntimeException('Order not found.');
        }

        if ($order->payment->handler_class !== __CLASS__) {
            throw new \RuntimeException(sprintf('Order not found with %s handler.', __CLASS__));
        }

        if ($order->isPaid()) {
            throw new \RuntimeException('Order paid.');
        }

        $this->setOrder($order);
        $listener = new Listener($order, $this->isDebug());

        return $listener->handle($request, $options);
    }

    /**
     * {@inheritdoc}
     */
    public function handleTicket()
    {
        return new TicketResponse();
    }

    /**
     * {@inheritdoc}
     */
    public static function getFormFieldsConfig(FormConfig $config, Translator $translator, array $options = [])
    {
        return [
            'email' => [
                'class' => TextField::class,
                'label' => $translator->trans('Email'),
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public static function getDefaultOptions()
    {
        return array(
            'email' => '',
        );
    }

    /**
     * Restituisce l’indirizzo per il pagamento di questo ordine con PayPal.
     *
     * @param array $options Elenco delle opzioni.
     * @throws \OutOfBoundsException
     * @return string L’indirizzo del carrello di PayPal.
     */
    protected function getPaymentUrl(array $options = array())
    {
        $order = $this->getOrder();

        /** @var Province $province */
        $province = Province::find($order->invoice_province_id);

        /** @var Country $country */
        $country = Country::find($order->invoice_country_id);

        if (!$country) {
            $country = Country::find($order->shipping_country_id);
        }

        if (!$country) {
            throw new \RuntimeException('The country is not defined!');
        }

        if ($options['email'] == '') {
            throw new \OutOfBoundsException('You must set an email for proceed to PayPal payment!');
        }

        $pp_button = new AggregateCartUploadButton($this->isDebug());

        $pp_button
            ->setBusiness($options['email'])
            ->setCurrencyCode($country->getCurrencyCode())
            ->setAmount($order->total())
            ->setName($this->getPayPalCartName($order))
            ->setAddress(
                Address::newInstance()
                    ->setFirstName($order->invoice_name)
                    ->setLastName($order->invoice_surname)
                    ->setAddress1($order->invoice_address1)
                    ->setAddress2($order->invoice_address2)
                    ->setCity($order->invoice_city)
                    ->setState($province ? $province->name : '')
                    ->setZip($order->invoice_postcode)
                    ->setCountry($country->getCode())
                    ->setLc($country->getCode())
                    ->setNightPhoneA($country->getPhoneCode())
                    ->setNightPhoneB($order->invoice_telephone),
                true)
            ->setEmail($order->getCustomerEmail())
            ->setCustom(serialize(array('order' => $order->id)))
            ->setShippingAddressMode(AggregateCartUploadButton::SHIPPING_MODE_NO);

        // Pagina di ritorno se il pagamento va a buon fine
        if (array_key_exists('ticket_url', $options)) {
            $pp_button->setReturnUrl($options['ticket_url']);
        }

        // Pagina di ritorno per annullare il pagamento
        if (array_key_exists('cancel_url', $options)) {
            $pp_button->setCancelUrl($options['cancel_url']);
        }

        // Pagina di ritorno per annullare il pagamento
        if (array_key_exists('listener_url', $options)) {
            $pp_button->setNotifyUrl($options['listener_url']);
        }

        return $pp_button->getPaymentUrl();
    }

    /**
     * Restituisce l’indirizzo per il carrello di PayPal.
     *
     * @return string
     */
    protected function getPayPalUrl()
    {
        return $this->isDebug() ? 'https://www.sandbox.paypal.com/webscr' : 'https://www.paypal.com/cgi-bin/webscr';
    }

    /**
     * @param Order $order
     * @return string
     */
    protected function getPayPalCartName(Order $order)
    {
        /** @var \Application\Core\Localization $lang */
        $lang = $this->container->get('language');

        return sprintf($lang->get('paypal_order_name'), $order->getId());
    }
}
