<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Payment;

use Application\Admin\Form\FormConfig;
use Application\Core\I18n\Translation\Translator;
use Application\Core\Localization;
use Application\Showcase\Model\Payment;
use Application\Showcase\Model\Order;
use Pongho\DependencyInjection\Container;
use Pongho\Http\Request;

/**
 * Base per i gestori dei pagamenti.
 */
abstract class BasePayment
{
    /**
     * Modello dell’ordine.
     *
     * @var Order
     */
    protected $order;

    /**
     * Modello del pagamento.
     *
     * @var Payment
     */
    protected $payment;

    /**
     * Contenitore delle dipendenze.
     *
     * @var Container
     */
    protected $container;

    /**
     * Indica se è attivo il debug per l’ambiente di testing.
     *
     * @var bool
     */
    protected $is_debug = false;

    /**
     * Costruttore.
     *
     * @param Order     $order     L’ordine da elaborare.
     * @param Container $container Contenitore delle dipendenze.
     * @param bool      $is_debug  Indica se è attiva la modalità di debug.
     */
    public function __construct(Order $order = null, Container $container = null, $is_debug = false)
    {
        $this->order = $order;
        $this->container = $container;
        $this->is_debug = (bool) $is_debug;
    }

    /**
     * Imposta l’ordine.
     *
     * @param Order $order L’ordine da elaborare.
     * @return $this
     */
    public function setOrder(Order $order)
    {
        $this->order = $order;

        return $this;
    }

    /**
     * Restituisce l’ordine.
     *
     * @param bool $thrown_exception Se lanciare l’eccezione nel caso non ci sia un ordine impostato.
     * @return Order Il modello dell’ordine.
     *
     * @throws \BadMethodCallException
     */
    public function getOrder($thrown_exception = true)
    {
        if ($this->order === null && $thrown_exception) {
            throw new \BadMethodCallException('Order is not set!');
        }

        return $this->order;
    }

    /**
     * Permette di impostare un modello di pagamento da usare in mancanza di un ordine.
     *
     * @param Payment $payment Il modello del pagamento scelto.
     * @return $this
     *
     * @throws \BadMethodCallException
     */
    public function setPayment(Payment $payment)
    {
        if ($this->order !== null) {
            throw new \BadMethodCallException('You can not set a payment if the order is already set!');
        }

        $this->payment = $payment;

        return $this;
    }

    /**
     * Restituisce il modello del pagamento.
     *
     * @return Payment
     */
    public function getPayment()
    {
        if ($this->order === null) {
            return $this->payment;
        }

        return $this->getOrder()->payment;
    }

    /**
     * Imposta il contenitore delle dipendenze.
     *
     * @param Container $container
     * @return $this
     */
    public function setContainer(Container $container)
    {
        $this->container = $container;

        return $this;
    }

    /**
     * Restituisce il contenitore delle dipendenze.
     *
     * @return Container $container
     */
    public function getContainer()
    {
        return $this->container;
    }

    /**
     * Imposta la modalità di debug.
     *
     * @param bool $is_debug
     * @return $this
     */
    public function setDebug($is_debug)
    {
        $this->is_debug = (bool) $is_debug;

        return $this;
    }

    /**
     * Indica se è attiva la modalità di debug.
     *
     * @return bool
     */
    public function isDebug()
    {
        return $this->is_debug;
    }

    /**
     * Restituisce il nome del metodo di pagamento.
     *
     * @return string
     *
     * @throws \BadMethodCallException
     */
    public static function getName()
    {
        $segments = explode('\\', get_called_class());

        return strtolower($segments[count($segments) - 2]);
    }

    /**
     * Prepara l’ordine per il pagamento.
     *
     * @return $this
     */
    abstract public function preparePayment();

    /**
     * Esegue il pagamento e restituisce la risposta HTTP da restituire in checkout.
     *
     * @param PaymentOptions $options Elenco delle opzioni necessarie ad eseguire il pagamento.
     * @return PaymentResponse
     */
    abstract public function handlePayment(PaymentOptions $options);

    /**
     * Restituisce la risposta HTTP in caso di annullamento del pagamento.
     *
     * @param string $redirect_url Il link di redirect da seguire dopo l’annullamento del pagamento.
     * @return PaymentFailResponse
     */
    abstract public function handleCancel($redirect_url);

    /**
     * Restituisce la risposta HTTP per il listener del pagamento.
     *
     * @param Request        $request La richiesta HTTP da soddisfare.
     * @param PaymentOptions $options Elenco delle opzioni necessarie ad eseguire il pagamento.
     * @return ListenerResponse È una risposta HTTP particolare che prevede alcuni metodi aggiuntivi.
     */
    abstract public function handleListener(Request $request, PaymentOptions $options);

    /**
     * Restituisce la risposta HTTP per lo scontrino dell’ordine.
     *
     * @return TicketResponse È una risposta HTTP particolare che prevede alcuni metodi aggiuntivi.
     */
    abstract public function handleTicket();

    /**
     * Indica se il gateway è in grado di fornire lo stato dell’ordine.
     *
     * @return bool
     */
    public function supportGatewayDetails()
    {
        return false;
    }

    /**
     * Restituisce lo stato dell’ordine, in un formato gestibile da Pongho, chiedendolo al gateway.
     *
     * @return string
     *
     * @throws \BadMethodCallException
     */
    public function getOrderStatus()
    {
        throw new \BadMethodCallException(sprintf(
            'Method %s is not supported by the %s gateway!',
            __METHOD__,
            self::getName()
        ));
    }

    /**
     * Restituisce i dettagli dell’ordine chiedendoli al gateway.
     *
     * Ogni gateway può restituire dati diversi, ma il parametro `status` è obbligatorio!
     *
     * @return array
     *
     * @throws \BadMethodCallException
     */
    public function getOrderDetails()
    {
        throw new \BadMethodCallException(
            sprintf('Method %s is not supported by the %s gateway!', __METHOD__, self::getName())
        );
    }

    /**
     * Avvisa il gateway che l’ordine è stato evaso.
     */
    public function processOrder()
    {
    }

    /**
     * Avvisa il gateway che l’ordine è stato annullato.
     */
    public function cancelOrder()
    {
    }

    /**
     * Restituisce il codice per impostare le opzioni in amministrazione.
     *
     * @param Localization $lang    La localizzazione da usare.
     * @param array        $options Elenco delle opzioni.
     * @return string Il codice HTML per la stampa dei campi.
     *
     * @deprecated
     */
    public static function getFieldsCode(Localization $lang, array $options = array())
    {
        return '';
    }

    /**
     * @param FormConfig $config
     * @param Translator $translator
     * @param array      $options
     * @return array
     */
    public static function getFormFieldsConfig(FormConfig $config, Translator $translator, array $options = [])
    {
        return [];
    }

    /**
     * Restituisce le opzioni di default.
     *
     * @return array
     */
    public static function getDefaultOptions()
    {
        return [];
    }
}
