<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Model;

use ActiveRecord\Base as Model;

/**
 * Modello per i tipi di taglia.
 *
 * @property int $id
 * @property int $module_site_id
 * @property int $node_id
 * @property string $title
 * @property \Application\Showcase\Model\Node $node
 */
class SizeType extends Model
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'nodes_sizes_types';

    /**
     * Relazioni 'belongs_to'.
     *
     * @var array
     */
    public static $belongs_to = [
        ['node', 'model' => Node::class],
    ];

    /**
     * {@inheritdoc}
     */
    public function delete()
    {
        if (!$this->isDeletable()) {
            return true;
        }

        if ($this->deleteSizeNames() && parent::delete()) {
            return true;
        }

        return false;
    }

    /**
     * Indica se il record è eliminabile.
     *
     * @return bool
     */
    public function isDeletable()
    {
        $count = Size::count([
            'joins'      => 'INNER JOIN ' . SizeName::tableName() . ' AS sn ON sn.id = `from`.size_name_id',
            'conditions' => ['sn.size_type_id = ?', $this->id],
        ]);

        return (bool)($count === 0);
    }

    /**
     * Restituisce la lista delle taglie associate a questo tipo di taglia.
     *
     * La lista restituita contiene modelli di `\Application\Showcase\Model\SizeName`.
     *
     * @return \Application\Showcase\Model\SizeName[]
     */
    public function getSizes()
    {
        $options = [
            'conditions' => ['size_type_id = ?', $this->id],
            'order'      => 'position ASC',
        ];

        $sizes = [];
        /** @var \Application\Showcase\Model\SizeName $size */
        foreach (SizeName::all($options) as $size) {
            $sizes[$size->id] = $size;
        }

        return $sizes;
    }

    /**
     * Imposta le taglie associate a questo tipo di taglia.
     *
     * L'array passato contiene un array del tipo:
     *
     *   array(
     *     array('name' => 'foo', 'position' => 1, 'id' => 123),
     *     array('name' => 'bar', 'position' => 2),
     *   )
     *
     * @param array $sizes Elenco delle nuove taglie.
     */
    public function setSizes(array $sizes)
    {
        /**
         * Ciclo le taglie già inserite in modo da ottenere un elenco id / taglia.
         *
         * @var \Application\Showcase\Model\SizeName[] $old_sizes
         */
        $old_sizes = [];
        foreach ($this->getSizes() as $old) {
            $old_sizes[$old->id] = $old;
        }

        // Ciclo le nuove taglie
        $new_sizes = [];
        foreach ($sizes as $size) {
            if (isset($size['id']) && isset($old_sizes[$size['id']])) {
                // Taglia esistente: mi limito ad aggiornare gli attributi.
                $old_sizes[$size['id']]->updateAttributes($size);
                $old_sizes[$size['id']]->save();
            } else {
                // Nuova taglia: la aggiungo in DB e la inserisco nell'elenco delle vecchie taglie.
                $size['size_type_id'] = $this->id;

                /** @var \Application\Showcase\Model\SizeName $new */
                $new = SizeName::create($size);

                $old_sizes[$new->id] = $new;
                $size['id'] = $new->id;
            }

            // Creo un elenco con le nuove taglie.
            $new_sizes[$size['id']] = $size;
        }

        // Ciclo di nuovo le vecchie taglie per eliminare quelle non più presenti nell'elenco passato.
        foreach ($old_sizes as $size_id => $size) {
            if (!isset($new_sizes[$size_id])) {
                $size->delete();
            }
        }
    }

    /**
     * Elimina le taglie associate.
     *
     * @return bool
     */
    public function deleteSizeNames()
    {
        $options = [
            'conditions' => ['size_type_id = ?', $this->id],
        ];

        foreach (SizeName::all($options) as $size) {
            if (!$size->delete()) {
                return false;
            }
        }

        return true;
    }
}
