<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Model;

use ActiveRecord\Base as Model;
use Application\Showcase\Utilities\Taxation;

/**
 * Modello per la gestione dei metodi di trasporto.
 *
 * @property int    $id
 * @property string $name
 * @property string $description
 */
class Shipping extends Model
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'shippings';

    /**
     * Condizioni da controllare per il calcolo dei costi di spedizione degli ordini.
     *
     * @var array
     */
    private $conditions;

    /**
     * {@inheritdoc}
     */
    public function delete()
    {
        if ($this->deleteShippingConditions()) {
            return parent::delete();
        }

        return false;
    }

    /**
     * Restituisce il nome del metodo di spedizione.
     *
     * @return string
     */
    public function name()
    {
        return $this->name;
    }

    /**
     * Restituisce la descrizione del metodo di spedizione.
     *
     * @return string
     */
    public function description()
    {
        return $this->description;
    }

    /**
     * Restituisce il costo del metodo di spedizione per l’ordine passato.
     *
     * @param \Application\Showcase\Model\Order        $order    Ordine sul quale calcolare il costo.
     * @param \Application\Showcase\Utilities\Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return float
     */
    public function cost(Order $order, Taxation $taxation = null)
    {
        $cost = $this->calculateCost($order);

        return $taxation === null ? $cost : $taxation->calculate($cost);
    }

    /**
     * Restituisce il costo formattato del metodo di spedizione per l’ordine passato.
     *
     * @param \Application\Showcase\Model\Order        $order    Ordine sul quale calcolare il costo.
     * @param \Application\Showcase\Utilities\Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return string
     */
    public function formatCost(Order $order, Taxation $taxation = null)
    {
        return format_price($this->cost($order, $taxation));
    }

    /**
     * Restituisce le condizioni da controllare per il calcolo dei costi di spedizione.
     *
     * @return array
     */
    public function getShippingConditions()
    {
        if ($this->conditions === null) {
            $this->conditions = [];

            $options = [
                'conditions' => ['shipping_id = ?', $this->id],
                'order'      => 'condition_by_price',
            ];

            /** @var ShippingCondition $condition */
            foreach (ShippingCondition::all($options) as $condition) {
                $this->conditions[$condition->id] = $condition;
            }
        }

        return $this->conditions;
    }

    /**
     * Permette di aggiungere una nuova condizione.
     *
     * Restituisce il modello della nuova condizione in caso di successo, oppure `false` in caso di errore.
     *
     * @param array $attributes
     * @return mixed
     */
    public function addShippingCondition(array $attributes)
    {
        $attributes['shipping_id'] = $this->id;
        $attributes['condition_by_price'] = parse_price($attributes['condition_by_price']);
        $attributes['cost'] = parse_price($attributes['cost']);

        /** @var ShippingCondition $condition */
        if (!($condition = ShippingCondition::create($attributes))) {
            return false;
        }

        $this->conditions[$condition->id] = $condition;

        return $condition;
    }

    /**
     * Permette di aggiornare una condizione.
     *
     * Restituisce il modello della condizione aggiornata in caso di successo, oppure `false` in caso di errore.
     *
     * @param integer $condition_id
     * @param array   $attributes
     * @return mixed
     */
    public function updateShippingCondition($condition_id, array $attributes)
    {
        /** @var ShippingCondition $condition */
        if (($condition = ShippingCondition::find($condition_id)) === null) {
            return false;
        }

        $attributes['condition_by_price'] = parse_price($attributes['condition_by_price']);
        $attributes['cost'] = parse_price($attributes['cost']);

        $condition->updateAttributes($attributes);

        return $condition->save();
    }

    /**
     * Permette di eliminare una condizione.
     *
     * Restituisce `true` in caso di successo, oppure `false` in caso di errore.
     *
     * @param integer $condition_id
     * @return bool
     */
    public function deleteShippingCondition($condition_id)
    {
        /** @var ShippingCondition $condition */
        if (($condition = ShippingCondition::find($condition_id)) === null) {
            return true;
        }

        if ($condition->delete()) {
            unset($this->conditions[$condition_id]);

            return true;
        }

        return false;
    }

    /**
     * Permette di eliminare tutte le condizioni.
     *
     * Restituisce `true` in caso di successo, oppure `false` in caso di errore.
     *
     * @return bool
     */
    public function deleteShippingConditions()
    {
        $options = [
            'conditions' => ['shipping_id = ?', $this->id],
        ];

        /** @var ShippingCondition $condition */
        foreach (ShippingCondition::all($options) as $condition) {
            if (!$condition->delete()) {
                return false;
            }
        }

        return true;
    }

    /**
     * @param Order $order
     * @return int
     */
    protected function calculateCost(Order $order)
    {
        $cost = 0;
        foreach ($this->getShippingConditions() as $condition) {
            if ($order->subtotal >= $condition->condition_by_price) {
                $cost = $condition->cost;
            }
        }

        return $cost;
    }
}
