<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Model;

use ActiveRecord\Base as Model;
use Application\Showcase\Payment\Simple\Payment as SimplePaymentMethod;
use Application\Showcase\Utilities\Taxation;

/**
 * Modello per la gestione dei metodi di pagamento.
 *
 * @property int    $id
 * @property string $name
 * @property string $checkout_description
 * @property string $email_description
 * @property float  $cost
 * @property string $handler_class
 * @property string $handler_options
 * @property int    $position
 * @property bool   $is_enabled
 */
class Payment extends Model
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'payments';

    /**
     * @return Payment[]
     */
    public static function allEnabled()
    {
        return self::all([
            'conditions' => ['is_enabled = :enabled', 'enabled' => true],
            'order'      => 'position ASC',
        ]);
    }

    /**
     * @param string $className
     * @return Payment
     */
    public static function findByHandlerClass($className)
    {
        return self::first([
            'conditions' => [
                'handler_class = :handler AND is_enabled = :enabled',
                'handler' => $className,
                'enabled' => true,
            ],
        ]);
    }

    /**
     * Restituisce il nome del metodo di pagamento.
     *
     * @return string
     */
    public function name()
    {
        return $this->name;
    }

    /**
     * Restituisce la descrizione del metodo di pagamento.
     *
     * @return string
     */
    public function description()
    {
        return $this->checkout_description;
    }

    /**
     * Restituisce il costo del metodo di pagamento.
     *
     * @param \Application\Showcase\Model\Order        $order    Ordine sul quale calcolare il costo.
     * @param \Application\Showcase\Utilities\Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return float
     */
    public function cost(Order $order, Taxation $taxation = null)
    {
        return $taxation === null ? $this->calculateCost($order) : $taxation->calculate($this->cost);
    }

    /**
     * Restituisce il costo formattato del metodo di pagamento.
     *
     * @param \Application\Showcase\Model\Order        $order    Ordine sul quale calcolare il costo.
     * @param \Application\Showcase\Utilities\Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return string
     */
    public function formatCost(Order $order, Taxation $taxation = null)
    {
        return format_price($this->cost($order, $taxation));
    }

    /**
     * Restituisce la classe `handler` per la gestione del pagamento in fase di checkout.
     *
     * @return \Application\Showcase\Payment\BasePayment
     */
    public function getHandler()
    {
        $handler_class = $this->getHandlerClass();

        /** @var \Application\Showcase\Payment\BasePayment $handler */
        $handler = new $handler_class();

        return $handler->setPayment($this);
    }

    /**
     * Restituisce il nome della classe `handler`.
     *
     * @return string|\Application\Showcase\Payment\BasePayment
     */
    public function getHandlerClass()
    {
        return $this->handler_class ?: SimplePaymentMethod::class;
    }

    /**
     * Restituisce un elenco delle opzioni salvate per l’handler.
     *
     * @return array
     */
    public function getHandlerOptions()
    {
        $options = unserialize($this->handler_options);
        $options = is_array($options) ? $options : [];

        $handler_class = $this->getHandlerClass();

        return array_merge($handler_class::getDefaultOptions(), $options);
    }

    /**
     * @param array $options
     * @return $this
     */
    public function setHandlerOptions(array $options = [])
    {
        $this->handler_options = serialize($options);

        return $this;
    }

    /**
     * @param string $key
     * @return string
     */
    public function getHandlerOption($key)
    {
        $options = $this->getHandlerOptions();

        if (array_key_exists($key, $options)) {
            return $options[$key];
        }

        return '';
    }

    /**
     * @param string $key
     * @param string $value
     * @return $this
     */
    public function setHandlerOption($key, $value)
    {
        $options = $this->getHandlerOptions();
        $options[$key] = $value;

        $this->handler_options = serialize($options);

        return $this;
    }

    /**
     * @param Order $order
     * @return float
     */
    protected function calculateCost(Order $order)
    {
        return $this->cost;
    }
}
