<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Model;

use ActiveRecord\Base as Model;
use Application\Showcase\Entity\OrderRowInterface;
use Application\Showcase\Utilities\Taxation;
use Psr\Log\LoggerInterface;

/**
 * Modello per le righe dell’ordine.
 *
 * @property int                               $id
 * @property int                               $order_id
 * @property int                               $product_id
 * @property int                               $size_id
 * @property int                               $row_id
 * @property int                               $quantity
 * @property float                             $price
 * @property float                             $offer
 * @property float                             $discount
 * @property string                            $discount_type
 * @property float                             $total
 * @property bool                              $is_previous_cart
 * @property bool                              $is_unavailable
 * @property int                               $shopping_points
 * @property \DateTime                         $created_at
 * @property \DateTime                         $updated_at
 * @property \Application\Showcase\Model\Node  $product
 * @property \Application\Showcase\Model\Order $order
 */
class OrderRow extends Model implements OrderRowInterface
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'orders_rows';

    /**
     * Relazioni 'belongs_to'.
     *
     * @var array
     */
    public static $belongs_to = [
        ['product', 'model' => Node::class],
        ['order', 'model' => Order::class],
    ];

    /**
     * Callback 'after_construct'.
     *
     * @var array
     */
    public static $after_construct = ['afterConstruct'];

    /**
     * Callback 'before_create'.
     *
     * @var array
     */
    public static $before_create = ['setRowId'];

    /**
     * Callback 'before_save'.
     *
     * @var array
     */
    public static $before_save = ['updateProductDetails', 'updateTotal'];

    /**
     * Taglia di riferimento per la riga.
     *
     * @var \Application\Showcase\Model\SizeName
     */
    protected $size;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @param LoggerInterface $logger
     * @return $this
     */
    public function setLogger(LoggerInterface $logger)
    {
        $this->logger = $logger;

        return $this;
    }

    /**
     * Controlla la presenza degli attributi `order_id` e `product_id`.
     *
     * @throws \InvalidArgumentException In mancanza di uno degli attributi `order_id` e `product_id`.
     */
    public function afterConstruct()
    {
        if ($this->isNewRecord() && (!$this->attributePresent('order_id') || !$this->attributePresent('product_id'))) {
            throw new \InvalidArgumentException(sprintf(
                'You can not create a %s without attributes `order_id` and `product_id`',
                get_class($this)
            ));
        }
    }

    /**
     * Cerca la riga per ordine, prodotto e taglia.
     *
     * @param integer $order_id
     * @param integer $product_id
     * @param integer $size_id
     * @return \Application\Showcase\Model\OrderRow
     */
    public static function findByOrderProductAndSize($order_id, $product_id, $size_id = null)
    {
        $conditions = [
            'order_id = :order AND product_id = :product',
            'order'   => $order_id,
            'product' => $product_id
        ];

        if ($size_id === null) {
            $conditions[0] .= ' AND size_id IS NULL';
        } else {
            $conditions[0] .= ' AND size_id = :size';
            $conditions['size'] = $size_id;
        }

        return self::first([
            'conditions' => $conditions,
        ]);
    }

    /**
     * {@inheritdoc}
     */
    public function getOrderId()
    {
        return $this->order_id;
    }

    /**
     * {@inheritdoc}
     */
    public function getProduct()
    {
        return $this->product;
    }

    /**
     * Permette di spostare la riga in un altro ordine.
     *
     * @param Order $order Il nuovo ordine.
     * @return bool Restituisce `true` se lo spostamento va a buon fine, altrimenti `false`.
     */
    public function changeOrder(Order $order)
    {
        // Controllo se il nuovo ordine contiene già il prodotto di questa riga
        foreach ($order->getRows() as $row) {
            if ($row->product_id === $this->product_id && $row->size_id === $this->size_id) {
                return false;
            }
        }

        $this->order_id = $order->id;
        $this->setRowId();

        return true;
    }

    /**
     * Imposta il `row_id` della riga.
     */
    public function setRowId()
    {
        $last_row_id = self::max('row_id', ['conditions' => ['order_id = ?', $this->order_id]]);

        $this->row_id = $last_row_id + 1;
    }

    /**
     * Aggiorna i dettagli del prodotto.
     */
    public function updateProductDetails()
    {
        if ($this->size_id === null) {
            $this->price = $this->product->price;
            $this->offer = $this->product->offer;
        } else {
            $this->price = $this->size()->price;
            $this->offer = $this->size()->offer;
        }

        $this->discount = $this->product->discount;
        $this->discount_type = $this->product->discount_type;
    }

    /**
     * Aggiorna il totale della riga.
     *
     * Il totale della riga è calcolato sul prezzo scontato.
     */
    public function updateTotal()
    {
        $this->total = $this->price() * $this->quantity;
    }

    /**
     * Aggiorna le disponibilità del prodotto.
     */
    public function updateBounds()
    {
        $product = $this->product;

        if ($product) {
            $product->updateAvailabilitiesAndBounds()->save();
        }

        if ($this->logger) {
            $this->logger->debug(
                'Bounds row has been updated.',
                [
                    'row' => $this->id,
                ]
            );
        }
    }

    /**
     * {@inheritdoc}
     */
    public function code()
    {
        return $this->product->code();
    }

    /**
     * {@inheritdoc}
     */
    public function sku()
    {
        $sku = $this->product_id;

        if ($this->size_id) {
            $sku .= '-' . $this->size_id;
        }

        return $sku;
    }

    /**
     * {@inheritdoc}
     */
    public function title($length = null, $continue = '…', $use_html_escape = true)
    {
        return $this->product->title($length, $continue, $use_html_escape);
    }

    /**
     * {@inheritdoc}
     */
    public function content()
    {
        return $this->product->content();
    }

    /**
     * {@inheritdoc}
     */
    public function excerpt($length = 20, $continue = '…')
    {
        return $this->product->excerpt($length, $continue);
    }

    /**
     * {@inheritdoc}
     */
    public function permalink()
    {
        return $this->product->permalink();
    }

    /**
     * Restituisce un campo personalizzato del prodotto.
     *
     * @see \Application\Cms\Model\Stuff::custom()
     *
     * @param string $custom_field
     * @param mixed  $default
     * @param string $type
     * @param string $format
     * @return string
     */
    public function custom($custom_field, $default = null, $type = 'string', $format = null)
    {
        return $this->product->custom($custom_field, $default, $type, $format);
    }

    /**
     * {@inheritdoc}
     */
    public function image()
    {
        return $this->product->image();
    }

    /**
     * {@inheritdoc}
     *
     * @todo Cache interna del valore.
     * @todo Disponibilità sul nodo.
     */
    public function availabilities()
    {
        if (!$this->product->getSiteModule()->getOption('enable_availabilities')) {
            return PHP_INT_MAX;
        }

        if ($this->size_id === null) {
            // @todo Disponibilità sul nodo.
            return PHP_INT_MAX;
        } else {
            if ($this->size() !== null) {
                return $this->size()->availabilities($this);
            }
        }

        return 0;
    }

    /**
     * {@inheritdoc}
     */
    public function size()
    {
        if ($this->size_id === null) {
            return null;
        }

        if ($this->size === null) {
            $this->size = Size::first(
                [
                    'select'     => '`from`.*, sn.name',
                    'joins'      => 'INNER JOIN ' . SizeName::tableName() . ' AS sn ON sn.id = `from`.size_name_id',
                    'conditions' => ['`from`.id = :size', 'size' => $this->size_id],
                ]
            );
        }

        return $this->size;
    }

    /**
     * {@inheritdoc}
     */
    public function sizename()
    {
        if ($this->size()) {
            return $this->size()->name();
        }

        return '';
    }

    /**
     * {@inheritdoc}
     */
    public function quantity()
    {
        return $this->quantity;
    }

    /**
     * Indica se il prodotto è scontato.
     *
     * @return bool
     */
    public function isDiscounted()
    {
        return (bool) $this->discount;
    }

    /**
     * Restituisce il valore formattato dello sconto.
     *
     * @return string
     */
    public function formatDiscount()
    {
        return $this->discount_type === '%' ? ($this->discount . '%') : format_price($this->discount * $this->quantity);
    }

    /**
     * Restituisce il prezzo reale del prodotto senza sconto.
     *
     * @param \Application\Showcase\Utilities\Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return float
     */
    public function realPrice(Taxation $taxation = null)
    {
        return $taxation === null ? $this->price : $taxation->calculate($this->price);
    }

    /**
     * Restituisce il prezzo reale formattato del prodotto senza sconto.
     *
     * @param \Application\Showcase\Utilities\Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return string
     */
    public function formatRealPrice(Taxation $taxation = null)
    {
        return format_price($this->realPrice($taxation));
    }

    /**
     * {@inheritdoc}
     */
    public function price(Taxation $taxation = null)
    {
        return $taxation === null ? $this->offer : $taxation->calculate($this->offer);
    }

    /**
     * {@inheritdoc}
     */
    public function formatPrice(Taxation $taxation = null)
    {
        return format_price($this->price($taxation));
    }

    /**
     * {@inheritdoc}
     */
    public function total(Taxation $taxation = null)
    {
        return $taxation === null ? $this->total : $taxation->calculate($this->total);
    }

    /**
     * {@inheritdoc}
     */
    public function formatTotal(Taxation $taxation = null)
    {
        return format_price($this->total($taxation));
    }

    /**
     * Resituisce il totale della riga senza sconto.
     *
     * @param \Application\Showcase\Utilities\Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return float
     */
    public function realTotal(Taxation $taxation = null)
    {
        $total = $this->quantity * $this->price;

        return $taxation === null ? $total : $taxation->calculate($total);
    }

    /**
     * Resituisce il totale formattato della riga senza sconto.
     *
     * @param \Application\Showcase\Utilities\Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return string
     */
    public function formatRealTotal(Taxation $taxation = null)
    {
        return format_price($this->realTotal($taxation));
    }

    /**
     * @param int $shopping_points
     * @return $this
     */
    public function setShoppingPoints($shopping_points)
    {
        $this->shopping_points = (int) $shopping_points;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getShoppingPoints()
    {
        return $this->quantity * $this->shopping_points;
    }
}
