<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Model;

use ActiveRecord\Base as Model;
use Application\Core\Model\Account;
use Application\Core\Model\Address;
use Application\Core\Model\Site;
use Application\Core\User;
use Application\Showcase\Entity\OrderInterface;
use Application\Showcase\Exception\CustomerUndefinedException;
use Application\Showcase\Exception\ShippingAddressUndefinedException;
use Application\Showcase\Exception\SizeNotAvailableException;
use Application\Showcase\ShoppingPoints\Exception\ShoppingPointsException;
use Application\Showcase\ShoppingPoints\Scheme;
use Application\Showcase\Utilities\CheckoutOptions;
use Application\Showcase\Utilities\Taxation;
use Pongho\Http\Request;
use Pongho\Http\Session;
use Pongho\Utilities\DateTime;
use Psr\Log\LoggerInterface;

/**
 * Modello per gli ordini.
 *
 * @property int                                    $id
 * @property int                                    $site_id
 * @property string                                 $session_id
 * @property int                                    $customer_id
 * @property string                                 $customer_ip
 * @property bool                                   $customer_is_guest
 * @property string                                 $customer_email
 * @property string                                 $status
 *
 * @property float                                  $products_total
 * @property int                                    $products_count
 * @property float                                  $discount_value TODO Potrebbe restare per cache interna?
 * @property float                                  $subtotal
 * @property int                                    $payment_id
 * @property float                                  $payment_cost
 * @property string                                 $payment_hash
 * @property string                                 $payment_details
 * @property int                                    $shipping_id
 * @property float                                  $shipping_cost
 * @property float                                  $taxable
 * @property float                                  $tax_cost
 * @property float                                  $total
 *
 * @property string                                 $invoice_name
 * @property string                                 $invoice_surname
 * @property string                                 $invoice_address1
 * @property string                                 $invoice_address2
 * @property string                                 $invoice_city
 * @property string                                 $invoice_postcode
 * @property string                                 $invoice_province_id
 * @property int                                    $invoice_country_id
 * @property string                                 $invoice_telephone
 * @property string                                 $invoice_address_text
 *
 * @property string                                 $invoice_code
 *
 * @property string                                 $customer_code_name
 * @property string                                 $customer_code_value
 *
 * @property int                                    $shipping_address_id
 * @property string                                 $shipping_name
 * @property string                                 $shipping_surname
 * @property string                                 $shipping_address1
 * @property string                                 $shipping_address2
 * @property string                                 $shipping_city
 * @property string                                 $shipping_postcode
 * @property string                                 $shipping_province_id
 * @property int                                    $shipping_country_id
 * @property string                                 $shipping_telephone
 * @property string                                 $shipping_address_text
 *
 * @property string                                 $note
 * @property string                                 $tracking_code
 * @property string                                 $process_note
 *
 * @property bool                                   $notify_sent
 * @property bool                                   $payment_fail_notify_sent
 *
 * @property \Pongho\Utilities\DateTime             $created_at
 * @property \Pongho\Utilities\DateTime             $updated_at
 * @property \Pongho\Utilities\DateTime             $ordered_at
 * @property \Pongho\Utilities\DateTime             $paid_at
 * @property \Pongho\Utilities\DateTime             $processed_at
 *
 * @property \Application\Core\Model\Site           $site
 * @property \Application\Core\Model\Account        $customer
 * @property \Application\Showcase\Model\Payment    $payment
 * @property \Application\Showcase\Model\Shipping   $shipping
 * @property \Application\Core\Model\Address        $invoice_address
 * @property \Application\Core\Model\Address        $shipping_address
 *
 * @property int                                       $shopping_points
 * @property int                                       $used_shopping_points
 */
class Order extends Model implements OrderInterface
{
    /**
     * Carrello.
     */
    const STATUS_CART = 'cart';

    /**
     * In attesa di pagamento (per bonifico e contrassegno).
     */
    const STATUS_PENDING = 'pending';

    /**
     * Annullato.
     */
    const STATUS_CANCELED = 'canceled';

    /**
     * Transazione in corso.
     *
     * @deprecated Mai usato in casi reali.
     */
    const STATUS_TRANSACTION = 'transaction';

    /**
     * Pagamento in corso.
     */
    const STATUS_PAYMENT = 'payment';

    /**
     * Pagamento in corso "bloccato".
     *
     * "Bloccato" significa che può tornare allo stato "CART" solo tramite la
     * procedura "paymentcancel" e NON tornando indietro con il browser.
     */
    const STATUS_PAYMENT_LOCKED = 'payment_locked';

    /**
     * Pagamento autorizzato.
     */
    const STATUS_PAYMENT_AUTHORIZED = 'payment_authorized';

    /**
     * Pagamento Fallito.
     */
    const STATUS_PAYMENT_FAILED = 'payment_failed';

    /**
     * Ordine pagato.
     */
    const STATUS_ORDER = 'order';

    /**
     * Stornato.
     */
    const STATUS_INVOICE_CANCELED = 'invoice_canceled';

    /**
     * Rimborsato.
     */
    const STATUS_REFUND = 'refund';

    /**
     * Rimborso annullato.
     */
    const STATUS_REFUND_CANCELED = 'refund_canceled';

    /**
     * Evaso.
     */
    const STATUS_PROCESSED = 'processed';

    /**
     * Spedito.
     */
    const STATUS_SHIPPED = 'shipped';

    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'orders';

    /**
     * Callback 'before_validation'.
     *
     * @var array
     */
    public static $before_validation = ['validateOrderFields'];

    /**
     * Callback 'before_save'.
     *
     * @var array
     */
    public static $before_save = ['updateTotal', 'checkStatus'];

    /**
     * Relazioni 'belongs_to'.
     *
     * @var array
     */
    public static $belongs_to = [
        ['site', 'model' => Site::class],
        ['customer', 'model' => Account::class],
        ['payment', 'model' => Payment::class],
        ['shipping', 'model' => Shipping::class],
        ['invoice_address', 'model' => Address::class],
        ['shipping_address', 'model' => Address::class],
    ];

    /**
     * @var CheckoutOptions
     */
    protected $checkoutOptions;

    /**
     * Righe dell’ordine.
     *
     * @var array
     */
    private $rows;

    /**
     * @var OrderOrderDiscount[]
     */
    private $discounts;

    /**
     * @var int
     */
    protected $available_customer_shopping_points;

    /**
     * Stati che definiscono l’ordine come carrello.
     *
     * @var array
     */
    public static $CART_STATUSES = [
        self::STATUS_CART,
        self::STATUS_PAYMENT,
        self::STATUS_TRANSACTION,
    ];

    /**
     * Stati che definiscono l’ordine in attesa per essere inteso come tale.
     *
     * @var array
     */
    public static $PAYMENT_STATUSES = [
        self::STATUS_PAYMENT,
        self::STATUS_PAYMENT_LOCKED,
        self::STATUS_TRANSACTION,
    ];

    /**
     * Stati che definiscono l’ordine come tale.
     *
     * @var array
     */
    public static $ORDER_STATUSES = [
        self::STATUS_ORDER,
        self::STATUS_PENDING,
        self::STATUS_PAYMENT_AUTHORIZED,
    ];

    /**
     * Stati che definiscono i prodotti in carrello come bloccati definitivamente.
     *
     * Sono esclusi gli stati dell’ordine evaso o annullato in quanto questi prevedono che la disponibilità sia già stata sottratta.
     *
     * @var array
     */
    public static $UNAVAILABLE_STATUSES = [
        self::STATUS_PAYMENT,
        self::STATUS_PAYMENT_LOCKED,
        self::STATUS_TRANSACTION,
        self::STATUS_ORDER,
        self::STATUS_PENDING,
        self::STATUS_PAYMENT_AUTHORIZED,
    ];

    /**
     * Stati che definiscono l’ordine come processato.
     *
     * @var array
     */
    public static $PROCESSED_STATUSES = [
        self::STATUS_PROCESSED,
        self::STATUS_SHIPPED,
        self::STATUS_REFUND_CANCELED,
    ];

    /**
     * Stati che indicano l’ordine finalizzato.
     *
     * Questo elenco viene usato per la validazione dei buoni sconto. Serve a capire se il buono è già stato
     * usato. Si escludono quindi gli stati di ordine annullato perché il buono può essere riutilizzato.
     *
     * @var array
     */
    public static $FINALIZED_STATUSES = [
        self::STATUS_PAYMENT,
        self::STATUS_PAYMENT_LOCKED,
        self::STATUS_TRANSACTION,
        self::STATUS_ORDER,
        self::STATUS_PENDING,
        self::STATUS_PAYMENT_AUTHORIZED,
        self::STATUS_PROCESSED,
        self::STATUS_SHIPPED,
    ];

    /**
     * Stati che definiscono l’ordine come annullati.
     *
     * @var array
     */
    public static $CANCELED_STATUSES = [
        self::STATUS_CANCELED,
        self::STATUS_PAYMENT_FAILED,
        self::STATUS_INVOICE_CANCELED,
        self::STATUS_REFUND,
    ];

    /**
     * Stati degli ordini da visualizzare nell’elenco ordini dell’utente.
     *
     * @var array
     */
    public static $USER_ORDERS_STATUSES = [
        self::STATUS_ORDER,
        self::STATUS_PENDING,
        self::STATUS_PAYMENT_AUTHORIZED,
        self::STATUS_PROCESSED,
        self::STATUS_SHIPPED,
        self::STATUS_CANCELED,
        self::STATUS_PAYMENT_FAILED,
        self::STATUS_INVOICE_CANCELED,
        self::STATUS_REFUND,
        self::STATUS_REFUND_CANCELED,
    ];

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @param LoggerInterface $logger
     * @return $this
     */
    public function setLogger(LoggerInterface $logger)
    {
        $this->logger = $logger;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * {@inheritdoc}
     */
    public function delete()
    {
        if ($this->delRows() && $this->removeAllDiscounts() && parent::delete()) {
            return true;
        }

        return false;
    }

    /**
     * Aggiorna il totale dell’ordine.
     *
     * Viene lanciato in automatico appena prima del salvataggio del modello grazie alla callback 'before_save'.
     */
    public function updateTotal()
    {
        if ($this->isOrder() || $this->isProcessed() || $this->isCanceled()) {
            return;
        }

        $products_total = 0;
        $products_count = 0;

        /** @var OrderRow $row */
        foreach ($this->getRows() as $row) {
            $products_total += $row->total;
            $products_count += $row->quantity;
        }

        $discountValue = 0;
        foreach ($this->getDiscounts() as $discount) {
            $discountValue += $discount->getValue();
        }

        $this->products_total = $products_total;
        $this->products_count = $products_count;

        $this->discount_value = $discountValue;
        $this->subtotal = $this->products_total - $discountValue;

        $this->payment_cost = $this->payment_id ? $this->payment->cost($this) : 0;
        $this->shipping_cost = $this->shipping_id ? $this->shipping->cost($this) : 0;

        $this->taxable = $this->subtotal + $this->payment_cost + $this->shipping_cost;

        if ($this->site->getOption('enable_tax')) {
            $taxation = new Taxation($this->site);

            $this->tax_cost = $taxation->get($this->taxable);
        } else {
            $this->tax_cost = 0;
        }

        $this->total = $this->taxable + $this->tax_cost;
    }

    /**
     * Esegue delle verifiche sullo stato.
     *
     * Questo metodo è lanciato automaticamente prima del salvataggio del modello.
     *
     * @throws \LogicException
     */
    public function checkStatus()
    {
        if ($this->status === self::STATUS_CART) {
            // Se l’ordine è un carrello svuoto i campi lasciati compilati da un eventuale pagamento fallito
            $this->payment_hash = '';
            $this->ordered_at = null;
            // $this->payment_details = ''; Da non fare assolutamente! Se deve essere svuotato quando si annulla un pagamento, lo deve fare l'handler.
        } else {
            if (!$this->customer_is_guest && $this->customer_id === Account::ANONYMOUS) {
                throw new \LogicException('Are you trying to make an order as an anonymous user?');
            }
        }
    }

    /**
     * @param CheckoutOptions $options
     * @return $this
     */
    public function setCheckoutOptions(CheckoutOptions $options)
    {
        $this->checkoutOptions = $options;

        return $this;
    }

    /**
     * Popola i campi relativi all’indirizzo di consegna.
     *
     * @return $this
     * @throws ShippingAddressUndefinedException
     * @throws CustomerUndefinedException
     */
    public function populateShippingAddressAttributes()
    {
        if ($this->customer_is_guest || !$this->checkoutOptions->isStandardShippingMethodsEnabled()) {
            return $this;
        }

        if ($this->shipping_address_id === null) {
            if ($this->customer_id === null) {
                throw new CustomerUndefinedException('Customer is not defined.');
            }

            if ($this->customer->shipping_address_id === null) {
                throw new ShippingAddressUndefinedException('Shipping address is not defined.');
            }

            $this->shipping_address_id = $this->customer->shipping_address_id;
        }

        $address = $this->shipping_address;

        $this->shipping_address_id = $address->id;
        $this->shipping_name = $address->name;
        $this->shipping_surname = $address->surname;
        $this->shipping_address1 = $address->address1;
        $this->shipping_address2 = $address->address2;
        $this->shipping_city = $address->city;
        $this->shipping_postcode = $address->postcode;
        $this->shipping_province_id = $address->province_id;
        $this->shipping_country_id = $address->country_id;
        $this->shipping_telephone = $address->telephone;
        $this->shipping_address_text = $address->render();

        return $this;
    }

    /**
     * Popola i campi relativi ai dati di fatturazione.
     *
     * @throws \Application\Showcase\Exception\ShippingAddressUndefinedException
     * @throws \Application\Showcase\Exception\CustomerUndefinedException
     * @return $this
     */
    public function populateInvoiceInfoAttributes()
    {
        if ($this->customer_is_guest || !$this->checkoutOptions->isStandardShippingMethodsEnabled()) {
            return $this;
        }

        if ($this->customer_id === null) {
            throw new CustomerUndefinedException('Customer is not defined.');
        }

        $customer = $this->customer;

        // Prendo il campo address come indicativo che l’indirizzo è stato compilato
        if ($customer->address === '') {
            if ($customer->shipping_address_id) {
                $address = $customer->shipping_address;
            } elseif ($this->shipping_address_id) {
                $address = $this->shipping_address;
            } else {
                throw new ShippingAddressUndefinedException('Shipping address is not defined.');
            }

            $this->invoice_name = $address->name;
            $this->invoice_surname = $address->surname;
            $this->invoice_address1 = $address->address1;
            $this->invoice_address2 = $address->address2;
            $this->invoice_city = $address->city;
            $this->invoice_postcode = $address->postcode;
            $this->invoice_province_id = $address->province_id;
            $this->invoice_country_id = $address->country_id;
            $this->invoice_telephone =  $address->telephone;

            $customer->name = $address->name;
            $customer->surname = $address->surname;
            $customer->address = $address->address1;
            $customer->address2 = $address->address2;
            $customer->city = $address->city;
            $customer->postcode = $address->postcode;
            $customer->province_id = $address->province_id;
            $customer->country_id = $address->country_id;
            $customer->telephone = $address->telephone;

            $this->invoice_address_text = $address->render();
        } else {
            $this->invoice_name = $customer->name;
            $this->invoice_surname = $customer->surname;
            $this->invoice_address1 = $customer->address;
            $this->invoice_address2 = $customer->address2;
            $this->invoice_city = $customer->city;
            $this->invoice_postcode = $customer->postcode;
            $this->invoice_province_id = $customer->province_id;
            $this->invoice_country_id = $customer->country_id;
            $this->invoice_telephone = $customer->telephone;

            $this->invoice_address_text = $customer->renderAddress();
        }

        // @todo Codice Fiscale (anche la tabella accounts potrebbe avere i campi customer_code_value|name)
        if ($customer->partita_iva) {
            $this->customer_code_name = 'partita_iva';
            $this->customer_code_value = $customer->partita_iva;
        } else {
            if ($customer->codice_fiscale) {
                $this->customer_code_name = 'codice_fiscale';
                $this->customer_code_value = $customer->codice_fiscale;
            }
        }

        return $this;
    }

    /**
     * Restituisce l’indirizzo di spedizione.
     *
     * @return string
     */
    public function getShippingAddress()
    {
        return $this->shipping_address_text;
    }

    /**
     * Restituisce l’indirizzo di fatturazione.
     *
     * @return string
     */
    public function getInvoiceAddress()
    {
        return $this->invoice_address_text;
    }

    /**
     * Restituisce l’indirizzo di fatturazione.
     *
     * @return string
     */
    public function getCustomerCodeName()
    {
        return $this->customer_code_name;
    }

    /**
     * Restituisce l’indirizzo di fatturazione.
     *
     * @return string
     */
    public function getCustomerCodeValue()
    {
        return $this->customer_code_value;
    }

    /**
     * Converte un carrello in ordine.
     *
     * Attenzione: il nuovo stato non viene impostato da questo metodo, in quanto lo devo fare il metodo di pagamento.
     *
     * @param string $status Lo stato da assegnare all’ordine. Può essere `payment`, `pending` o `order`.
     * @return bool
     * @throws \InvalidArgumentException
     */
    public function convertCartToOrder($status)
    {
        if (!in_array($status, self::$UNAVAILABLE_STATUSES)) {
            throw new \InvalidArgumentException(sprintf(
                'Status "%s" is not a valid status for a new order. Available status: "%s"',
                $status,
                implode('", "', self::$UNAVAILABLE_STATUSES)
            ));
        }

        if (!$this->customer_is_guest) {
            if ($this->customer_id === null) {
                return false;
            }

            $this->populateInvoiceInfoAttributes( /* todo */ );
            $this->populateShippingAddressAttributes();
        }

        $this->status = $status;
        $this->payment_hash = random(32);
        $this->ordered_at = new DateTime();

        // Aggiorno le disponibilità
        $this->updateBounds();

        return true;
    }

    /**
     * Annulla il pagamento in corso.
     *
     * @return $this
     */
    public function cancelPaymentTransaction()
    {
        if (!in_array($this->status, self::$PAYMENT_STATUSES)) {
            throw new \BadMethodCallException(
                "You can’t cancel the payment status for an order with status '{$this->status}'."
            );
        }

        $this->status = self::STATUS_CART;
        $this->payment_hash = '';
        $this->ordered_at = null;

        $this->updateBounds();

        return $this;
    }

    /**
     * Esegue la validazione dei campi come ordine (e non carrello).
     *
     * @return Order La classe stessa.
     */
    public function validateOrderFields()
    {
        if ($this->status === self::STATUS_CART) {
            return $this;
        }

        // Pagamento
        if ($this->payment_id && Payment::find($this->payment_id) === null) {
            $this->addError('payment_id', 'invalid_payment_id');
        }

        // Spedizione
        if ($this->shipping_id && Shipping::find($this->shipping_id) === null) {
            $this->addError('shipping_id', 'invalid_shipping_id');
        }

        return $this;
    }

    /**
     * Restituisce il carrello dell’utente.
     *
     * @param \Application\Core\User       $user
     * @param \Pongho\Http\Session         $session
     * @param \Application\Core\Model\Site $site
     * @return Order
     */
    public static function getCart(User $user, Session $session, Site $site)
    {
        $account = $user->getAccount();

        if ($user->isLogged()) {
            /** @var Order $cart */
            $cart = self::first([
                'conditions' => [
                    "site_id = :site AND status IN ('" . implode("', '", self::$CART_STATUSES) . "') AND ( session_id = :session OR customer_id = :customer )",
                    'site'     => $site->id,
                    'session'  => $session->getId(),
                    'customer' => $account->getId(),
                ],
                'order'      => 'created_at DESC',
            ]);

            // Se ho trovato il carrello sulla sessione, mi assicuro dell'associazione con l'utente.
            if ($cart) {
                $cart->customer_id = $account->getId();
                $cart->save(false);
            }
        } else {
            $cart = self::first([
                'conditions' => [
                    "site_id = :site AND status IN ('" . implode("', '", self::$CART_STATUSES) . "') AND session_id = :session AND customer_id = :customer",
                    'site'     => $site->id,
                    'session'  => $session->getId(),
                    'customer' => Account::ANONYMOUS,
                ],
                'order'      => 'created_at DESC',
            ]);
        }

        return $cart;
    }

    /**
     * Restituisce il carrello in base all’hash passato.
     *
     * @param string $hash
     * @return Order
     */
    public static function getOrderByHash($hash)
    {
        return static::first([
            'conditions' => ['payment_hash = :hash', 'hash' => md5($hash)],
        ]);
    }

    /**
     * Crea un nuovo carrello.
     *
     * @param \Application\Core\User       $user
     * @param \Pongho\Http\Session         $session
     * @param \Application\Core\Model\Site $site
     * @param \Pongho\Http\Request         $request
     * @return Order
     */
    public static function createCart(User $user, Session $session, Site $site, Request $request)
    {
        $customer = $user->getAccount();

        return new self([
            'site_id'        => $site->id,
            'session_id'     => $session->getId(),
            'customer_id'    => $customer->getId(),
            'customer_email' => $customer->getEmail(),
            'customer_ip'    => $request->getClientIp(),
            'status'         => 'cart',
        ]);
    }

    /**
     * Restituisce i carrelli caduti nel limbo.
     *
     * Si tratta di carrelli creati da utenti anonimi e poi abbandonati.
     *
     * @param \DateInterval $interval
     * @return array
     */
    public static function findLimboCarts(\DateInterval $interval)
    {
        $expired = new \DateTime();
        $expired->sub($interval);

        return self::all([
            'conditions' => [
                "customer_id = :anonymous AND status IN ('" . implode("', '", self::$CART_STATUSES) . "') AND created_at < :cart_expired_at",
                'anonymous'       => 1,
                'cart_expired_at' => $expired,
            ]
        ]);
    }

    /**
     * Restituisce i carrelli precedenti.
     *
     * @param \Application\Core\User       $user
     * @param \Pongho\Http\Session         $session
     * @param \Application\Core\Model\Site $site
     * @return Order[]
     */
    public static function findPreviousCarts(User $user, Session $session, Site $site)
    {
        if ($user->isLogged()) {
            $cart = self::getCart($user, $session, $site);

            return self::all([
                'conditions' => [
                    "site_id = :site AND status IN ('" . implode("', '", self::$CART_STATUSES) . "') AND ( session_id = :session OR customer_id = :customer ) AND id <> :cart_id",
                    'site'     => $site->id,
                    'session'  => $session->getId(),
                    'customer' => $user->getAccount()->getId(),
                    'cart_id'  => $cart->id
                ],
            ]);
        }

        return [];
    }

    /**
     * {@inheritdoc}
     *
     * @return OrderRow[]
     */
    public function getRows($force_update = false)
    {
        if ($this->isNewRecord()) {
            return [];
        }

        if ($this->rows === null || $force_update) {
            $this->rows = [];

            $options = [
                'conditions' => ['order_id = ?', $this->id],
                'order'      => 'row_id',
                'include'    => 'product',
            ];

            /** @var \Application\Showcase\Model\OrderRow $row */
            foreach (OrderRow::all($options) as $row) {
                if ($this->logger) {
                    $row->setLogger($this->logger);
                }

                $row->order = $this;
                $this->rows[$row->row_id] = $row;
            }
        }

        return $this->rows;
    }

    /**
     * {@inheritdoc}
     *
     * @todo   Disponibilità sul nodo
     */
    public function addProduct(Node $product, $quantity = 1, Size $size = null)
    {
        $size_id = $size ? $size->id : null;

        if ($this->logger) {
            $this->logger->info(
                '[CART] Adding a product to the cart.',
                [
                    'cart'     => $this->getId() ?: 'NULL',
                    'status'   => $this->status,
                    'product'  => $product->id,
                    'quantity' => $quantity,
                    'size'     => $size_id ?: 'NULL',
                ]
            );
        }

        if ($this->isNewRecord()) {
            $this->save();

            if ($this->logger) {
                $this->logger->debug(
                    '[CART] The cart has been created.',
                    [
                        'cart'     => $this->getId(),
                        'product'  => $product->id,
                        'size'     => $size_id ?: 'NULL',
                        'quantity' => $quantity,
                    ]
                );
            }
        }

        /** @var OrderRow $row */
        $row = OrderRow::findByOrderProductAndSize($this->id, $product->id, $size_id);

        if ($size) {
            // Disponibilità sulla taglia
            if ($product->getSiteModule()->getOption('enable_availabilities')) {
                // Controllo la disponibilità
                if ($size->availabilities() < $quantity) {
                    if ($this->logger) {
                        $this->logger->error(
                            '[CART] The quantity is higher than the size availabilities, throw SizeNotAvailableException.',
                            [
                                'cart'     => $this->getId(),
                                'product'  => $product->id,
                                'size'     => $size_id ?: 'NULL',
                                'quantity' => $quantity,
                                'row'      => $row ? $row->id : 'NULL',
                            ]
                        );
                    }

                    throw new SizeNotAvailableException();
                }
            }
        } else {
            // @todo Disponibilità sul nodo
        }

        if ($row) {
            if ($this->logger) {
                $this->logger->debug(
                    '[CART] The row of the cart already exists, then update the quantity.',
                    [
                        'cart'     => $this->getId(),
                        'product'  => $product->id,
                        'size'     => $size_id ?: 'NULL',
                        'quantity' => $quantity,
                        'row'      => $row ? $row->id : 'NULL',
                    ]
                );
            }

            // Aggiorno la quantità della riga esistente
            $row->quantity += $quantity;
            $row->shopping_points = $product->getShoppingPoints();

            if (!$row->save()) {
                return false;
            }
        } else {
            // Creo la nuova riga
            $attributes = [
                'order_id'        => $this->id,
                'product_id'      => $product->id,
                'size_id'         => $size_id,
                'quantity'        => $quantity,
                'shopping_points' => $product->getShoppingPoints(),
            ];

            if (!($row = OrderRow::create($attributes))) {
                if ($this->logger) {
                    $this->logger->critical(
                        '[CART] Could not create a new row.',
                        [
                            'cart'     => $this->getId(),
                            'product'  => $product->id,
                            'size'     => $size_id ?: 'NULL',
                            'quantity' => $quantity,
                            'row'      => $row ? $row->id : 'NULL',
                        ]
                    );
                }

                return false;
            }

            if ($this->logger) {
                $this->logger->debug(
                    '[CART] The row has been created.',
                    [
                        'cart'     => $this->getId(),
                        'product'  => $product->id,
                        'size'     => $size_id ?: 'NULL',
                        'quantity' => $quantity,
                        'row'      => $row ? $row->id : 'NULL',
                    ]
                );
            }
        }

        // Aggiungo la riga nell’elenco delle righe se questo è definito
        if ($this->rows !== null) {
            $this->rows[$row->row_id] = $row;
        }

        if (!$this->save()) {
            return false;
        }

        // Aggiorno le disponibilità.
        $product->updateAvailabilitiesAndBounds()->save();

        $this->discounts = null;

        return true;
    }

    /**
     * Aggiorna le quantità del carrello.
     *
     * @param array $quantities
     * @return bool
     *
     * @todo   Disponibilità sul nodo
     */
    public function changeQuantities(array $quantities)
    {
        if ($this->logger) {
            $this->logger->info(
                '[CART] Change the cart quantities.',
                [
                    'cart'       => $this->getId() ?: 'NULL',
                    'status'     => $this->status,
                    'quantities' => $quantities,
                ]
            );
        }

        if ($this->isNewRecord()) {
            if ($this->logger) {
                $this->logger->debug(
                    '[CART] The cart is a new record, then return true.',
                    [
                        'cart'       => 'NULL',
                        'quantities' => $quantities,
                    ]
                );
            }

            return true;
        }

        $rows = $this->getRows();

        if (empty($rows)) {
            if ($this->logger) {
                $this->logger->debug(
                    '[CART] The cart is empty, then return true.',
                    [
                        'cart'       => 'NULL',
                        'quantities' => $quantities,
                    ]
                );
            }

            return true;
        }

        foreach ($quantities as $row_id => $quantity) {
            $quantity = (int) $quantity;

            if (array_key_exists($row_id, $rows)) {
                /** @var OrderRow $row */
                $row = $rows[$row_id];

                if ($quantity === 0) {
                    if ($this->logger) {
                        $this->logger->info(
                            '[CART] The quantity is equals to zero, then I remove the row.',
                            [
                                'cart'    => $this->getId(),
                                'row'     => $row->id,
                                'product' => $row->product_id,
                                'size'    => $row->size_id ?: 'NULL',
                            ]
                        );
                    }

                    if (!$this->delRow($row)) {
                        if ($this->logger) {
                            $this->logger->critical(
                                '[CART] Row deletion failed.',
                                [
                                    'cart'    => $this->getId(),
                                    'row'     => $row->id,
                                    'product' => $row->product_id,
                                    'size'    => $row->size_id ?: 'NULL',
                                ]
                            );
                        }

                        return false;
                    }
                } else {
                    $product = $row->product;
                    $size = null;

                    if ($product->getSiteModule()->getOption('enable_availabilities')) {
                        if ($row->size_id) {
                            /**
                             * Disponibilità sulla taglia
                             *
                             * @var \Application\Showcase\Model\Size $size
                             */
                            $size = Size::find($row->size_id);
                            $availabilities = $size->availabilities($row);

                            if ($availabilities < $quantity) {
                                if ($this->logger) {
                                    $this->logger->notice(
                                        '[CART] The quantity is higher than the availabilities, then set the row as unavailable.',
                                        [
                                            'cart'           => $this->getId(),
                                            'row'            => $row->id,
                                            'product'        => $row->product_id,
                                            'size'           => $row->size_id,
                                            'quantity'       => $quantity,
                                            'availabilities' => $availabilities,
                                        ]
                                    );
                                }

                                $row->is_unavailable = true;
                            }
                        } else {
                            // @todo Disponibilità sul nodo
                        }
                    }

                    if ($this->logger) {
                        $this->logger->debug(
                            '[CART] Change the row quantity.',
                            [
                                'cart'         => $this->getId(),
                                'row'          => $row->id,
                                'product'      => $row->product_id,
                                'size'         => $row->size_id,
                                'old_quantity' => $row->quantity,
                                'new_quantity' => $quantity,
                            ]
                        );
                    }

                    $row->quantity = $quantity;

                    if (!$row->save()) {
                        if ($this->logger) {
                            $this->logger->critical(
                                '[CART] Could not save the row.',
                                [
                                    'cart'    => $this->getId(),
                                    'row'     => $row->id,
                                    'product' => $row->product_id,
                                    'size'    => $row->size_id,
                                ]
                            );
                        }

                        return false;
                    }

                    // Aggiorno le disponibilità.
                    $product->updateAvailabilitiesAndBounds()->save();
                }
            }
        }

        $this->discounts = null;

        // salvo l’ordine per aggiornare i totali
        return $this->save();
    }

    /**
     * Elimina una riga dal carrello.
     *
     * @param mixed $row_or_row_id
     * @return bool
     * @throws \InvalidArgumentException
     */
    public function delRow($row_or_row_id)
    {
        if ($this->logger) {
            $this->logger->info(
                '[CART] Deleting row.',
                [
                    'cart'   => $this->getId(),
                    'row'    => $row_or_row_id instanceof OrderRow ? $row_or_row_id->id : ($row_or_row_id ?: 'NULL'),
                    'status' => $this->status,
                ]
            );
        }

        if ($this->isNewRecord()) {
            if ($this->logger) {
                $this->logger->debug(
                    '[CART] The cart is a new record, then return true.',
                    [
                        'cart' => $this->getId(),
                        'row'  => 'NULL', // se non c'è un carrello, non c'è nemmeno la riga
                    ]
                );
            }

            return true;
        }

        if ($row_or_row_id instanceof OrderRow) {
            $row = $row_or_row_id;
        } else if (is_numeric($row_or_row_id)) {
            $row = OrderRow::first([
                'conditions' => ['order_id = ? AND row_id = ?', $this->id, $row_or_row_id]
            ]);
        } else {
            if ($this->logger) {
                $this->logger->critical(
                    '[CART] Argument one must be an integer or instance of "Application\\Showcase\\Model\\OrderRow"!',
                    [
                        'cart' => $this->getId(),
                        'row'  => $row_or_row_id,
                    ]
                );
            }

            throw new \InvalidArgumentException('Argument one must be an integer or instance of <strong>Application\\Showcase\\Model\\OrderRow</strong>!');
        }

        if ($row === null) {
            if ($this->logger) {
                $this->logger->debug(
                    '[CART] The row not exists, then return true.',
                    [
                        'cart' => $this->getId(),
                        'row'  => 'NULL',
                    ]
                );
            }

            return true;
        }

        if (!$row->delete()) {
            if ($this->logger) {
                $this->logger->critical(
                    '[CART] Could not delete the row.',
                    [
                        'cart'    => $this->getId(),
                        'row'     => $row->id,
                        'product' => $row->product_id,
                        'size'    => $row->size_id,
                    ]
                );
            }

            return false;
        }

        if ($this->rows !== null) {
            unset($this->rows[$row->row_id]);
        }

        if (!$this->save()) {
            return false;
        }

        // Aggiorno le disponibilità.
        $row->updateBounds();

        if ($this->logger) {
            $this->logger->debug(
                '[CART] The row has been deleted.',
                [
                    'cart'    => $this->getId(),
                    'row'     => $row->id,
                    'product' => $row->product_id,
                    'size'    => $row->size_id,
                ]
            );
        }

        $this->discounts = null;

        return true;
    }

    /**
     * Svuota il carrello.
     *
     * @return bool
     */
    public function delRows()
    {
        if ($this->logger) {
            $this->logger->info(
                '[CART] Deleting all rows in the cart.',
                [
                    'cart' => $this->getId() ?: 'NULL',
                    'status' => $this->status,
                ]
            );
        }

        if ($this->isNewRecord()) {
            $this->rows = [];

            if ($this->logger) {
                $this->logger->debug(
                    '[CART] The cart is a new record, then return true.',
                    [
                        'cart' => 'NULL',
                    ]
                );
            }

            return true;
        }

        $options = [
            'conditions' => ['order_id = ?', $this->id],
        ];

        /** @var OrderRow $row */
        foreach (OrderRow::all($options) as $row) {
            if (!$this->delRow($row)) {
                if ($this->logger) {
                    $this->logger->critical(
                        '[CART] Could not delete the row.',
                        [
                            'cart' => $this->getId(),
                            'row'  => $row->id,
                        ]
                    );
                }

                return false;
            }
        }

        if ($this->logger) {
            $this->logger->debug(
                '[CART] All rows have been deleted.',
                [
                    'cart' => $this->getId() ?: 'NULL',
                ]
            );
        }

        $this->discounts = null;

        return $this->save();
    }

    /**
     * Aggiorna le disponibilità del prodotto.
     */
    public function updateBounds()
    {
        if ($this->logger) {
            $this->logger->info(
                'Updating rows bounds.',
                [
                    'cart' => $this->getId() ?: 'NULL',
                ]
            );
        }

        /** @var OrderRow $row */
        foreach ($this->getRows() as $row) {
            $row->updateBounds();
        }
    }

    /**
     * Restituisce gli sconti applicati all’ordine.
     *
     * @return OrderOrderDiscount[]
     */
    public function getDiscounts()
    {
        $this->prepareDiscounts();

        return $this->discounts;
    }

    /**
     * Indica se l’ordine ha degli sconti applicati.
     *
     * @return bool
     */
    public function hasDiscounts()
    {
        $this->prepareDiscounts();

        return !empty($this->discounts);
    }

    /**
     * Applica uno sconto all’ordine.
     *
     * Questo metodo non verifica se esiste già uno sconto dello stesso tipo prima di applicarlo.
     *
     * @see Order::updateOrApplyDiscount()
     *
     * @param OrderDiscount $discount
     * @param string        $name
     * @param float         $value
     * @return $this
     */
    public function applyDiscount(OrderDiscount $discount, $name, $value)
    {
        OrderOrderDiscount::create([
            'order_id'    => $this->getId(),
            'discount_id' => $discount->id,
            'name'        => $name,
            'value'       => $value,
            'priority'    => $this->getDiscountPriority($discount),
        ]);

        // Forzo l’aggiornamento degli sconti
        $this->discounts = null;

        return $this;
    }

    /**
     * Aggiorna uno sconto, oppure lo applica se non è ancora applicato.
     *
     * @param OrderDiscount $discount
     * @param string        $name
     * @param float         $value
     * @return $this
     */
    public function updateOrApplyDiscount(OrderDiscount $discount, $name, $value)
    {
        /** @var OrderOrderDiscount $rel */
        $rel = OrderOrderDiscount::first([
            'conditions' => [
                'order_id = :order AND discount_id = :discount',
                'order'    => $this->getId(),
                'discount' => $discount->id,
            ]
        ]);

        if ($rel) {
            $rel->name = $name;
            $rel->value = $value;
            $rel->priority = $this->getDiscountPriority($discount);
            $rel->save();
        } else {
            $this->applyDiscount($discount, $name, $value);
        }

        // Forzo l’aggiornamento degli sconti
        $this->discounts = null;

        return $this;
    }

    /**
     * Rimuove uno sconto.
     *
     * @param OrderDiscount $discount
     * @return $this
     */
    public function removeDiscount(OrderDiscount $discount)
    {
        /** @var OrderOrderDiscount $rel */
        $rel = OrderOrderDiscount::first([
            'conditions' => [
                'order_id = :order AND discount_id = :discount',
                'order'    => $this->getId(),
                'discount' => $discount->id,
            ]
        ]);

        if ($rel) {
            $rel->delete();
        }

        // Forzo l’aggiornamento degli sconti
        $this->discounts = null;

        return $this;
    }

    /**
     * Rimuove tutti gli sconti applicati.
     *
     * @return $this
     */
    public function removeAllDiscounts()
    {
        $options = [
            'conditions' => [
                'order_id = :order',
                'order' => $this->getId(),
            ]
        ];

        foreach (OrderOrderDiscount::all($options) as $rel) {
            $rel->delete();
        }

        // Forzo l’aggiornamento degli sconti
        $this->discounts = null;

        return $this;
    }

    /**
     * @param OrderDiscount $discount
     * @return int
     */
    private function getDiscountPriority(OrderDiscount $discount)
    {
        $discountsOption = $this->site->getOption('order_discounts');

        foreach ($discountsOption as $index => $discountHandlerName) {
            if ($discountHandlerName === $discount->getHandlerName()) {
                return $index + 1;
            }
        }

        throw new \InvalidArgumentException('Discount "' . $discount->getHandlerName() . '" does not exist.');
    }

    /**
     * Prepara l’elenco interno degli sconti per i metodi che ne fanno uso.
     *
     * @internal
     */
    private function prepareDiscounts()
    {
        if ($this->discounts !== null) {
            return;
        }

        $this->discounts = OrderOrderDiscount::all([
            'conditions' => ['order_id = :order', 'order' => $this->id],
            'order'      => 'priority ASC',
            'include'    => ['discount'],
        ]);
    }

    /**
     * Restituisce il valore del buono sconto applicato all’ordine.
     *
     * @return string
     */
    public function couponCode()
    {
        foreach ($this->getDiscounts() as $orderDiscount) {
            if ($orderDiscount->getDiscount()->getHandlerName() === 'coupon_code') {
                return $orderDiscount->getDiscount()->getCode();
            }
        }

        return '';
    }

    /**
     * {@inheritdoc}
     */
    public function getGainedCustomerShoppingPoints()
    {
        if ($this->customerIsGuest()) {
            return 0;
        }

        if ($this->used_shopping_points) {
            return $this->getAvailableCustomerShoppingPoints();
        }

        return $this->getAvailableCustomerShoppingPoints() + $this->shopping_points;
    }

    /**
     * {@inheritdoc}
     */
    public function getAvailableCustomerShoppingPoints()
    {
        if ($this->customerIsGuest()) {
            return 0;
        }

        if ($this->available_customer_shopping_points === null) {
            $statuses = "'" . implode("', '", self::$ORDER_STATUSES) . "'";

            $sql = <<<SQL
SELECT SUM(used_shopping_points) AS n
FROM pongho_orders
WHERE status IN ($statuses)
  AND customer_id = :customer
GROUP BY customer_id
SQL;

            $points = Order::connection()
                ->query($sql, ['customer' => $this->customer_id])
                ->fetchColumn();

            $points = (int) $points;

            $this->available_customer_shopping_points = $this->customer->shopping_points - $points;
        }

        return $this->available_customer_shopping_points;
    }

    /**
     * {@inheritdoc}
     */
    public function setUsedShoppingPoints($used_points, Scheme $scheme)
    {
        if ($used_points && !$scheme->arePointsApplicableToOrder($this, $used_points)) {
            throw new ShoppingPointsException();
        }

        $this->used_shopping_points = $used_points;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function updateEarnedShoppingPoints(Scheme $scheme)
    {
        $this->shopping_points = $scheme->calculateFromOrder($this);

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getShoppingPoints()
    {
        return $this->shopping_points;
    }

    /**
     * {@inheritdoc}
     */
    public function productsTotal(Taxation $taxation = null)
    {
        return $taxation === null ? $this->products_total : $taxation->calculate($this->products_total);
    }

    /**
     * {@inheritdoc}
     */
    public function formatProductsTotal(Taxation $taxation = null)
    {
        return format_price($this->productsTotal($taxation));
    }

    /**
     * {@inheritdoc}
     */
    public function subtotal(Taxation $taxation = null)
    {
        return $taxation === null ? $this->subtotal : $taxation->calculate($this->subtotal);
    }

    /**
     * {@inheritdoc}
     */
    public function formatSubtotal(Taxation $taxation = null)
    {
        return format_price($this->subtotal($taxation));
    }

    /**
     * {@inheritdoc}
     */
    public function paymentCost(Taxation $taxation = null)
    {
        return $taxation === null ? $this->payment_cost : $taxation->calculate($this->payment_cost);
    }

    /**
     * {@inheritdoc}
     */
    public function formatPaymentCost(Taxation $taxation = null)
    {
        return format_price($this->paymentCost($taxation));
    }

    /**
     * {@inheritdoc}
     */
    public function shippingCost(Taxation $taxation = null)
    {
        return $taxation === null ? $this->shipping_cost : $taxation->calculate($this->shipping_cost);
    }

    /**
     * {@inheritdoc}
     */
    public function formatShippingCost(Taxation $taxation = null)
    {
        return format_price($this->shippingCost($taxation));
    }

    /**
     * {@inheritdoc}
     */
    public function taxable()
    {
        return $this->taxable;
    }

    /**
     * {@inheritdoc}
     */
    public function formatTaxable()
    {
        return format_price($this->taxable());
    }

    /**
     * {@inheritdoc}
     */
    public function taxCost()
    {
        return $this->tax_cost;
    }

    /**
     * {@inheritdoc}
     */
    public function formatTaxCost()
    {
        return format_price($this->taxCost());
    }

    /**
     * {@inheritdoc}
     */
    public function total()
    {
        return $this->total;
    }

    /**
     * {@inheritdoc}
     */
    public function formatTotal()
    {
        return format_price($this->total());
    }

    /**
     * Indica se l’ordine è vuoto, nel senso che non ha prodotti inseriti.
     *
     * Questo metodo è pensato per il template.
     *
     * @return bool
     */
    public function isEmpty()
    {
        $rows = $this->getRows();

        return empty($rows);
    }

    /**
     * Restituisce la lista dei prodotti in carrello / ordine.
     *
     * Questo metodo è pensato per il template.
     *
     * @return \Application\Showcase\Model\OrderRow[]
     */
    public function rows()
    {
        return $this->getRows();
    }

    /**
     * {@inheritdoc}
     */
    public function quantities()
    {
        return (int)$this->products_count;
    }

    /**
     * Restituisce le note dell’ordine.
     *
     * @return string
     */
    public function note()
    {
        return nl2br(html_escape($this->note));
    }

    /**
     * Imposta un metodo di pagamento.
     *
     * @param Payment $payment
     * @return $this
     * @throws \BadMethodCallException
     */
    public function setPayment(Payment $payment)
    {
        if (!$this->isCart()) {
            throw new \BadMethodCallException('You can not set the payment method to a confirmed order.');
        }

        $this->payment = $payment;
        $this->payment_id = $payment->id;

        $this->payment_fail_notify_sent = false;

        if ($this->logger) {
            $this->logger->debug(
                'The payment method has been applied.',
                [
                    'cart'    => $this->getId(),
                    'payment' => '[' . $payment->id . '] ' . $payment->name,
                ]
            );
        }

        return $this;
    }

    /**
     * Imposta un metodo di spedizione.
     *
     * @param Shipping $shipping
     * @return $this
     * @throws \BadMethodCallException
     */
    public function setShipping(Shipping $shipping)
    {
        if (!$this->isCart()) {
            throw new \BadMethodCallException('You can not set the shipping method to a confirmed order.');
        }

        $this->shipping = $shipping;
        $this->shipping_id = $shipping->id;

        if ($this->logger) {
            $this->logger->debug(
                'The shipping method has been applied.',
                [
                    'cart'     => $this->getId(),
                    'shipping' => '[' . $shipping->id . '] ' . $shipping->name,
                ]
            );
        }

        return $this;
    }

    /**
     * Restituisce il gestore del pagamento.
     *
     * @return \Application\Showcase\Payment\BasePayment
     * @throws \RuntimeException Se il pagamento non è stato definito.
     */
    public function getPaymentHandler()
    {
        if ($this->payment) {
            $class = $this->payment->handler_class;

            return new $class($this);
        }

        throw new \RuntimeException('Payment is not defined!');
    }

    /**
     * setPaymentFailNotified
     *
     * @return $this
     */
    public function setPaymentFailNotified()
    {
        $this->payment_fail_notify_sent = true;

        return $this;
    }

    /**
     * isPaymentFailNotified
     *
     * @return bool
     */
    public function isPaymentFailNotified()
    {
        return $this->payment_fail_notify_sent;
    }

    /**
     * Indica se l’ordine è ancora allo stato di carrello.
     *
     * @return bool
     */
    public function isCart()
    {
        return in_array($this->status, self::$CART_STATUSES);
    }

    /**
     * Indica se l’ordine è in fase di pagamento.
     *
     * @return bool
     */
    public function isPayment()
    {
        return in_array($this->status, self::$PAYMENT_STATUSES);
    }

    /**
     * Indica se l’ordine è stato pagato.
     *
     * @return bool
     */
    public function isPaid()
    {
        if ($this->paid_at instanceof DateTime) {
            return !$this->paid_at->isNull();
        }

        return $this->paid_at !== null;
    }

    /**
     * Indica se è un ordine.
     *
     * @return bool
     */
    public function isOrder()
    {
        return in_array($this->status, self::$ORDER_STATUSES);
    }

    /**
     * Indica se è un ordine processato.
     *
     * @return bool
     */
    public function isProcessed()
    {
        if ($this->processed_at instanceof DateTime) {
            return !$this->processed_at->isNull();
        }

        return $this->processed_at !== null && in_array($this->status, self::$PROCESSED_STATUSES);
    }

    /**
     * Indica se è un ordine annullato.
     *
     * @return bool
     */
    public function isCanceled()
    {
        return in_array($this->status, self::$CANCELED_STATUSES);
    }

    /**
     * {@inheritdoc}
     */
    public function customerIsGuest()
    {
        return $this->customer_id === Account::ANONYMOUS && $this->customer_is_guest;
    }

    /**
     * {@inheritdoc}
     */
    public function customerIsLogged()
    {
        return $this->customer_id !== Account::ANONYMOUS;
    }

    /**
     * {@inheritdoc}
     */
    public function getCustomerEmail()
    {
        if ($this->customer_is_guest) {
            return $this->customer_email;
        }

        return $this->customer->email;
    }

    /**
     * {@inheritdoc}
     */
    public function getCustomerName()
    {
        if ($this->customer_is_guest) {
            return trim($this->invoice_name . ' ' . $this->invoice_surname);
        }

        return $this->customer->name();
    }

    /**
     * {@inheritdoc}
     */
    public function getCustomerFirstName()
    {
        if ($this->customer_is_guest) {
            return $this->invoice_name;
        }

        return $this->customer->getName();
    }

    /**
     * {@inheritdoc}
     */
    public function getCustomerLastName()
    {
        if ($this->customer_is_guest) {
            return $this->invoice_surname;
        }

        return $this->customer->getSurname();
    }

    /**
     * @return string
     */
    public function getTrackingCode()
    {
        return $this->tracking_code;
    }

    /**
     * @param string $tracking_code
     * @return $this
     */
    public function setTrackingCode($tracking_code)
    {
        $this->tracking_code = trim($tracking_code);

        return $this;
    }

    /**
     * @return DateTime|null
     */
    public function getProcessedAt()
    {
        return $this->processed_at;
    }

    /**
     * @param DateTime $processedAt
     * @return $this
     */
    public function setProcessed(DateTime $processedAt)
    {
        $this->processed_at = $processedAt;
        $this->status = self::STATUS_PROCESSED;

        if ($this->attributePresent('paid_at') && ($this->paid_at === null || $this->paid_at->isNull())) {
            $this->paid_at = $processedAt;
        }

        return $this;
    }

    /**
     * @param string $process_note
     * @return $this
     */
    public function setProcessNote($process_note)
    {
        $this->process_note = trim($process_note);

        return $this;
    }

    /**
     * @return bool
     */
    public function hasProcessNote()
    {
        return $this->process_note !== '';
    }

    /**
     * @return string
     */
    public function getProcessNote()
    {
        return nl2br($this->process_note);
    }
}
