<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Model;

use Application\Cms\Model\Node as BaseNode;
use Application\Showcase\Model\Helper\ProductUrlGenerator;
use Application\Showcase\Model\Helper\ProductUrlGeneratorInterface;
use Application\Showcase\Utilities\Taxation;
use Pongho\Core\Kernel;
use Pongho\Utilities\DateTime;

/**
 * Modello per i prodotti.
 *
 * @property int                        $size_type_id
 * @property int                        $availabilities
 * @property int                        $bound_in_cart
 * @property \Pongho\Utilities\DateTime $bound_in_cart_updated_at
 * @property int                        $bound_in_payment
 * @property \Pongho\Utilities\DateTime $bound_in_payment_updated_at
 * @property int                        $bound_in_order
 * @property string                     $gs_color
 * @property string                     $gs_gender
 * @property string                     $gs_age_group
 * @property string                     $gs_material
 * @property string                     $gs_pattern
 * @property int                        $shopping_points
 */
class Node extends BaseNode
{
    const BOUND_IN_CART_EXPIRY = '1 hour ago';
    const BOUND_IN_PAYMENT_EXPIRY = '2 hours ago';

    /**
     * Cache interna - Elenco delle taglie legate al nodo.
     *
     * @var array
     */
    protected $sizes;

    /**
     * Cache interna - Modello del tipo di taglia legato al nodo.
     *
     * @var \Application\Showcase\Model\SizeType
     */
    protected $size_type;

    /**
     * @var ProductUrlGeneratorInterface
     */
    private $urlGenerator;

    /**
     * {@inheritdoc}
     */
    public function delete($in_recursion = false)
    {
        if ($this->status === 'home' || !$this->isDeletable()) {
            return true;
        }

        if ($this->deleteSizes() && parent::delete()) {
            return true;
        }

        return false;
    }

    /**
     * {@inheritdoc}
     */
    public function isDeletable()
    {
        if (!parent::isDeletable()) {
            return false;
        }

        return (bool) (OrderRow::count(['conditions' => ['product_id = ?', $this->id]]) === 0);
    }

    /**
     * Converte un nodo normale in nodo showcase.
     *
     * @param \Application\Cms\Model\Node $node
     */
    public function cast(BaseNode $node)
    {
        $this->updateAttributes($node->attributes());
        $this->new_record = $node->isNewRecord();
    }

    /**
     * {@inheritdoc}
     */
    protected function getSiteModuleClass()
    {
        return 'Application\\Showcase\\Model\\NodeType';
    }

    /**
     * {@inheritdoc}
     */
    public function save($perform_validation = true)
    {
        /** @var \Application\Showcase\Discount\DiscountCalculatorInterface $discountCalculator */
        $discountCalculator = Kernel::instance()->getContainer()->get('shop_discount_calculator');

        $this->offer = $discountCalculator->calculate(
            $this->price,
            $this->discount,
            $this->discount_type
        );

        return parent::save($perform_validation);
    }

    /**
     * @param ProductUrlGeneratorInterface $generator
     * @return $this
     */
    public function setUrlGenerator(ProductUrlGeneratorInterface $generator)
    {
        $this->urlGenerator = $generator;

        return $this;
    }

    /**
     * @return ProductUrlGeneratorInterface
     */
    private function getUrlGenerator()
    {
        if ($this->urlGenerator === null) {
            $this->urlGenerator = new ProductUrlGenerator();
        }

        return $this->urlGenerator;
    }

    /**
     * {@inheritdoc}
     */
    public function permalink()
    {
        return $this->getUrlGenerator()->generatePermalink($this);
    }

    /**
     * Restituisce il collegamento per aggiungere il prodotto al carrello.
     *
     * @return string
     */
    public function addtocart()
    {
        return $this->getUrlGenerator()->generateAddToCartUrl($this);
    }

    /**
     * Restituisce il prezzo del prodotto.
     *
     * @param \Application\Showcase\Utilities\Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return float
     */
    public function price(Taxation $taxation = null)
    {
        return $taxation === null ? $this->offer : $taxation->calculate($this->offer);
    }

    /**
     * Restituisce il prezzo formattato del prodotto.
     *
     * @param \Application\Showcase\Utilities\Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return string
     */
    public function formatPrice(Taxation $taxation = null)
    {
        return format_price($this->price($taxation));
    }

    /**
     * Indica se il prodotto è scontato.
     *
     * @return bool
     */
    public function isDiscounted()
    {
        return (bool) $this->discount;
    }

    /**
     * Restituisce il valore formattato dello sconto.
     *
     * @return string
     */
    public function formatDiscount()
    {
        return $this->discount_type === '%' ? ($this->discount . '%') : format_price($this->discount);
    }

    /**
     * Restituisce il prezzo reale del prodotto senza sconto.
     *
     * @param \Application\Showcase\Utilities\Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return float
     */
    public function realPrice(Taxation $taxation = null)
    {
        return $taxation === null ? $this->price : $taxation->calculate($this->price);
    }

    /**
     * Restituisce il prezzo reale formattato del prodotto senza sconto.
     *
     * @param \Application\Showcase\Utilities\Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return string
     */
    public function formatRealPrice(Taxation $taxation = null)
    {
        return format_price($this->realPrice($taxation));
    }

    /**
     * Restituisce il nodo in base al codice specificato.
     *
     * @param string $code
     * @return Node
     */
    public static function findByCode($code)
    {
        return self::first(['conditions' => ['code = ?', $code]]);
    }

    /**
     * Restituisce il codice del prodotto.
     *
     * @return string
     */
    public function code()
    {
        return $this->code;
    }

    /**
     * @return bool
     */
    public function hasAvailabilities()
    {
        if (!$this->getSiteModule()->getOption('enable_availabilities')) {
            return true;
        }

        if ($this->getSiteModule()->getOption('enable_sizes')) {
            $sizes = $this->getSizes(true);

            if ($sizes) {
                foreach ($sizes as $size) {
                    if ($size->availabilities()) {
                        return true;
                    }
                }

                return false;
            }
        }

        // @todo Disponibilità sul nodo

        return true;
    }

    /**
     * Indica se è possibile acquistare il prodotto.
     *
     * Al momento questo metodo verifica la semplice disponibilità, ma in futuro potrbbero essere implementati
     * ulteriori controlli, come il permesso sull'utente o altre variabili.
     *
     * @return bool
     */
    public function isPurchasable()
    {
        return $this->hasAvailabilities();
    }

    /**
     * Alias di getSizes().
     *
     * @see Node::getSizes()
     *
     * @param bool $all
     * @return \Application\Showcase\Model\Size[]
     */
    public function sizes($all = false)
    {
        return $this->getSizes($all);
    }

    /**
     * Restituisce la select per le taglie.
     *
     * @param bool $all
     * @return array
     */
    public function sizesSelectOptions($all = false)
    {
        $options = [];

        $availabilities_enabled = $this->getSiteModule()->getOption('enable_availabilities');

        foreach ($this->getSizes($all) as $size) {
            $attributes = [
                $size->name(),
                'data-price' => $size->price(),
                'data-real-price' => $size->realPrice(),
            ];

            if ($availabilities_enabled) {
                $attributes['data-availabilities'] = $size->availabilities();
            }

            $options[$size->id] = $attributes;

            if ($availabilities_enabled && $size->availabilities() <= 0) {
                $options[$size->id]['disabled'] = 'disabled';
            }
        }

        return $options;
    }

    /**
     * Restituisce l’elenco delle taglie disponibili per questo nodo.
     *
     * @param bool $all Se impostato a `true` restituisce tutte le taglie, altrimenti solo quelle disponibili.
     *                  Questo argomento è ignorato se le disponiblità non sono abilitate.
     * @return \Application\Showcase\Model\Size[]
     */
    public function getSizes($all = false)
    {
        // Stabilizzo la variabile $all
        $all = ($all || !$this->getSiteModule()->getOption('enable_availabilities')) ? 1 : 0;

        if ($this->sizes === null) {
            if ($this->isNewRecord() && $this->size_type_id === null) {
                return [];
            }

            $options = [
                'select' => '`from`.*, sn.name, sn.id AS size_name_id, (CASE WHEN `from`.bound_in_cart_updated_at > :cart_expiry THEN `from`.bound_in_cart ELSE 0 END + CASE WHEN `from`.bound_in_payment_updated_at > :payment_expiry THEN `from`.bound_in_payment ELSE 0 END + `from`.bound_in_order) AS bound',
                'order'  => 'sn.position',
            ];

            if ($this->size_type_id === null) {
                $options['joins'] = 'RIGHT JOIN ' . SizeName::tableName() . ' AS sn ON sn.id = `from`.size_name_id';

                $options['conditions'] = [
                    'sn.size_type_id IS NULL AND `from`.node_id = :node',
                    'node'           => $this->id,
                    'cart_expiry'    => new DateTime(Node::BOUND_IN_CART_EXPIRY),
                    'payment_expiry' => new DateTime(Node::BOUND_IN_PAYMENT_EXPIRY),
                ];

                $all_sizes = Size::all($options);
            } else {
                $options['joins'] = 'RIGHT JOIN ' . SizeName::tableName() . ' AS sn ON sn.id = `from`.size_name_id AND `from`.node_id = :node';

                $options['conditions'] = [
                    'sn.size_type_id = :size_type',
                    'node'           => $this->id,
                    'size_type'      => $this->size_type_id,
                    'cart_expiry'    => new DateTime(Node::BOUND_IN_CART_EXPIRY),
                    'payment_expiry' => new DateTime(Node::BOUND_IN_PAYMENT_EXPIRY),
                ];

                $all_sizes = Size::all($options);

                // Se non trovo taglie associate a questo nodo significa che non sono ancora state create,
                // quindi le aggiungo impostando i valori vuoti.
                if (empty($all_sizes)) {
                    $options = [
                        'conditions' => ['size_type_id = ?', $this->size_type_id],
                        'order'      => 'position ASC',
                    ];

                    /** @var SizeName $size_name */
                    foreach (SizeName::all($options) as $size_name) {
                        // Valori come "price", "offer" e "availabilities" sono già di default a zero, quindi evito di riportarli.
                        $all_sizes[] = new Size(
                            [
                                'node_id'      => $this->id,
                                'size_name_id' => $size_name->id,
                                'name'         => $size_name->name,
                                'bound'        => 0,
                            ]
                        );
                    }
                }
            }

            $this->sizes = [
                0 => [],
                1 => $all_sizes,
            ];

            /** @var Size $size */
            foreach ($this->sizes[1] as $size) {
                if ($size->availabilities > $size->bound) {
                    $this->sizes[0][] = $size;
                }
            }
        }

        return $this->sizes[$all];
    }

    /**
     * @return string
     * @ignore
     *
     * @deprecated
     */
    public static function getProductBoundQuery()
    {
        return 'SELECT r.product_id, r.size_id, SUM(r.quantity) AS bound
                FROM ' . OrderRow::tableName() . ' AS r
                    INNER JOIN ' . Order::tableName() . ' AS o ON o.id = r.order_id
                WHERE ' . self::getProductBoundConditions('r') . '
                GROUP BY r.product_id, r.size_id';
    }

    /**
     * @param string $order_rows_alias
     * @return string
     * @ignore
     *
     * @deprecated
     */
    public static function getProductBoundConditions($order_rows_alias)
    {
        $bound_conditions = [
            "o.status = '" . Order::STATUS_CART . "' AND " . $order_rows_alias . ".updated_at > SUBTIME(NOW(), '01:00:00')",
            "o.status IN ('" . Order::STATUS_PAYMENT . "', '" . Order::STATUS_PAYMENT_LOCKED . "', '" . Order::STATUS_TRANSACTION . "') AND " . $order_rows_alias . ".updated_at > SUBTIME(NOW(), '02:00:00')",
            "o.status IN ('" . Order::STATUS_ORDER . "', '" . Order::STATUS_PENDING . "', '" . Order::STATUS_PAYMENT_AUTHORIZED . "')",
        ];

        return '(' . implode(') OR (', $bound_conditions) . ')';
    }

    /**
     * Restituisce il tipo di taglia associato al nodo.
     *
     * @return null|\Application\Showcase\Model\SizeType
     */
    public function getSizeType()
    {
        if ($this->size_type === null && $this->size_type_id) {
            $this->size_type = SizeType::find($this->size_type_id);
        }

        return $this->size_type;
    }

    /**
     * Imposta le taglie.
     *
     * L’array delle taglie passato deve essere del tipo:
     *
     *   array($size_name_id => $attributes)
     *
     * @param array $sizes
     */
    public function setSizes(array $sizes)
    {
        foreach ($sizes as $size_name_id => $attributes) {
            $size = Size::findByNodeIdAndSizeNameId($this->id, $size_name_id);

            if (!$size) {
                $size = new Size([
                    'node_id'      => $this->id,
                    'size_name_id' => $size_name_id,
                ]);
            }

            $size->updateAttributes($attributes);
            $size->save();
        }

        $this->updateAvailabilitiesAndBounds();
    }

    /**
     * Imposta le taglie personalizzate.
     *
     * L’array delle taglie passato deve essere del tipo:
     *
     *   array($size_id => $attributes)
     *
     * @param array $sizes
     */
    public function setCustomSizes(array $sizes)
    {
        /**
         * @var \Application\Showcase\Model\SizeName $size_name
         * @var \Application\Showcase\Model\Size     $size
         */

        $position = 1;
        foreach ($sizes as $size_name_id => $attributes) {
            // Il nome è un campo importante in quanto mi permette di rimuovere la taglia
            $name = isset($attributes['name']) ? $attributes['name'] : '';

            // Vediamo se ho già un record "Size" corrispondente
            $size = null;
            $size_name = null;
            if (is_numeric($size_name_id)) {
                $size = Size::findByNodeIdAndSizeNameId($this->id, $size_name_id);

                if ($size) {
                    $size_name = $size->size_name;
                }
            }

            if (!$name) {
                if ($size) {
                    if ($size_name) {
                        $size_name->delete();
                    } else {
                        $size->delete();
                    }
                }

                continue;
            }

            if ($size_name === null) {
                $size_name = new SizeName([
                    'size_type_id' => null,
                ]);
            }

            $size_name->name = $name;
            $size_name->position = $position;
            $size_name->save();

            if (!$size) {
                $size = new Size([
                    'node_id'      => $this->id,
                    'size_name_id' => $size_name->id,
                ]);
            }

            $size->updateAttributes($attributes);
            $size->save();

            $position++;
        }

        $this->updateAvailabilitiesAndBounds();
    }

    /**
     * Elimina le taglie associate.
     *
     * @return bool
     */
    public function deleteSizes()
    {
        if ($this->isNewRecord()) {
            $this->sizes = [];

            return true;
        }

        $options = ['conditions' => ['node_id = ?', $this->id]];

        foreach (Size::all($options) as $size) {
            if (!$size->delete()) {
                return false;
            }
        }

        $this->sizes = [];

        $this->updateAvailabilitiesAndBounds();

        return true;
    }

    /**
     * {@inheritdoc}
     *
     * @fixme Questo metodo è pensato per togliere i prodotti non disponibili dai correlati, ma fa riferimento a campi non più presenti in database.
     */
    protected function getRelatedQueryOptions($status = 'publish')
    {
        return parent::getRelatedQueryOptions($status);

        //$options = parent::getRelatedQueryOptions();
        //
        //// Aggiungo il controllo sulla disponibilità del prodotto
        //if ( $this->getSiteModule()->getOption('enable_availabilties') )
        //{
        //	$join = 'LEFT JOIN ' . Size::table_name() . ' AS s ON s.node_id = `from`.id';
        //
        //	if ( isset($options['joins']) )
        //	{
        //		$options['joins'] .= ' ' . $join;
        //	}
        //	else
        //	{
        //		$options['joins'] = $join;
        //	}
        //
        //	$options['group'] = '`from`.id';
        //
        //	$options['having'] = 'COUNT(s.bound_availabilities) <> 0';
        //}
        //
        //return $options;
    }

    /**
     * Restituisce il nodo per la guida legata alla taglia
     *
     * @return Node
     */
    public function getSizeGuide()
    {
        if ($this->getSiteModule()->getOption('size_guide_node_type') && ($size_type = $this->getSizeType()) !== null && $size_type->node_id) {
            return $size_type->node;
        }

        return null;
    }

    /**
     * Aggiorna le disponibilità e le quantità occupate.
     *
     * Al momento questo metodo delega al modello delle taglie il compito di calcolare le disponibilità bloccate. Questo
     * significa che vengono eseguite 4 query per ogni taglia e se il prodotto ha tante taglie potrebbe rappresentare
     * un processo lento. Si può ottimizzare replicando la logica presente nel modello delle taglie e portare tutto
     * a 3-4 query, più quelle di salvataggio dei dati (1 per taglia + quella del prodotto).
     *
     * @return $this
     */
    public function updateAvailabilitiesAndBounds()
    {
        if ($this->size_type_id) {
            $options = [
                'select' => '`from`.*',
                'joins' => 'LEFT JOIN ' . SizeName::tableName() . ' AS sn ON sn.id = `from`.size_name_id',
                'conditions' => [
                    '`from`.node_id = :node AND sn.size_type_id = :size_type',
                    'node' => $this->id,
                    'size_type' => $this->size_type_id,
                ],
            ];
        } else {
            $options = [
                'select' => '`from`.*',
                'joins' => 'LEFT JOIN ' . SizeName::tableName() . ' AS sn ON sn.id = `from`.size_name_id',
                'conditions' => [
                    '`from`.node_id = :node AND sn.size_type_id IS NULL',
                    'node' => $this->id,
                ],
            ];
        }

        $availabilities = 0;
        $bound_in_cart = 0;
        $bound_in_cart_updated_at = null;
        $bound_in_payment = 0;
        $bound_in_payment_updated_at = null;
        $bound_in_order = 0;

        /** @var Size $size */
        foreach (Size::all($options) as $size) {
            $size->updateBounds()->save();

            $availabilities += $size->availabilities;
            $bound_in_cart += $size->bound_in_cart;
            $bound_in_payment += $size->bound_in_payment;
            $bound_in_order += $size->bound_in_order;

            if ($size->bound_in_cart_updated_at && ($bound_in_cart_updated_at === null || $size->bound_in_cart_updated_at > $bound_in_cart_updated_at)) {
                $bound_in_cart_updated_at = $size->bound_in_cart_updated_at;
            }

            if ($size->bound_in_payment_updated_at && ($bound_in_payment_updated_at === null || $size->bound_in_payment_updated_at > $bound_in_payment_updated_at)) {
                $bound_in_payment_updated_at = $size->bound_in_payment_updated_at;
            }
        }

        $this->availabilities = $availabilities;
        $this->bound_in_cart = $bound_in_cart;
        $this->bound_in_cart_updated_at = $bound_in_cart_updated_at;
        $this->bound_in_payment = $bound_in_payment;
        $this->bound_in_payment_updated_at = $bound_in_payment_updated_at;
        $this->bound_in_order = $bound_in_order;

        return $this;
    }

    /**
     * @return int
     */
    public function getShoppingPoints()
    {
        return $this->shopping_points;
    }
}
