<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase;

require __DIR__ . '/functions.php';

use Application\Admin\Form\FormConfig;
use Application\Core\I18n\Translation\Translator;
use Application\Showcase\Controller\ShopController;
use Application\Showcase\DependencyInjection\AdminSidebarServiceProvider;
use Application\Showcase\DependencyInjection\GoogleShoppingServiceProvider;
use Application\Showcase\DependencyInjection\RoutingServiceProvider;
use Application\Showcase\DependencyInjection\ShowcaseServiceProvider;
use Application\Showcase\DependencyInjection\StatsServiceProvider;
use Application\Showcase\Installer\ShopInstaller;
use Application\Showcase\Installer\ShowcaseInstaller;
use Pongho\Core\Plugin;
use Pongho\EventDispatcher\Event;
use Pongho\Form\Field\TextField;
use Pongho\Form\Fieldset;
use Pongho\Template\Theme;
use Pongho\Utilities\Inflector;

/**
 * Kernel dell’applicazione.
 */
class Kernel extends Plugin
{
    const VERSION = '1.0.0-dev';
    const FRONT_CONTROLLER = ShopController::class;

    /**
     * {@inheritdoc}
     */
    public static function depends()
    {
        return ['Cms'];
    }

    /**
     * Carrello.
     *
     * @var \Application\Showcase\Model\Order
     */
    protected $cart;

    /**
     * {@inheritdoc}
     */
    public function setUp()
    {
        if (class_exists('Application\\Core\\Kernel') && class_exists('Application\\Admin\\Kernel') && class_exists('Application\\Cms\\Kernel')) {
            return true;
        }

        return false;
    }

    /**
     * {@inheritdoc}
     */
    public function load()
    {
        // Creazione di un nuovo sito
        $this->connect('core.after_create_site', 'addSomeStuffOnSiteCreation');

        // Aggiunge il controllo sugli ordini nella verifica dell’eliminabilità di un account.
        $this->connect('model.pongho_users.is_deletable', 'checkAccountDeletable');

        // Aggiungo le misure per le immagini dei prodotti nell’email di notifica ordine.
        $this->connect('core.filter_view', 'filterView');

        // Aggiungo lo showcase nelle traduzioni in amministrazione.
        $this->connect('admin.translations.filter_partitions', 'filterTranslationsPartitions');

        // Aggiunge il codice HTML per la conversione da tipo di nodo base a catalogo.
        $this->connect('cms.options.node_type_conversions', 'addNodeTypeConversionHtml');

        // Campi per Google Shopping
        $this->connect('admin.node_product.term_category.filter_form_options', 'filterTermsFormConfig');

        // Campo "Punti accumulati" nel profilo utente
        $this->connect('admin.users.filter_form_options', 'filterUserFormConfig');

        // Servizi
        $this->container->register(new ShowcaseServiceProvider());
        $this->container->register(new AdminSidebarServiceProvider());
        $this->container->register(new RoutingServiceProvider());
        $this->container->register(new StatsServiceProvider());
        $this->container->register(new GoogleShoppingServiceProvider());
    }

    /**
     * {@inheritdoc}
     */
    public function install()
    {
        // creazione del nodo `product`
        // creazione della tassonomia `category`
        // impostazioni base del modulo (mi viene in mente solo l’immagine in evidenza)
        ShowcaseInstaller::newInstance($this->container)->install();
        ShopInstaller::newInstance($this->container)->install();
    }

    /**
     * {@inheritdoc}
     */
    public function uninstall()
    {
        ShopInstaller::newInstance($this->container)->uninstall();
        ShowcaseInstaller::newInstance($this->container)->uninstall();
    }

    /**
     * Indica se la versione passata del CMS è compatibile con questo showcase.
     *
     * @param string $cms_version
     * @return bool
     */
    public static function isCmsCompatible($cms_version)
    {
        return true;
    }

    /**
     * Prepara le basi per un nuovo sito.
     *
     * Nel dettaglio: aggiunge il tipo di nodo `page`, crea la home page, aggiunge il modulo per la ricerca ed il
     * modulo per il feed.
     *
     * @param \Pongho\EventDispatcher\Event $event
     *
     * @ignore
     */
    public function addSomeStuffOnSiteCreation(Event $event)
    {
        /** @var \Application\Core\Controller\Admin\SitesController $controller */
        $controller = $event->getSubject();

        /** @var \Application\Core\Model\LanguageSite $language_site */
        $language_site = $event['language_site'];

        /** @var \Application\Core\Model\LanguageSite $source_language_site */
        $source_language_site = $event['source_language_site'];

        $controller->addModuleSite(
            ShopController::class,
            $language_site,
            $source_language_site
        );
    }

    /**
     * Evento 'core.filter_view'.
     *
     * Aggiunge le misure per le immagini dei prodotti nell’email di notifica ordine.
     *
     * @param \Pongho\EventDispatcher\Event $event
     * @param \Pongho\Template\Theme        $view
     * @return \Pongho\Template\Theme
     *
     * @ignore
     */
    public function filterView(Event $event, Theme $view)
    {
        $view->setOption(
            'image_sizes',
            array_merge(
                $view->getOption('image_sizes'),
                [
                    'emailorder'     => [80, 80, 'fill'],
                    'googleshopping' => [1000, 1000, 'auto'],
                ]
            )
        );

        return $view;
    }

    /**
     * Evento 'admin.translations.filter_partitions'.
     *
     * Aggiunge lo showcase nelle traduzioni in amministrazione.
     *
     * @param \Pongho\EventDispatcher\Event $event
     * @param array                         $partitions
     * @return array
     *
     * @ignore
     */
    public function filterTranslationsPartitions(Event $event, array $partitions)
    {
        $partitions[] = 'showcase';

        return $partitions;
    }

    /**
     * @param Event $event
     */
    public function checkAccountDeletable(Event $event)
    {
        /** @var \Application\Core\Model\Account $account */
        $account = $event->getSubject();

        $event->setParameter('is_deletable', $account->total_orders === 0);
    }

    /**
     * Aggiunge il codice HTML per la conversione da tipo di nodo base a catalogo.
     *
     * @param Event $event
     */
    public function addNodeTypeConversionHtml(Event $event)
    {
        /**
         * @var Translator $translator
         * @var \Application\Admin\Utilities\PonghoUrl $pongho_url
         * @var \Application\Cms\Model\NodeType $node_type
         */
        $translator = $this->container->get('translator');
        $pongho_url = $this->container->get('pongho_url');
        $node_type = $event['node_type'];

        $enable_showcase_url = $pongho_url->url(
            '/' . Inflector::pluralize($node_type->node_type) . '-showcase-converter/convert/'
        );

        $message = $translator->trans(
            'The conversion of this node type in showcase implies the addition of some options in the creation / edit panel of the nodes. You will also have access to a further range of options including those related to the management of the shop. %link_start%Click here%link_end% to convert this node type in showcase.',
            ['%link_start%' => '<a href="' . $enable_showcase_url . '">', '%link_end%' => '</a>']
        );

        echo '<fieldset class="panel"><h3>', $translator->trans('Showcase'), '</h3><p>', $message, '</p></fieldset>';
    }

    /**
     * @param Event      $event
     * @param FormConfig $config
     * @return FormConfig
     */
    public function filterTermsFormConfig(Event $event, FormConfig $config)
    {
        /** @var \Application\Core\Model\Site $site */
        $site = $this->container->get('site');

        if ($site->getOption('enable_google_shopping')) {
            /** @var \Application\Showcase\GoogleShopping\Form\TermFormConfigFilter $filter */
            $filter = $this->container->get('google_shopping_term_form_config_filter');
            $filter->filter($config);
        }

        return $config;
    }

    /**
     * @param Event      $event
     * @param FormConfig $config
     * @return FormConfig
     */
    public function filterUserFormConfig(Event $event, FormConfig $config)
    {
        /** @var \Application\Showcase\Discount\OrderDiscounts $orderDiscounts */
        $orderDiscounts = $this->container->get('shop_order_discounts');

        if ($orderDiscounts->isEnabled('shopping_points')) {
            /** @var \Application\Core\I18n\Translation\Translator $translator */
            $translator = $this->container->get('translator');

            $config->addField(
                'content/main/shopping-points',
                [
                    'class'      => Fieldset::class,
                    'attributes' => ['class' => 'panel'],
                    'label'      => $translator->trans('Shopping points'),
                ]
            );

            $config->addField(
                'content/main/shopping-points/shopping_points',
                [
                    'class' => TextField::class,
                    'attributes' => ['class' => 'smallest'],
                    'label' => $translator->trans('Collected points'),
                ]
            );
        }

        return $config;
    }
}
