<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\GoogleShopping\Form;

use Application\Admin\Form\FormConfig;
use Application\Core\I18n\Translation\Translator;
use Pongho\Form\Field\SelectField;
use Pongho\Form\Field\TextField;
use Pongho\Sdk\GoogleShopping\Property\AgeGroupProperty;
use Pongho\Sdk\GoogleShopping\Property\GenderProperty;
use Pongho\Template\HeadHelper;

/**
 * NodeFormConfigFilter.
 *
 * Gestisce la configurazione della form di aggiunta / modifica dei prodotti in amministrazione.
 */
class NodeFormConfigFilter
{
    /**
     * @var Translator
     */
    protected $translator;

    /**
     * @var HeadHelper
     */
    protected $head;

    /**
     * @param Translator $translator
     * @param HeadHelper $head
     */
    public function __construct(Translator $translator, HeadHelper $head)
    {
        $this->translator = $translator;
        $this->head = $head;
    }

    /**
     * @param FormConfig $config
     * @return FormConfig
     */
    public function filter(FormConfig $config)
    {
        $config->addTab('content/google-shopping', $this->translator->trans('Google Shopping'));
        $config->addPanel('content/google-shopping/fields');

        $this
            ->addColorField($config)
            ->addGenderField($config)
            ->addAgeGroupField($config)
            ->addMaterialField($config)
            ->addPatternField($config);

        return $config;
    }

    /**
     * @param FormConfig $config
     * @return $this
     */
    protected function addColorField(FormConfig $config)
    {
        $config->addField('content/google-shopping/fields/gs_color', $this->fieldColor());

        return $this;
    }

    /**
     * @return array
     */
    protected function fieldColor()
    {
        return [
            'class'   => TextField::class,
            'label'   => $this->translator->trans('Color'),
        ];
    }

    /**
     * @param FormConfig $config
     * @return $this
     */
    protected function addGenderField(FormConfig $config)
    {
        $config->addField('content/google-shopping/fields/gs_gender', $this->fieldGender());

        return $this;
    }

    /**
     * @return array
     */
    protected function fieldGender()
    {
        return [
            'class'   => SelectField::class,
            'label'   => $this->translator->trans('Gender'),
            'options' => [
                GenderProperty::GENDER_UNISEX => $this->translator->trans('Unisex'),
                GenderProperty::GENDER_FEMALE => $this->translator->trans('Female'),
                GenderProperty::GENDER_MALE   => $this->translator->trans('Male'),
            ],
        ];
    }

    /**
     * @param FormConfig $config
     * @return $this
     */
    protected function addAgeGroupField(FormConfig $config)
    {
        $config->addField('content/google-shopping/fields/gs_age_group', $this->fieldAgeGroup());

        return $this;
    }

    /**
     * @return array
     */
    protected function fieldAgeGroup()
    {
        return [
            'class'   => SelectField::class,
            'label'   => $this->translator->trans('Age group'),
            'options' => [
                AgeGroupProperty::AGE_GROUP_NEWBORN => $this->translator->trans('Newborn'),
                AgeGroupProperty::AGE_GROUP_INFANT  => $this->translator->trans('Infant'),
                AgeGroupProperty::AGE_GROUP_TODDLER => $this->translator->trans('Toddler'),
                AgeGroupProperty::AGE_GROUP_KIDS    => $this->translator->trans('Kids'),
                AgeGroupProperty::AGE_GROUP_ADULT   => $this->translator->trans('Adult'),
            ],
        ];
    }

    /**
     * @param FormConfig $config
     * @return $this
     */
    protected function addMaterialField(FormConfig $config)
    {
        $config->addField('content/google-shopping/fields/gs_material', $this->fieldMaterial());

        return $this;
    }

    /**
     * @return array
     */
    protected function fieldMaterial()
    {
        return [
            'class' => TextField::class,
            'label' => $this->translator->trans('Material'),
        ];
    }

    /**
     * @param FormConfig $config
     * @return $this
     */
    protected function addPatternField(FormConfig $config)
    {
        $config->addField('content/google-shopping/fields/gs_pattern', $this->fieldPattern());

        return $this;
    }

    /**
     * @return array
     */
    protected function fieldPattern()
    {
        return [
            'class' => TextField::class,
            'label' => $this->translator->trans('Pattern'),
        ];
    }
}
