<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Form\Subject;

use Application\Core\I18n\Address\Address as SimpleAddress;
use Application\Core\I18n\Address\AddressInterface;
use Application\Core\I18n\Address\AddressRendererFactory;
use Application\Core\I18n\Address\AddressValidatorFactory;
use Application\Core\Localization;
use Application\Core\Model\Account;
use Application\Core\Model\Address as ModelAddress;
use Application\Core\Model\Country;
use Application\Core\Model\Manager\ApplicationManagerInterface;
use Application\Core\Model\Province;
use Application\Core\Model\Role;
use Application\Core\Model\Site;
use Application\Core\User;
use Application\Core\Utilities\Mailer;
use Application\Core\Utilities\Validator;
use Application\Showcase\Exception\ShippingAddressUndefinedException;
use Application\Showcase\Model\Order;
use Pongho\Form\Subject\ArrayRowSubject;

/**
 * LoginSubject.
 */
class LoginSubject extends ArrayRowSubject
{
    const TYPE_LOGIN = 'login';
    const TYPE_SUBSCRIBE = 'subscribe';

    const INVOICE_ADDRESS_SAME = 'same';
    const INVOICE_ADDRESS_DIFFERENT = 'different';

    const CHECKOUT_GUEST = 'guest';
    const CHECKOUT_NORMAL = 'normal';

    /**
     * @var Order
     */
    protected $cart;

    /**
     * @var User
     */
    protected $user;

    /**
     * @var Localization
     */
    protected $lang;

    /**
     * @var Mailer
     */
    protected $mailer;

    /**
     * @var Site
     */
    protected $site;

    /**
     * @var ApplicationManagerInterface
     */
    protected $applicationManager;

    /**
     * @var array
     */
    protected $errors = [];

    /**
     * @var AddressValidatorFactory
     */
    protected $addressValidatorFactory;

    /**
     * @var AddressRendererFactory
     */
    protected $addressRendererFactory;

    /**
     * @var \Application\Core\Model\Account
     */
    protected $account;

    /**
     * @var string
     */
    protected $email;

    /**
     * @var \Application\Core\I18n\Address\AddressInterface|\Application\Core\Model\Address
     */
    protected $shipping_address;

    /**
     * @var \Application\Core\I18n\Address\AddressInterface
     */
    protected $invoice_address;

    /**
     * @var string
     */
    protected $customer_code;

    /**
     * @var bool
     */
    protected $hasShippingAddressException = false;

    /**
     * @var array
     */
    protected $provinceIdentityMap = [];

    /**
     * @var array
     */
    protected $countryIdentityMap = [];

    /**
     * @param Order                       $cart
     * @param User                        $user
     * @param Localization                $lang
     * @param Mailer                      $mailer
     * @param Site                        $site
     * @param ApplicationManagerInterface $applicationManager
     * @param AddressValidatorFactory     $addressValidatorFactory
     * @param AddressRendererFactory      $addressRendererFactory
     */
    public function __construct(
        Order $cart,
        User $user,
        Localization $lang,
        Mailer $mailer,
        Site $site,
        ApplicationManagerInterface $applicationManager,
        AddressValidatorFactory $addressValidatorFactory,
        AddressRendererFactory $addressRendererFactory
    ) {
        $this->cart = $cart;
        $this->user = $user;
        $this->lang = $lang;
        $this->mailer = $mailer;
        $this->site = $site;
        $this->applicationManager = $applicationManager;
        $this->addressValidatorFactory = $addressValidatorFactory;
        $this->addressRendererFactory = $addressRendererFactory;

        parent::__construct($this->getInitialValues());
    }

    /**
     * getInitialValue
     *
     * @return array
     */
    protected function getInitialValues()
    {
        $values = [
            'type'                 => self::TYPE_LOGIN,
            'invoice_address_mode' => self::INVOICE_ADDRESS_SAME,
        ];

        if ($this->site->getOption('enable_guest_checkout')) {
            $values['checkout_mode'] = self::CHECKOUT_GUEST;

            if ($this->cart->customerIsGuest()) {
                $values = array_merge(
                    $values,
                    $this->getGuestInitialValues()
                );
            }
        } else {
            $values['checkout_mode'] = self::CHECKOUT_NORMAL;
        }

        return $values;
    }

    /**
     * getGuestInitialValues
     *
     * @return array
     */
    protected function getGuestInitialValues()
    {
        return [
            'type'                         => self::TYPE_SUBSCRIBE,

            'shipping_address_name'        => $this->cart->shipping_name,
            'shipping_address_surname'     => $this->cart->shipping_surname,
            'shipping_address_address1'    => $this->cart->shipping_address1,
            'shipping_address_address2'    => $this->cart->shipping_address2,
            'shipping_address_postcode'    => $this->cart->shipping_postcode,
            'shipping_address_city'        => $this->cart->shipping_city,
            'shipping_address_province_id' => $this->cart->shipping_province_id,
            'shipping_address_country_id'  => $this->cart->shipping_country_id,
            'shipping_address_telephone'   => $this->cart->shipping_telephone,

            // @todo controllare se i due indirizzi sono effettivamente diversi
            'invoice_address_mode'         => self::INVOICE_ADDRESS_DIFFERENT,

            'invoice_info_name'            => $this->cart->invoice_name,
            'invoice_info_surname'         => $this->cart->invoice_surname,
            'invoice_info_address1'        => $this->cart->invoice_address1,
            'invoice_info_address2'        => $this->cart->invoice_address2,
            'invoice_info_postcode'        => $this->cart->invoice_postcode,
            'invoice_info_city'            => $this->cart->invoice_city,
            'invoice_info_province_id'     => $this->cart->invoice_province_id,
            'invoice_info_country_id'      => $this->cart->invoice_country_id,
            'invoice_info_telephone'       => $this->cart->invoice_telephone,

            'username_or_email'            => $this->cart->customer_email,
            'customer_code'                => $this->cart->customer_code_value,
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function internalSave()
    {
        if ($this->get('type') === self::TYPE_SUBSCRIBE) {
            if ($this->site->getOption('enable_guest_checkout')) {
                $checkout_mode = $this->get('checkout_mode');
            } else {
                $checkout_mode = LoginSubject::CHECKOUT_NORMAL;
            }

            if ($checkout_mode === LoginSubject::CHECKOUT_GUEST) {
                return $this->handleGuest();
            }

            return $this->handleSubscribe();
        }

        return $this->handleLogin();
    }

    /**
     * {@inheritdoc}
     */
    public function getErrors()
    {
        return $this->errors;
    }

    /**
     * handleLogin
     *
     * @return bool
     */
    protected function handleLogin()
    {
        $username_or_email = $this->get('username_or_email');
        $password = clean_password($this->get('login_password'));

        $errors = [];
        if ($this->user->login($username_or_email, $password, false, $errors)) {
            $account = $this->user->getAccount();

            $this->cart->customer_id = $account->getId();
            $this->cart->customer = $account;

            try {
                $this->cart->populateShippingAddressAttributes();
                $this->cart->populateInvoiceInfoAttributes();
            } catch (ShippingAddressUndefinedException $e) {
                $this->hasShippingAddressException = true;
            }

            $this->cart->save(false);

            return true;
        }

        if (isset($errors['username_or_email'])) {
            $this->errors['username_or_email'] = $errors['username_or_email'];
        }

        if (isset($errors['password'])) {
            $this->errors['login_password'] = $errors['password'];
        }

        return false;
    }

    /**
     * handleSubscribe
     *
     * @return bool
     */
    protected function handleSubscribe()
    {
        if (!$this->get('shipping_address_country_id') || !$this->get('invoice_info_country_id')) {
            /** @var Country $country */
            $country = Country::first(['conditions' => ["code = 'IT'"]]);

            if (!$this->get('shipping_address_country_id')) {
                $this->set('shipping_address_country_id', $country->getId());
            }

            if (!$this->get('invoice_info_country_id')) {
                $this->set('invoice_info_country_id', $country->getId());
            }
        }

        if ($this->applicationManager->isEnabled('Newsletter', $this->site)) {
            $account_attributes['newsletter'] = (bool)$this->model['newsletter'];
        }

        $this->validateOnSubscribe();

        if (empty($this->errors)) {
            $this->account->updateAttributes([
                'name'        => $this->invoice_address->getFirstName(),
                'surname'     => $this->invoice_address->getLastName(),
                'address'     => $this->invoice_address->getStreet(),
                'address2'    => $this->invoice_address->getAdditionalInfo(),
                'postcode'    => $this->invoice_address->getPostCode(),
                'city'        => $this->invoice_address->getCity(),
                'province_id' => $this->invoice_address->getProvince()->id,
                'country_id'  => $this->invoice_address->getCountry()->id,
                'telephone'   => $this->invoice_address->getTelephone(),
            ]);

            $this->account->save();

            $this->shipping_address->user_id = $this->account->id;
            $this->shipping_address->save(false);

            // Associo l'indirizzo di spedizione appena creato, come preferito dell'utente.
            $this->account->shipping_address_id = $this->shipping_address->id;
            $this->account->save();

            // Aggiorno il carrello
            $this->cart->updateAttributes(
                $this->getSubscribeCartAttributes()
            );

            $this->cart->save();

            // Email di notifica
            $this->sendSubscriptionNotificationEmails($this->account);

            // Login automatico
            $this->user->forceLogin($this->account);

            return true;
        }

        return false;
    }

    /**
     * handleGuest
     *
     * @return bool
     */
    protected function handleGuest()
    {
        if (!$this->get('shipping_address_country_id') || !$this->get('invoice_info_country_id')) {
            /** @var Country $country */
            $country = Country::first(['conditions' => ["code = 'IT'"]]);

            if (!$this->get('shipping_address_country_id')) {
                $this->set('shipping_address_country_id', $country->getId());
            }

            if (!$this->get('invoice_info_country_id')) {
                $this->set('invoice_info_country_id', $country->getId());
            }
        }

        $this->validateOnGuest();

        if (empty($this->errors)) {
            // Aggiorno il carrello
            $this->cart->updateAttributes(
                $this->getGuestCartAttributes()
            );

            $this->cart->save();

            return true;
        }

        return false;
    }

    /**
     * validateOnSubscribe
     */
    protected function validateOnSubscribe()
    {
        $this->validateAccount();
        $this->validateSubscribeShippingAddress();
        $this->validateInvoiceAddress();
        $this->validateCustomerCode();
    }

    /**
     * validateOnGuest
     */
    protected function validateOnGuest()
    {
        $this->validateEmail();
        $this->validateGuestShippingAddress();
        $this->validateInvoiceAddress();
        $this->validateCustomerCode();
    }

    /**
     * validateEmail
     */
    protected function validateEmail()
    {
        $this->email = trim($this->get('username_or_email'));

        if (empty($this->email)) {
            $this->errors['username_or_email'] = 'required_email';
        } elseif (!preg_match(REGEXP_VALIDATE_EMAIL, $this->email)) {
            $this->errors['username_or_email'] = 'invalid_email';
        }
    }

    /**
     * validateAccount
     */
    protected function validateAccount()
    {
        $this->account = new Account($this->getSubscribeAccountAttributes());
        $this->account->validate();

        // Ci possono essere errori solo su 3 campi: email, username e password.
        // Devo ignorare gli errori sullo username, perché non è un campo presente nella form.
        // Gli errori sullo username sono: required_username, invalid_username_uniqueness, e invalid_username_format.
        // Se non ho uno username, non ho nemmeno la mail, quindi segnalo l'errore sulla mail.
        // Se lo username è già presente, vuol dire che c'è anche una mail già presente.
        // Se lo username non è valido, vuol dire che anche la mail non è valida.
        // Quindi ignoro gli errori sullo username e segnalo solo quelli sulla mail.

        $account_errors = $this->account->errors->all_as_hash();

        if (isset($account_errors['email'])) {
            $this->errors['username_or_email'] = $account_errors['email'];
        }

        if (isset($account_errors['password'])) {
            $this->errors['account_password'] = $account_errors['password'];
        }

        if (isset($account_errors['confirm_password'])) {
            $this->errors['account_confirm_password'] = $account_errors['confirm_password'];
        }
    }

    /**
     * getSubscribeAccountAttributes
     *
     * @return array
     */
    protected function getSubscribeAccountAttributes()
    {
        return [
            'language_id'      => $this->lang->getLanguageId(),
            'role_id'          => Role::USER_LOGGED,
            'is_active'        => true,
            'is_founder'       => false,
            'email'            => trim($this->get('username_or_email')),
            'privacy'          => true, // @todo
            'password'         => $this->get('account_password'),
            'confirm_password' => $this->get('account_confirm_password'),
            'username'         => trim($this->get('username_or_email')),
        ];
    }

    /**
     * validateSubscribeShippingAddress
     */
    protected function validateSubscribeShippingAddress()
    {
        $this->shipping_address = new ModelAddress($this->getShippingAddressAttributes());

        $this->validateShippingAddress();
    }

    /**
     * validateShippingAddress
     */
    protected function validateGuestShippingAddress()
    {
        $shipping_address_attributes = $this->getShippingAddressAttributes();

        $this->shipping_address = new SimpleAddress(
            $shipping_address_attributes['name'],
            $shipping_address_attributes['surname'],
            '',
            $shipping_address_attributes['address1'],
            $shipping_address_attributes['address2'],
            $shipping_address_attributes['postcode'],
            $shipping_address_attributes['city'],
            $shipping_address_attributes['province'],
            $shipping_address_attributes['country'],
            $shipping_address_attributes['telephone']
        );

        $this->validateShippingAddress();
    }

    /**
     * validateShippingAddress
     */
    protected function validateShippingAddress()
    {
        $errors = [];
        $this->validateAddress($this->shipping_address, $errors);

        if (!empty($errors)) {
            $errorsMap = [
                'name'      => 'shipping_address_name',
                'surname'   => 'shipping_address_surname',
                'address'   => 'shipping_address_address1',
                'address2'  => 'shipping_address_address2',
                'postcode'  => 'shipping_address_postcode',
                'city'      => 'shipping_address_city',
                'province'  => 'shipping_address_province_id',
                'country'   => 'shipping_address_country_id',
                'telephone' => 'shipping_address_telephone',
            ];

            foreach ($errorsMap as $key => $original) {
                if (isset($errors[$key])) {
                    $this->errors[$original] = $errors[$key];
                }
            }
        }
    }

    /**
     * validateInvoiceAddress
     */
    protected function validateInvoiceAddress()
    {
        if ($this->get('invoice_address_mode') === self::INVOICE_ADDRESS_DIFFERENT) {
            $invoice_address_attributes = $this->getInvoiceAddressAttributes();

            $this->invoice_address = new SimpleAddress(
                $invoice_address_attributes['name'],
                $invoice_address_attributes['surname'],
                '',
                $invoice_address_attributes['address1'],
                $invoice_address_attributes['address2'],
                $invoice_address_attributes['postcode'],
                $invoice_address_attributes['city'],
                $invoice_address_attributes['province'],
                $invoice_address_attributes['country'],
                $invoice_address_attributes['telephone']
            );

            $errors = [];
            $this->validateAddress($this->invoice_address, $errors);

            if (!empty($errors)) {
                $errorsMap = [
                    'name'      => 'invoice_info_name',
                    'surname'   => 'invoice_info_surname',
                    'address'   => 'invoice_info_address1',
                    'address2'  => 'invoice_info_address2',
                    'postcode'  => 'invoice_info_postcode',
                    'city'      => 'invoice_info_city',
                    'province'  => 'invoice_info_province_id',
                    'country'   => 'invoice_info_country_id',
                    'telephone' => 'invoice_info_telephone',
                ];

                foreach ($errorsMap as $key => $original) {
                    if (isset($errors[$key])) {
                        $this->errors[$original] = $errors[$key];
                    }
                }
            }
        } else {
            $invoice_address_attributes = $this->getShippingAddressAttributes();

            $this->invoice_address = new SimpleAddress(
                $invoice_address_attributes['name'],
                $invoice_address_attributes['surname'],
                '',
                $invoice_address_attributes['address1'],
                $invoice_address_attributes['address2'],
                $invoice_address_attributes['postcode'],
                $invoice_address_attributes['city'],
                $invoice_address_attributes['province'],
                $invoice_address_attributes['country'],
                $invoice_address_attributes['telephone']
            );
        }
    }

    /**
     * validateCustomerCode
     *
     * @todo I18n
     */
    protected function validateCustomerCode()
    {
        $this->customer_code = $this->get('customer_code');

        if (empty($this->customer_code)) {
            $this->errors['customer_code'] = 'required_codice_fiscale';
        } elseif (!Validator::codice_fiscale($this->customer_code)) {
            $this->errors['customer_code'] = 'invalid_codice_fiscale';
        }
    }

    /**
     * getShippingAddressAttributes
     *
     * @return array
     */
    protected function getShippingAddressAttributes()
    {
        static $attributes;

        if ($attributes === null) {
            $attributes = $this->getAddressAttributesValues('shipping_address');
        }

        return $attributes;
    }

    /**
     * getInvoiceAddressAttributes
     *
     * @return array
     */
    protected function getInvoiceAddressAttributes()
    {
        static $attributes;

        if ($attributes === null) {
            $attributes = $this->getAddressAttributesValues('invoice_info');
        }

        return $attributes;
    }

    /**
     * getAddressAttributes
     *
     * @param string $prefix
     * @return array
     */
    protected function getAddressAttributesValues($prefix)
    {
        $attributes = [];

        foreach ($this->getAddressAttributesNames() as $attribute) {
            $attributes[$attribute] = $this->get($prefix . '_' . $attribute);
        }

        $attributes['province_id'] = (int) $attributes['province_id'];
        $attributes['country_id'] = (int) $attributes['country_id'];

        if (!array_key_exists($attributes['province_id'], $this->provinceIdentityMap)) {
            $this->provinceIdentityMap[$attributes['province_id']] = Province::find($attributes['province_id']);
        }

        $attributes['province'] = $this->provinceIdentityMap[$attributes['province_id']];

        if (!array_key_exists($attributes['country_id'], $this->countryIdentityMap)) {
            $this->countryIdentityMap[$attributes['country_id']] = Country::find($attributes['country_id']);
        }

        $attributes['country'] = $this->countryIdentityMap[$attributes['country_id']];

        return $attributes;
    }

    /**
     * getAddressAttributesNames
     *
     * @return array
     */
    protected function getAddressAttributesNames()
    {
        return [
            'name',
            'surname',
            'address1',
            'address2',
            'postcode',
            'city',
            'province_id',
            'country_id',
            'telephone',
        ];
    }

    /**
     * getSubscribeCartAttributes
     *
     * @return array
     */
    protected function getSubscribeCartAttributes()
    {
        return [
            'shipping_address_id'   => $this->shipping_address->id,
            'shipping_name'         => $this->shipping_address->getFirstName(),
            'shipping_surname'      => $this->shipping_address->getLastName(),
            'shipping_address1'     => $this->shipping_address->getStreet(),
            'shipping_address2'     => $this->shipping_address->getAdditionalInfo(),
            'shipping_postcode'     => $this->shipping_address->getPostcode(),
            'shipping_city'         => $this->shipping_address->getCity(),
            'shipping_province_id'  => $this->shipping_address->getProvince()->id,
            'shipping_country_id'   => $this->shipping_address->getCountry()->id,
            'shipping_telephone'    => $this->shipping_address->getTelephone(),

            'shipping_address_text' => $this->renderAddress($this->shipping_address),

            'invoice_name'          => $this->invoice_address->getFirstName(),
            'invoice_surname'       => $this->invoice_address->getLastName(),
            'invoice_address1'      => $this->invoice_address->getStreet(),
            'invoice_address2'      => $this->invoice_address->getAdditionalInfo(),
            'invoice_postcode'      => $this->invoice_address->getPostCode(),
            'invoice_city'          => $this->invoice_address->getCity(),
            'invoice_province_id'   => $this->invoice_address->getProvince()->id,
            'invoice_country_id'    => $this->invoice_address->getCountry()->id,
            'invoice_telephone'     => $this->invoice_address->getTelephone(),

            'invoice_address_text'  => $this->renderAddress($this->invoice_address),

            'customer_is_guest'     => false,
            'customer_id'           => $this->account->id,
            'customer_email'        => $this->account->email,
            'customer_code_name'    => 'codice_fiscale', // @todo I18n
            'customer_code_value'   => $this->customer_code,
        ];
    }

    /**
     * getGuestCartAttributes
     *
     * @return array
     */
    protected function getGuestCartAttributes()
    {
        return [
            'shipping_name'         => $this->shipping_address->getFirstName(),
            'shipping_surname'      => $this->shipping_address->getLastName(),
            'shipping_address1'     => $this->shipping_address->getStreet(),
            'shipping_address2'     => $this->shipping_address->getAdditionalInfo(),
            'shipping_postcode'     => $this->shipping_address->getPostcode(),
            'shipping_city'         => $this->shipping_address->getCity(),
            'shipping_province_id'  => $this->shipping_address->getProvince()->id,
            'shipping_country_id'   => $this->shipping_address->getCountry()->id,
            'shipping_telephone'    => $this->shipping_address->getTelephone(),

            'shipping_address_text' => $this->renderAddress($this->shipping_address),

            'invoice_name'          => $this->invoice_address->getFirstName(),
            'invoice_surname'       => $this->invoice_address->getLastName(),
            'invoice_address1'      => $this->invoice_address->getStreet(),
            'invoice_address2'      => $this->invoice_address->getAdditionalInfo(),
            'invoice_postcode'      => $this->invoice_address->getPostCode(),
            'invoice_city'          => $this->invoice_address->getCity(),
            'invoice_province_id'   => $this->invoice_address->getProvince()->id,
            'invoice_country_id'    => $this->invoice_address->getCountry()->id,
            'invoice_telephone'     => $this->invoice_address->getTelephone(),

            'invoice_address_text'  => $this->renderAddress($this->invoice_address),

            'customer_is_guest'     => true,
            'customer_id'           => Account::ANONYMOUS,
            'customer_email'        => $this->email,
            'customer_code_name'    => 'codice_fiscale', // @todo I18n
            'customer_code_value'   => $this->customer_code,
        ];
    }

    /**
     * Invia le e-mail di notifica del nuovo iscritto.
     *
     * @param Account $account
     */
    protected function sendSubscriptionNotificationEmails(Account $account)
    {
        // Variabili per l'e-mail
        $email_vars = [
            'USER_EMAIL'    => $account->email,
            'USER_PASSWORD' => html_escape($this->get('account_password')),
            'USER_ADMIN'    => pongho_url("/users/edit/{$account->id}/?site=" . $this->site->id),
        ];

        // Email al cliente
        $from = [$this->site->getOption('company_email') => $this->site->getOption('company_name')];
        $to = [$account->email => $account->name()];
        $subject = sprintf($this->lang->get('shop_email_subscribe_subject_to_user'), $this->site->getName());

        $content = $this->mailer->content(
            $this->site->getThemesPath('email/shop_subscribe_to_user.php'),
            $email_vars
        );

        try {
            $this->mailer->send($from, $to, $subject, $content);
        } catch (\Exception $e) {
            // @todo Log
        }

        // Email all’amministratore
        $from = [$this->site->getOption('company_email') => $this->site->getOption('company_name')];
        $to = [$this->site->getOption('company_email') => $this->site->getOption('company_name')];
        $reply_to = [$account->email => $account->name()];
        $subject = $this->lang->get('shop_email_subscribe_subject_to_admin');

        $content = $this->mailer->content(
            $this->site->getThemesPath('email/shop_subscribe_to_admin.php'),
            $email_vars
        );

        try {
            $this->mailer->send($from, $to, $subject, $content, $reply_to);
        } catch (\Exception $e) {
            // @todo Log
        }
    }

    /**
     * hasShippingAddressException
     *
     * @return bool
     */
    public function hasShippingAddressException()
    {
        return $this->hasShippingAddressException;
    }

    /**
     * @param AddressInterface $address
     * @param array            $errors
     * @return bool
     */
    protected function validateAddress(AddressInterface $address, array &$errors)
    {
        $country = $address->getCountry();
        $countryCode = $country ? $country->code : 'IT';

        return $this->addressValidatorFactory->create($countryCode)->validate($address, $errors);
    }

    /**
     * @param AddressInterface $address
     * @return string
     */
    protected function renderAddress(AddressInterface $address)
    {
        $country = $address->getCountry();
        $countryCode = $country ? $country->code : 'IT';

        return $this->addressRendererFactory->create($countryCode)->render($address);
    }
}
