<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Form\Builder;

use Application\Core\Localization;
use Application\Core\Model\Address;
use Application\Core\Model\Province;
use Pongho\Form\Field\HiddenField;
use Pongho\Form\Field\SelectField;
use Pongho\Form\Field\TextField;
use Pongho\Form\Subject\ModelSubject;
use Pongho\Form\Utilities\FormBuilder;
use Pongho\Form\Utilities\FormConfig;

/**
 * AddressFormBuilder.
 */
class AddressFormBuilder
{
    /**
     * @var Address
     */
    protected $address;

    /**
     * @var Localization
     */
    protected $lang;

    /**
     * @param Address      $address
     * @param Localization $lang
     */
    public function __construct(Address $address, Localization $lang)
    {
        $this->address = $address;
        $this->lang = $lang;
    }

    /**
     * build
     *
     * @return \Pongho\Form\Form
     */
    public function build()
    {
        $config = new FormConfig('address', $this->getSubject());

        $config->addFields(
            '',
            [
                'name'        => [
                    'class'      => TextField::class,
                    'label'      => $this->lang->get('name'),
                    'attributes' => [
                        'maxlength' => 50,
                    ],
                ],
                'surname'     => [
                    'class'      => TextField::class,
                    'label'      => $this->lang->get('surname'),
                    'attributes' => [
                        'maxlength' => 50,
                    ],
                ],
                'address1'    => [
                    'class'      => TextField::class,
                    'label'      => $this->lang->get('address'),
                    'attributes' => [
                        'maxlength' => 50,
                    ],
                ],
                'address2'    => [
                    'class'       => TextField::class,
                    'label'       => $this->lang->get('address_info'),
                    'description' => $this->lang->get('address_info_help'),
                    'attributes'  => [
                        'maxlength' => 50,
                    ],
                ],
                'postcode'    => [
                    'class'      => TextField::class,
                    'label'      => $this->lang->get('postcode'),
                    'attributes' => [
                        'maxlength' => 50,
                    ],
                ],
                'city'        => [
                    'class'      => TextField::class,
                    'label'      => $this->lang->get('city'),
                    'attributes' => [
                        'maxlength' => 50,
                    ],
                ],
                'province_id' => [
                    'class'    => SelectField::class,
                    'label'    => $this->lang->get('province'),
                    'options'  => $this->getProvinces(),
                    'settings' => [
                        'placeholder' => $this->lang->get('select_province'),
                    ],
                ],
                'country_id'  => [
                    'class' => HiddenField::class,
                    'label' => $this->lang->get('country'),
                ],
                'telephone'   => [
                    'class'      => TextField::class,
                    'label'      => $this->lang->get('telephone'),
                    'attributes' => [
                        'maxlength' => 50,
                    ],
                ],
            ]
        );

        return FormBuilder::buildConfig($config);
    }

    /**
     * getSubject
     *
     * @return ModelSubject
     */
    protected function getSubject()
    {
        return new ModelSubject($this->address);
    }

    /**
     * getProvinces
     *
     * @return array
     *
     * @todo I18n
     */
    protected function getProvinces()
    {
        static $provinces;

        if ($provinces === null) {
            $options = ['order' => 'name'];

            /** @var Province $province */
            $provinces = [];
            foreach (Province::all($options) as $province) {
                $provinces[$province->id] = $province->name;
            }
        }

        return $provinces;
    }
}
