<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Export\DataExporter;

use Application\Core\Export\DataExporterInterface;
use Application\Core\I18n\Translation\Translator;
use Doctrine\DBAL\Connection;
use Pongho\Core\LocalizationInterface;
use Pongho\Utilities\DateTime;

/**
 * OrdersExporter
 */
class OrdersExporter implements DataExporterInterface
{
    /**
     * @var Connection
     */
    protected $connection;

    /**
     * @var Translator
     */
    protected $translator;

    /**
     * @var LocalizationInterface
     */
    protected $language;

    /**
     * @param Connection            $connection
     * @param Translator            $translator
     * @param LocalizationInterface $language
     */
    public function __construct(Connection $connection, Translator $translator, LocalizationInterface $language)
    {
        $this->connection = $connection;
        $this->translator = $translator;
        $this->language = $language;
    }

    /**
     * {@inheritdoc}
     */
    public function headers()
    {
        return [
            'order_id'           => $this->translator->trans('Id'),
            'customer_fullname'  => $this->translator->trans('Customer'),
            'customer_email'     => $this->translator->trans('Email'),
            'payment_method'     => $this->translator->trans('Payment Method'),
            'shipping_method'    => $this->translator->trans('Shipping Method'),
            'order_status'       => $this->translator->trans('Status'),
            'order_ordered_at'   => $this->translator->trans('Ordered at'),
            'order_paid_at'      => $this->translator->trans('Paid at'),
            'order_processed_at' => $this->translator->trans('Processed at'),
            'order_total'        => $this->translator->trans('Total'),
            'row_id'             => $this->translator->trans('Row ID'),
            'product_code'       => $this->translator->trans('Code'),
            'product_title'      => $this->translator->trans('Product'),
            'size_name'          => $this->translator->trans('Size'),
            'row_quantity'       => $this->translator->trans('Quantity'),
            'row_price'          => $this->translator->trans('Price'),
            'row_total'          => $this->translator->trans('Row total'),
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function export()
    {
        $sth = $this->getStatement();
        $sth->execute();

        foreach ($sth as $row) {
            yield $this->parseRow($row);
        }
    }

    /**
     * @return \Doctrine\DBAL\Driver\Statement
     * @throws \Doctrine\DBAL\DBALException
     */
    protected function getStatement()
    {
        return $this->connection->prepare($this->getQuery());
    }

    /**
     * @return \Doctrine\DBAL\Query\QueryBuilder
     */
    protected function getQuery()
    {
        $qb = $this->connection->createQueryBuilder();

        $qb
            ->select([
                'order_id'           => 'orders.id AS order_id',
                'customer_fullname'  => 'customers.fullname AS customer_fullname',
                'customer_email'     => 'customers.email AS customer_email',
                'payment_method'     => 'payments.name AS payment_method',
                'shipping_method'    => 'shippings.name AS shipping_method',
                'order_status'       => 'orders.status AS order_status',
                'order_ordered_at'   => 'orders.ordered_at AS order_ordered_at',
                'order_paid_at'      => 'orders.paid_at AS order_paid_at',
                'order_processed_at' => 'orders.processed_at AS order_processed_at',
                'order_total'        => 'orders.total AS order_total',
                'row_id'             => 'rows.row_id',
                'product_code'       => "CONCAT('=\"', products.code, '\"') AS product_code",
                'product_title'      => 'products_trans.title AS product_title',
                'size_name'          => 'sizes_names.name AS size_name',
                'row_quantity'       => 'rows.quantity AS row_quantity',
                'row_price'          => 'rows.price AS row_price',
                'row_total'          => 'rows.total AS row_total',
            ])
            ->from('pongho_orders', 'orders')
            ->innerJoin('orders', 'pongho_orders_rows', 'rows', 'rows.order_id = orders.id')
            ->leftJoin('rows', 'pongho_nodes_sizes', 'sizes', 'sizes.id = rows.size_id')
            ->leftJoin('sizes', 'pongho_nodes_sizes_names', 'sizes_names', 'sizes_names.id = sizes.size_name_id')
            ->innerJoin('rows', 'pongho_nodes', 'products', 'products.id = rows.product_id')
            ->innerJoin('rows', 'pongho_nodes_translations', 'products_trans', 'products_trans.node_id = rows.product_id AND products_trans.language_id = 1')
            ->innerJoin('orders', 'pongho_users', 'customers', 'customers.id = orders.customer_id')
            ->innerJoin('orders', 'pongho_payments', 'payments', 'payments.id = orders.payment_id')
            ->innerJoin('orders', 'pongho_shippings', 'shippings', 'shippings.id = orders.shipping_id')
            ->where("orders.status NOT IN ('cart', 'payment', 'transaction')")
            ->orderBy('orders.ordered_at', 'DESC')
            ->addOrderBy('rows.row_id', 'ASC')
        ;

        return $qb;
    }

    /**
     * @param array $row
     * @return mixed
     */
    protected function parseRow(array $row)
    {
        // status
        $row['order_status'] = $this->language->get('order_status_' . $row['order_status']);

        // dates
        $row['order_ordered_at'] = $this->parseDate($row['order_ordered_at']);
        $row['order_paid_at'] = $this->parseDate($row['order_paid_at']);
        $row['order_processed_at'] = $this->parseDate($row['order_processed_at']);

        // prices
        $row['order_total'] = $this->formatPrice($row['order_total']);
        $row['row_price'] = $this->formatPrice($row['row_price']);
        $row['row_total'] = $this->formatPrice($row['row_total']);

        return $row;
    }

    /**
     * @param string $date
     * @return string
     */
    protected function parseDate($date)
    {
        if (!$date) {
            return '';
        }

        try {
            return (new DateTime($date))->localeFormat($this->getLocaleDateFormat());
        } catch (\Exception $e) {
            return '';
        }
    }

    /**
     * @return string
     */
    protected function getLocaleDateFormat()
    {
        return '%x';
    }

    /**
     * @param string $price
     * @return string
     */
    protected function formatPrice($price)
    {
        return format_price($price);
    }
}
