<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Domain\Order\Command;

use Application\Core\I18n\Translation\Translator;
use Application\Showcase\Domain\CommandInterface;
use Application\Showcase\Domain\CommandResult;
use Application\Showcase\Model\Order;
use Pongho\Utilities\DateTime;

/**
 * ConfirmOrderPaymentCommand
 */
class ConfirmOrderPaymentCommand implements CommandInterface
{
    /**
     * @var Translator
     */
    protected $translator;

    /**
     * @var Order
     */
    protected $order;

    /**
     * @var DateTime
     */
    protected $date;

    /**
     * @var bool
     */
    protected $successful = true;

    /**
     * @var array
     */
    protected $alerts = [];

    /**
     * @var array
     */
    protected $errors = [];

    /**
     * @param Translator $translator
     */
    public function __construct(Translator $translator)
    {
        $this->translator = $translator;
    }

    /**
     * Il reset deve essere esguito prima di processare un’altro ordine.
     *
     * @return $this
     */
    public function reset()
    {
        $this->order = null;
        $this->date = null;

        return $this;
    }

    /**
     * @param Order $order
     * @return $this
     */
    public function setOrder(Order $order)
    {
        $this->order = $order;

        return $this;
    }

    /**
     * @param DateTime $date
     * @return $this
     */
    public function setDate(DateTime $date)
    {
        $this->date = $date;

        return $this;
    }

    /**
     * @inheritdoc
     */
    public function handle()
    {
        if (null === $this->order) {
            throw new \LogicException('The order must be set before handle the command.');
        }

        if ($this->order->isPaid()) {
            return new CommandResult(true);
        }

        // reset result
        $this->successful = true;
        $this->alerts = [];
        $this->errors = [];

        // handle
        try {
            $this->handleRaw();
            $this->persist();
        } catch (\Exception $e) {
            $this->successful = false;
            $this->errors[] = $e->getMessage();
        }

        $message = '';
        if ($this->successful) {
            $message = $this->translator->trans('The order payment has been confirmed.');
        }

        return new CommandResult(
            $this->successful,
            $message,
            $this->errors,
            $this->alerts
        );
    }

    /**
     * Internal handler for customization purposes.
     */
    protected function handleRaw()
    {
        $this->order->status = Order::STATUS_ORDER;
        $this->order->paid_at = $this->date;
    }

    protected function persist()
    {
        $this->order->save();
        $this->order->updateBounds();
    }
}
