<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Domain\Order\Command;

use Application\Core\I18n\Translation\Translator;
use Application\Showcase\Domain\CommandInterface;
use Application\Showcase\Domain\CommandResult;
use Application\Showcase\Model\Order;

/**
 * CancelOrderCommand
 */
class CancelOrderCommand implements CommandInterface
{
    /**
     * @var Translator
     */
    protected $translator;

    /**
     * @var Order
     */
    protected $order;

    /**
     * @var bool
     */
    protected $successful = true;

    /**
     * @var array
     */
    protected $alerts = [];

    /**
     * @var array
     */
    protected $errors = [];

    /**
     * @param Translator $translator
     */
    public function __construct(Translator $translator)
    {
        $this->translator = $translator;
    }

    /**
     * Il reset deve essere esguito prima di processare un’altro ordine.
     *
     * @return $this
     */
    public function reset()
    {
        $this->order = null;

        return $this;
    }

    /**
     * @param Order $order
     * @return $this
     */
    public function setOrder(Order $order)
    {
        $this->order = $order;

        return $this;
    }

    /**
     * @inheritdoc
     */
    public function handle()
    {
        if (null === $this->order) {
            throw new \LogicException('The order must be set before handle the command.');
        }

        if ($this->order->isCanceled()) {
            return new CommandResult(true);
        }

        // reset result
        $this->successful = true;
        $this->alerts = [];
        $this->errors = [];

        try {
            $this->handleRaw();
            $this->persist();
        } catch (\Exception $e) {
            $this->successful = false;
            $this->errors[] = $e->getMessage();
        }

        $message = '';
        if ($this->successful) {
            $message = $this->translator->trans('The order has been canceled.');
        }

        return new CommandResult(
            $this->successful,
            $message,
            $this->errors,
            $this->alerts
        );
    }

    /**
     * Internal handler for customization purposes.
     */
    protected function handleRaw()
    {
        $this->order->status = $this->order->isPaid() ? Order::STATUS_INVOICE_CANCELED : Order::STATUS_CANCELED;
    }

    protected function persist()
    {
        if (!$this->successful) {
            return;
        }

        $this->order->save();
        $this->order->updateBounds();
    }
}
