<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Discount;

use Application\Core\Localization;
use Application\Showcase\Discount\Order\CouponCodeDiscount;
use Application\Showcase\Discount\Order\OrderAmountDiscount;
use Application\Showcase\Discount\Order\ShoppingPointsDiscount;
use Application\Showcase\Exception\CouponException;
use Application\Showcase\Model\Order;
use Application\Showcase\Model\OrderDiscount;
use Application\Showcase\ShoppingPoints\Exception\ShoppingPointsException;
use Psr\Log\LoggerInterface;

/**
 * OrderDiscountsFacade
 */
class OrderDiscountsFacade
{
    /**
     * @var CouponCodeDiscount
     */
    private $couponCodeDiscount;

    /**
     * @var OrderAmountDiscount
     */
    private $orderAmountDiscount;

    /**
     * @var ShoppingPointsDiscount
     */
    private $shoppingPointsDiscount;

    /**
     * @var OrderDiscounts
     */
    private $enabledDiscounts;

    /**
     * @var Localization
     */
    private $lang;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @param CouponCodeDiscount     $couponCodeDiscount
     * @param OrderAmountDiscount    $orderAmountDiscount
     * @param ShoppingPointsDiscount $shoppingPointsDiscount
     * @param OrderDiscounts         $enabledDiscounts
     * @param Localization           $lang
     * @param LoggerInterface        $logger
     */
    public function __construct(
        CouponCodeDiscount $couponCodeDiscount,
        OrderAmountDiscount $orderAmountDiscount,
        ShoppingPointsDiscount $shoppingPointsDiscount,
        OrderDiscounts $enabledDiscounts,
        Localization $lang,
        LoggerInterface $logger
    ) {
        $this->couponCodeDiscount = $couponCodeDiscount;
        $this->orderAmountDiscount = $orderAmountDiscount;
        $this->shoppingPointsDiscount = $shoppingPointsDiscount;
        $this->enabledDiscounts = $enabledDiscounts;
        $this->lang = $lang;
        $this->logger = $logger;
    }

    /**
     * @param Order  $order
     * @param string $couponCode
     */
    public function applyCouponCode(Order $order, $couponCode)
    {
        $this->logger->info(
            'Updating the coupon discount.',
            [
                'cart' => $order->getId() ?: 'NULL',
                'code' => $couponCode,
            ]
        );

        if (empty($couponCode)) {
            $this->logger->info(
                'No coupon code found, then remove the discount.',
                [
                    'cart' => $order->getId() ?: 'NULL',
                ]
            );

            $this->removeCouponCodeDiscount($order);
        } else {
            /**
             * @var OrderDiscount $coupon
             * @todo OrderDiscountManager
             */
            $coupon = OrderDiscount::findByCode($couponCode);

            if ($coupon === null) {
                $this->logger->notice(
                    'Coupon not exists, then remove the discount.',
                    [
                        'cart' => $order->getId() ?: 'NULL',
                        'code' => $couponCode,
                    ]
                );

                $this->removeCouponCodeDiscount($order);

                $order->errors->add_to_base('coupon_error_not_exists');
            } else {
                // Buono trovato
                $this->applyCouponCodeDiscount($order, $coupon);
            }
        }
    }

    /**
     * @param Order $order
     * @return bool
     */
    public function hasCouponCode(Order $order)
    {
        return $this->couponCodeDiscount->has($order);
    }

    /**
     * @param Order $order
     */
    public function updateOrderAmount(Order $order)
    {
        $this->orderAmountDiscount->update($order);
    }

    /**
     * @param Order $order
     * @param int   $usedPoints
     */
    public function updateShoppingPoints(Order $order, $usedPoints)
    {
        try {
            $this->updateShoppingPointsDiscount($order, $usedPoints);
        } catch (ShoppingPointsException $e) {
            $scheme = $this->shoppingPointsDiscount->getScheme();

            // Hai inserito un numero di punti non valido. Ti ricordiamo
            // che devi utilizzare almento XXX punti fino ad un massimo di YYY.
            $order->errors->add_to_base(
                sprintf(
                    $this->lang->get('shopping_points_checkout_error'),
                    $scheme->getMinPoints(),
                    $scheme->calculateMaxUsableOrderPoints($order)
                )
            );
        }
    }

    /**
     * @param Order $order
     */
    public function updateDiscounts(Order $order)
    {
        $orderDiscounts = $order->getDiscounts();
        $total = $order->products_total;

        foreach ($this->enabledDiscounts->all() as $enabledDiscount) {
            switch ($enabledDiscount) {
                case 'coupon_code':
                    // In ottica futura, potrebbero esserci più coupon applicati allo stesso ordine: quindi ciclo
                    // tutto quello che trovo.
                    foreach ($orderDiscounts as $orderDiscount) {
                        if ($orderDiscount->getDiscount()->getHandlerName() === 'coupon_code') {
                            $total -= $this->applyCouponCodeDiscount($order, $orderDiscount->getDiscount(), $total);
                        }
                    }

                    break;

                case 'order_amount':
                    // Aggiorno semplicemente sul nuovo valore.
                    $total -= $this->updateOrderAmountDiscount($order, $total);
                    break;

                case 'shopping_points':
                    //
                    $total -= $this->updateShoppingPointsDiscount($order, $order->used_shopping_points);
                    break;
            }
        }
    }

    /**
     * @param Order         $order
     * @param OrderDiscount $coupon
     * @param float         $total
     * @return float The discount value
     */
    protected function applyCouponCodeDiscount(Order $order, OrderDiscount $coupon, $total = null)
    {
        try {
            $value = $this->couponCodeDiscount->apply($order, $coupon, $total);

            $this->logger->debug(
                'The coupon has been applied.',
                [
                    'cart'   => $order->getId() ?: 'NULL',
                    'code'   => $coupon->getCode(),
                    'coupon' => $coupon->getId(),
                ]
            );

            return $value;
        } catch (CouponException $e) {
            $this->logger->notice(
                $e->getMessage(),
                [
                    'cart'   => $order->getId() ?: 'NULL',
                    'code'   => $coupon->getCode(),
                    'coupon' => $coupon->getId(),
                ]
            );

            $this->removeCouponCodeDiscount($order);

            $order->errors->add_to_base($e->getMessage());

            return 0;
        }
    }

    /**
     * @param Order $order
     */
    protected function removeCouponCodeDiscount(Order $order)
    {
        $this->couponCodeDiscount->remove($order);
    }

    /**
     * @param Order $order
     * @param int   $total
     * @return float The discount value
     */
    protected function updateOrderAmountDiscount(Order $order, $total = null)
    {
        return $this->orderAmountDiscount->update($order, $total);
    }

    /**
     * @param Order $order
     * @param int   $usedPoints
     * @return float The discount value
     */
    protected function updateShoppingPointsDiscount(Order $order, $usedPoints)
    {
        try {
            return $this->shoppingPointsDiscount->update($order, $usedPoints);
        } catch (ShoppingPointsException $e) {
            $scheme = $this->shoppingPointsDiscount->getScheme();

            // Hai inserito un numero di punti non valido. Ti ricordiamo
            // che devi utilizzare almento XXX punti fino ad un massimo di YYY.
            $order->errors->add_to_base(
                sprintf(
                    $this->lang->get('shopping_points_checkout_error'),
                    $scheme->getMinPoints(),
                    $scheme->calculateMaxUsableOrderPoints($order)
                )
            );

            return 0;
        }
    }
}
