<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Discount\Order;

use Application\Core\Localization;
use Application\Showcase\Model\Order;
use Application\Showcase\Model\OrderDiscount;
use Application\Showcase\Model\OrderDiscountCondition;

/**
 * OrderAmountDiscount.
 */
class OrderAmountDiscount
{
    /**
     * @var Localization
     */
    protected $lang;

    /**
     * @var OrderDiscount
     */
    private $discount;

    /**
     * @param Localization $lang
     */
    public function __construct(Localization $lang)
    {
        $this->lang = $lang;
    }

    /**
     * Aggiorna lo sconto sull’importo dell’ordine e restituisce il valore dello sconto.
     *
     * @param Order $order
     * @param float  $total
     * @return float
     */
    public function update(Order $order, $total = null)
    {
        if ($total === null) {
            $total = $order->products_total;
        }

        $discount = $this->getDiscount();
        $condition = $this->findCondition($total);
        $value = $this->getValue($total, $condition);

        if ($value) {
            $order->updateOrApplyDiscount(
                $discount,
                $this->getName($condition),
                $value
            );
        } else {
            $order->removeDiscount($discount);
        }

        return $value;
    }

    /**
     * @param OrderDiscountCondition|null $condition
     * @return string
     */
    protected function getName(OrderDiscountCondition $condition = null)
    {
        if ($condition === null) {
            return '';
        }

        if ($condition->value_type === '%') {
            return sprintf($this->lang->get('order_amount_discount_label_percent'), $condition->value . '%');
        } else {
            return $this->lang->get('order_amount_discount_label_fixed');
        }
    }

    /**
     * @param float                       $total
     * @param OrderDiscountCondition|null $condition
     * @return float|int
     */
    protected function getValue($total, OrderDiscountCondition $condition = null)
    {
        if ($condition === null) {
            return 0;
        }

        if ($condition->value_type === '%') {
            return $total * $condition->value / 100;
        } else {
            return $condition->value;
        }
    }

    /**
     * @return OrderDiscount
     */
    protected function getDiscount()
    {
        if ($this->discount === null) {
            $this->discount = OrderDiscount::findByHandler('order_amount');

            if ($this->discount === null) {
                $this->discount = OrderDiscount::create([
                    'handler_name' => 'order_amount',
                ]);
            }
        }

        return $this->discount;
    }

    /**
     * @param float $total
     * @return OrderDiscountCondition|null
     */
    protected function findCondition($total)
    {
        foreach ($this->getDiscount()->getConditions() as $condition) {
            if ($condition->condition <= $total) {
                return $condition;
            }
        }

        return null;
    }
}
