<?php

/**
 * Questo file è parte di alpretec.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   @todo
 */

namespace Application\Showcase\Discount\Order\DiscountType;

use Application\Core\Localization;
use Application\Showcase\Discount\Order\OrderDiscountTypeInterface;
use Application\Showcase\Model\Order;
use Application\Showcase\Model\OrderDiscount;
use Application\Showcase\Model\OrderDiscountCondition;

/**
 * OrderAmountDiscountType
 */
final class OrderAmountDiscountType implements OrderDiscountTypeInterface
{
    /**
     * @var Localization
     */
    private $lang;

    /**
     * @var OrderDiscount
     */
    private $discount;

    /**
     * @param Localization $lang
     */
    public function __construct(Localization $lang)
    {
        $this->lang = $lang;
    }

    /**
     * @inheritdoc
     */
    public function applyToOrder(Order $order)
    {
        $discount = $this->getDiscount();
        $condition = $this->findCondition($order->products_total);
        $value = $this->getValue($order->products_total, $condition);

        if ($value) {
            $order->updateOrApplyDiscount(
                $discount,
                $this->getName($condition),
                $value
            );
        }
    }

    /**
     * @inheritdoc
     */
    public function removeFromOrder(Order $order)
    {
        $order->removeDiscount(
            $this->getDiscount()
        );
    }

    /**
     * @inheritdoc
     */
    public function recalculateDiscountValue(Order $order, $total = null)
    {
        if (null === $total) {
            $total = $order->products_total;
        }

        $discount = $this->getDiscount();
        $condition = $this->findCondition($total);
        $value = $this->getValue($total, $condition);

        if ($value) {
            $order->updateOrApplyDiscount(
                $discount,
                $this->getName($condition),
                $value
            );
        } else {
            $order->removeDiscount($discount);
        }

        return $value;
    }

    /**
     * @param OrderDiscountCondition|null $condition
     * @return string
     */
    protected function getName(OrderDiscountCondition $condition = null)
    {
        if ($condition === null) {
            return '';
        }

        if ($condition->value_type === '%') {
            return sprintf($this->lang->get('order_amount_discount_label_percent'), $condition->value . '%');
        } else {
            return $this->lang->get('order_amount_discount_label_fixed');
        }
    }

    /**
     * @param float                       $total
     * @param OrderDiscountCondition|null $condition
     * @return float|int
     */
    protected function getValue($total, OrderDiscountCondition $condition = null)
    {
        if ($condition === null) {
            return 0;
        }

        if ($condition->value_type === '%') {
            return $total * $condition->value / 100;
        } else {
            return $condition->value;
        }
    }

    /**
     * @return OrderDiscount
     */
    protected function getDiscount()
    {
        if (null === $this->discount) {
            $this->discount = OrderDiscount::findByHandler('order_amount');

            if (null === $this->discount) {
                $this->discount = OrderDiscount::create([
                    'handler_name' => 'order_amount',
                ]);
            }
        }

        return $this->discount;
    }

    /**
     * @param float $total
     * @return OrderDiscountCondition|null
     */
    protected function findCondition($total)
    {
        foreach ($this->getDiscount()->getConditions() as $condition) {
            if ($condition->condition <= $total) {
                return $condition;
            }
        }

        return null;
    }
}
