<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Discount\Order;

use Application\Cms\Model\Manager\NodeTypeManager;
use Application\Cms\Model\NodeTerm;
use Application\Core\Localization;
use Application\Core\Model\Account;
use Application\Core\Model\GroupUser;
use Application\Core\User;
use Application\Newsletter\Filter\FilterRequest;
use Application\Newsletter\Filter\GroupFilter;
use Application\Newsletter\Filter\RoleFilter;
use Application\Newsletter\Filter\UserFilter;
use Application\Showcase\Controller\NodeController;
use Application\Showcase\Exception\CouponException;
use Application\Showcase\Model\Order;
use Application\Showcase\Model\OrderDiscount;
use Application\Showcase\Model\OrderOrderDiscount;
use Application\Showcase\Model\OrderRow;
use Pongho\Utilities\DateTime;

/**
 * CouponCodeDiscount.
 */
class CouponCodeDiscount
{
    /**
     * @var Localization
     */
    protected $lang;

    /**
     * @var User
     */
    protected $user;

    /**
     * @var NodeTypeManager
     */
    protected $nodeTypeManager;

    /**
     * @var OrderRow[][]
     */
    protected $_discountableOrderRows = [];

    /**
     * @param Localization    $lang
     * @param User            $user
     * @param NodeTypeManager $nodeTypeManager
     */
    public function __construct(Localization $lang, User $user, NodeTypeManager $nodeTypeManager)
    {
        $this->lang = $lang;
        $this->user = $user;
        $this->nodeTypeManager = $nodeTypeManager;
    }

    /**
     * Applica un coupon e restituisce il valore dello sconto applicato.
     *
     * @param Order         $order
     * @param OrderDiscount $coupon
     * @param float         $total
     * @return float
     */
    public function apply(Order $order, OrderDiscount $coupon, $total = null)
    {
        if (!$this->isApplicable($order, $coupon)) {
            $this->remove($order);

            return 0;
        }

        $discountableTotal = 0;
        foreach ($this->findDiscountableRows($order, $coupon) as $row) {
            $discountableTotal += $row->total;
        }

        if ($total === null || $discountableTotal !== $order->products_total) {
            $total = $discountableTotal;
        }

        $value = $this->getValue($total, $coupon);

        $order->updateOrApplyDiscount(
            $coupon,
            $this->getName($coupon),
            $value
        );

        return $value;
    }

    /**
     * @param Order $order
     * @return bool
     */
    public function has(Order $order)
    {
        foreach ($order->getDiscounts() as $orderDiscount) {
            if ($orderDiscount->getDiscount()->getHandlerName() === 'coupon_code') {
                return true;
            }
        }

        return false;
    }

    /**
     * @param Order $order
     */
    public function remove(Order $order)
    {
        foreach ($order->getDiscounts() as $orderDiscount) {
            if ($orderDiscount->getDiscount()->getHandlerName() === 'coupon_code') {
                $order->removeDiscount($orderDiscount->getDiscount());
            }
        }
    }

    /**
     * Controlla se il buono può essere utilizzato.
     *
     * Il buono sconto può essere utilizzato:
     *
     * - entro la scadenza prestabilita
     * - solo su ordini che raggiungano un importo minimo
     * - una sola volta per utente
     *
     * @param Order         $order
     * @param OrderDiscount $coupon
     * @throws CouponException
     * @return bool
     */
    protected function isApplicable(Order $order, OrderDiscount $coupon)
    {
        if ($order->customerIsGuest()) {
            return false;
        }

        $this->checkDateCondition($coupon);
        $this->checkOrderAmountCondition($order, $coupon);
        $this->checkIfAlreadyUsed($order, $coupon);
        $this->checkUserCondition($order, $coupon);
        $this->checkProductsCondition($order, $coupon);

        return true;
    }

    /**
     * @param OrderDiscount $coupon
     * @return string
     */
    protected function getName(OrderDiscount $coupon)
    {
        if ($coupon->condition->value_type === '%') {
            return sprintf($this->lang->get('coupon_label_percent'), $coupon->condition->value);
        } else {
            return $this->lang->get('coupon_label_fixed');
        }
    }

    /**
     * @param float         $total
     * @param OrderDiscount $coupon
     * @return float
     */
    protected function getValue($total, OrderDiscount $coupon)
    {
        if ($coupon->condition->value_type === '%') {
            return $total * $coupon->condition->value / 100;
        } else {
            return $coupon->condition->value;
        }
    }

    /**
     * Check if the coupon is valid or expired.
     *
     * @param OrderDiscount $coupon
     * @throws CouponException
     */
    protected function checkDateCondition(OrderDiscount $coupon)
    {
        $today = new DateTime();

        if ($coupon->getValidFrom()) {
            $validFrom = clone $coupon->getValidFrom();
        } else {
            $validFrom = $today;
        }

        $expireOn = clone $coupon->getExpireOn();

        $validFrom->setTime(0, 0, 0);
        $expireOn->setTime(23, 59, 59);

        if ($today < $validFrom) {
            throw new CouponException($this->lang->get('coupon_error_not_valid'));
        }

        if ($today > $expireOn) {
            throw new CouponException($this->lang->get('coupon_error_expired'));
        }
    }

    /**
     * @param Order         $order
     * @param OrderDiscount $coupon
     * @throws CouponException
     */
    protected function checkOrderAmountCondition(Order $order, OrderDiscount $coupon)
    {
        if ($order->productsTotal() < $coupon->condition->condition) {
            throw new CouponException(sprintf(
                $this->lang->get('coupon_error_min_price'),
                $coupon->condition->formatCondition()
            ));
        }
    }

    /**
     * @param Order         $order
     * @param OrderDiscount $coupon
     * @throws CouponException
     */
    protected function checkIfAlreadyUsed(Order $order, OrderDiscount $coupon)
    {
        $options = [
            'joins'      => 'INNER JOIN ' . OrderOrderDiscount::tableName() . ' AS rel ON rel.order_id = `from`.id',
            'conditions' => [
                "`from`.customer_id = :customer AND rel.discount_id = :discount AND `from`.id <> :order AND `from`.status IN ('" . implode("', '", Order::$FINALIZED_STATUSES) . "')",
                'customer' => $this->user->getAccount()->getId(),
                'discount' => $coupon->id,
                'order'    => $order->id,
            ]
        ];

        if (Order::count($options)) {
            throw new CouponException($this->lang->get('coupon_error_already_used'));
        }
    }

    /**
     * Check if the customer can use this coupon
     *
     * @param Order         $order
     * @param OrderDiscount $coupon
     * @throws CouponException
     */
    protected function checkUserCondition(Order $order, OrderDiscount $coupon)
    {
        $userRelMode = $coupon->getSetting('user_rel_mode', 'all');

        switch ($userRelMode) {
            case 'all':
                // Ok, do nothing
                return;

            case 'user':
                $userIds = $coupon->getSetting('user_user', []);

                if (!is_array($userIds)) {
                    $userIds = [];
                }

                if (!in_array($order->customer_id, $userIds)) {
                    throw new CouponException($this->lang->get('coupon_error_not_valid'));
                }

                return;

            case 'group':
                $groupIds = $coupon->getSetting('user_group', []);

                if (!is_array($groupIds)) {
                    $groupIds = [];
                }

                if (empty($groupIds)) {
                    throw new CouponException($this->lang->get('coupon_error_not_valid'));
                }

                $groups = implode(', ', array_map('intval', $groupIds));

                $total = Account::count([
                    'joins' => 'INNER JOIN ' . GroupUser::tableName() . ' AS gu ON gu.user_id = `from`.id',
                    'conditions' => [
                        '`from`.id = :customer AND gu.group_id IN (' . $groups . ')',
                        'customer' => $order->customer_id,
                    ]
                ]);

                if (!$total) {
                    throw new CouponException($this->lang->get('coupon_error_not_valid'));
                }

                return;

            case 'role':
                $roleIds = $coupon->getSetting('user_role', []);

                if (!is_array($roleIds)) {
                    $roleIds = [];
                }

                if (!in_array($order->customer->getRoleId(), $roleIds)) {
                    throw new CouponException($this->lang->get('coupon_error_not_valid'));
                }

                return;
        }
    }

    /**
     * @param Order         $order
     * @param OrderDiscount $coupon
     * @throws CouponException
     */
    protected function checkProductsCondition(Order $order, OrderDiscount $coupon)
    {
        $discountableRows = $this->findDiscountableRows($order, $coupon);

        if (empty($discountableRows)) {
            throw new CouponException($this->lang->get('coupon_error_not_valid'));
        }
    }

    /**
     * @param Order         $order
     * @param OrderDiscount $coupon
     * @return OrderRow[]
     */
    protected function findDiscountableRows(Order $order, OrderDiscount $coupon)
    {
        $key = $order->getId() . '-' . $coupon->getId();

        if (!isset($this->_discountableOrderRows[$key])) {
            $this->_discountableOrderRows[$key] = [];

            /** @var \Application\Core\Model\Account $account */
            $account = $this->user->getAccount();
            $languageId = $account->getLanguageId();

            $nodeTypes = $this->nodeTypeManager->findAllNodeTypesByController(NodeController::class, $languageId);

            foreach ($nodeTypes as $nodeType) {
                $nodeRelMode = $coupon->getSetting('node_' . $nodeType->node_type . '_rel_mode', 'all');

                if ($nodeRelMode === 'all') {
                    foreach ($order->getRows() as $row) {
                        if ($row->getProduct()->getNodeType() === $nodeType->node_type) {
                            $this->_discountableOrderRows[$key][$row->id] = $row;
                        }
                    }
                } elseif ($nodeRelMode === 'node') {
                    $discountNodeIds = $coupon->getSetting('node_' . $nodeType->node_type . '_node', []);

                    if (!is_array($discountNodeIds)) {
                        $discountNodeIds = [];
                    }

                    if ($discountNodeIds) {
                        foreach ($order->getRows() as $row) {
                            if (in_array($row->product_id, $discountNodeIds)) {
                                $this->_discountableOrderRows[$key][$row->id] = $row;
                            }
                        }
                    }
                } else {
                    $discountTermIds = [];
                    foreach ($nodeType->getTaxonomies() as $taxonomy) {
                        if ($nodeRelMode === 'tax_' . $taxonomy->name) {
                            $fieldName = 'node_' . $nodeType->node_type . '_tax_' . $taxonomy->name;
                            $discountTermIds = $coupon->getSetting($fieldName, []);

                            break;
                        }
                    }

                    if (is_array($discountTermIds) && !empty($discountTermIds)) {
                        $discountTermIds = array_map('intval', $discountTermIds);

                        $options = [
                            'joins'      => 'INNER JOIN ' . NodeTerm::tableName() . ' AS nt ON nt.node_id = `from`.product_id',
                            'conditions' => [
                                '`from`.order_id = :order AND nt.term_id IN (' . implode(', ', $discountTermIds) . ')',
                                'order' => $order->getId(),
                            ],
                        ];

                        /** @var OrderRow $row */
                        foreach (OrderRow::all($options) as $row) {
                            $this->_discountableOrderRows[$key][$row->id] = $row;
                        }
                    }
                }
            }
        }

        return $this->_discountableOrderRows[$key];
    }

    /**
     * @param OrderDiscount $coupon
     * @return FilterRequest
     */
    public function buildRecipientsFilter(OrderDiscount $coupon)
    {
        if (!class_exists(FilterRequest::class)) {
            throw new \BadMethodCallException(
                'The class "' . FilterRequest::class . '" does not exist. Maybe the Newsletter app is not installed or enabled?'
            );
        }

        $filter = new FilterRequest();

        switch ($coupon->getSetting('user_rel_mode', 'all')) {
            case 'user':
                $userIds = $coupon->getSetting('user_user', []);

                if (!is_array($userIds)) {
                    $userIds = [];
                }

                foreach ($userIds as $userId) {
                    $filter->addFilter([
                        'type'      => 'user',
                        'user_cond' => UserFilter::IS_EQUAL_TO,
                        'user_id'   => $userId,
                    ]);
                }

                break;

            case 'group':
                $groupIds = $coupon->getSetting('user_group', []);

                if (!is_array($groupIds)) {
                    $groupIds = [];
                }

                foreach ($groupIds as $groupId) {
                    $filter->addFilter([
                        'type'      => 'group',
                        'user_cond' => GroupFilter::IS_EQUAL_TO,
                        'group_id'   => $groupId,
                    ]);
                }

                break;

            case 'role':
                $roleIds = $coupon->getSetting('user_role', []);

                if (!is_array($roleIds)) {
                    $roleIds = [];
                }

                foreach ($roleIds as $roleId) {
                    $filter->addFilter([
                        'type'      => 'role',
                        'user_cond' => RoleFilter::IS_EQUAL_TO,
                        'role_id'   => $roleId,
                    ]);
                }

                break;

            default: // all
                break;
        }

        return $filter;
    }
}
