<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\DependencyInjection;

use Application\Admin\Panels\PanelsManager;
use Application\Admin\Widgets\PanelWidgetsCollection;
use Application\Admin\Widgets\WidgetsCollection;
use Application\Showcase\Stats\StatsFactory;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

/**
 * StatsServiceProvider
 */
class StatsServiceProvider implements ServiceProviderInterface
{
    /**
     * {@inheritdoc}
     */
    public function register(Container $container)
    {
        $container->share('showcase_stats_panels_manager', [$this, 'getPanelsManager']);

        $container->share('showcase_stats_sold_panel', [$this, 'getSoldPanel']);
        $container->share('showcase_stats_sold_widgets_collection', [$this, 'getSoldWidgetsCollection']);

        $container->share('showcase_stats_ordered_panel', [$this, 'getOrderedPanel']);
        $container->share('showcase_stats_ordered_widgets_collection', [$this, 'getOrderedWidgetsCollection']);
    }

    /**
     * @param Container $container
     * @return PanelsManager
     */
    public function getPanelsManager(Container $container)
    {
        $panels_manager = new PanelsManager(true, $container->getParameter('path'), $container->get('pongho_url'));

        /** @var \Application\Core\User $user */
        $user = $container->get('current_user');

        if ($user->hasPermit('shop.admin.stats.sold')) {
            $panels_manager->add($container->get('showcase_stats_sold_panel'));
        }

        if ($user->hasPermit('shop.admin.stats.ordered')) {
            $panels_manager->add($container->get('showcase_stats_ordered_panel'));
        }

        return $panels_manager;
    }

    /**
     * @param Container $container
     * @return PanelWidgetsCollection
     */
    public function getSoldPanel(Container $container)
    {
        /** @var \Application\Core\I18n\Translation\Translator $translator */
        $translator = $container->get('translator');

        $panel = new PanelWidgetsCollection(
            'sold',
            $translator->trans('Sold'),
            $container->get('showcase_stats_sold_widgets_collection')
        );

        return $panel;
    }

    /**
     * @param Container $container
     * @return WidgetsCollection
     */
    public function getSoldWidgetsCollection(Container $container)
    {
        /** @var \Application\Core\Model\LanguageSite $language_site */
        $language_site = $container->get('language_site');

        $coll = new WidgetsCollection();

        $widgetStatsFactory = new StatsFactory(
            $container->get('translator'),
            $container->get('widget_google_chart_factory')
        );

        $coll->add($widgetStatsFactory->createLastMonthSold());
        $coll->add($widgetStatsFactory->createLastQuarterSold());
        $coll->add($widgetStatsFactory->createLastHalfYearSold());
        $coll->add($widgetStatsFactory->createLastYearSold());
        $coll->add($widgetStatsFactory->createLastMonthTopSoldProducts($language_site->language_id));
        $coll->add($widgetStatsFactory->createLastYearTopSoldProducts($language_site->language_id));

        return $coll;
    }

    /**
     * @param Container $container
     * @return PanelWidgetsCollection
     */
    public function getOrderedPanel(Container $container)
    {
        /** @var \Application\Core\I18n\Translation\Translator $translator */
        $translator = $container->get('translator');

        $panel = new PanelWidgetsCollection(
            'ordered',
            $translator->trans('Ordered'),
            $container->get('showcase_stats_ordered_widgets_collection')
        );

        return $panel;
    }

    /**
     * @param Container $container
     * @return WidgetsCollection
     */
    public function getOrderedWidgetsCollection(Container $container)
    {
        /** @var \Application\Core\Model\LanguageSite $language_site */
        $language_site = $container->get('language_site');

        $coll = new WidgetsCollection();

        $widgetStatsFactory = new StatsFactory(
            $container->get('translator'),
            $container->get('widget_google_chart_factory')
        );

        $coll->add($widgetStatsFactory->createLastMonthOrdered());
        $coll->add($widgetStatsFactory->createLastQuarterOrdered());
        $coll->add($widgetStatsFactory->createLastHalfYearOrdered());
        $coll->add($widgetStatsFactory->createLastYearOrdered());
        $coll->add($widgetStatsFactory->createLastMonthTopOrderedProducts($language_site->language_id));
        $coll->add($widgetStatsFactory->createLastYearTopOrderedProducts($language_site->language_id));

        return $coll;
    }
}
