<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\DependencyInjection;

use Application\Cms\Search\SearchableFields;
use Application\Core\I18n\Translation\Translator;
use Application\Core\Model\Site;
use Application\Showcase\Controller\Admin\Orders\EditPanel\CancelOrderPanel;
use Application\Showcase\Controller\Admin\Orders\EditPanel\ConfirmOrderPaymentPanel;
use Application\Showcase\Controller\Admin\Orders\EditPanel\ProcessOrderPanel;
use Application\Showcase\Discount\DiscountCalculator;
use Application\Showcase\Domain\Order\Command\CancelOrderCommandFactory;
use Application\Showcase\Domain\Order\Command\ConfirmOrderPaymentCommandFactory;
use Application\Showcase\Domain\Order\Command\ProcessOrderCommandFactory;
use Application\Showcase\Domain\Order\Command\ProcessOrderNotifier;
use Application\Showcase\Discount\Order\CouponCodeDiscount;
use Application\Showcase\Discount\Order\OrderAmountDiscount;
use Application\Showcase\Discount\OrderDiscounts;
use Application\Showcase\Discount\Order\ShoppingPointsDiscount;
use Application\Showcase\Discount\OrderDiscountsFacade;
use Application\Showcase\Export\DataExporter\OrdersExporter;
use Application\Showcase\Form\Builder\LoginFormBuilder;
use Application\Showcase\Form\Subject\LoginSubject;
use Application\Showcase\Installer\NodeTypeConverter;
use Application\Showcase\Model\Manager\ProductManager;
use Application\Showcase\Model\Order;
use Application\Showcase\ShoppingPoints;
use Application\Showcase\Utilities\CheckoutBreadcrumbBuilder;
use Application\Showcase\Utilities\CheckoutOptions;
use Application\Showcase\Utilities\Config;
use Application\Showcase\Utilities\OrderNotification;
use Application\Showcase\Utilities\Taxation;
use Monolog\Formatter\HtmlFormatter;
use Monolog\Handler\BufferHandler;
use Monolog\Handler\FingersCrossedHandler;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Handler\SwiftMailerHandler;
use Monolog\Logger;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;
use Pongho\EventDispatcher\Event;
use Pongho\Template\View;

/**
 * Class ShowcaseServiceProvider
 */
class ShowcaseServiceProvider implements ServiceProviderInterface
{
    /**
     * @param Container $container
     */
    public function register(Container $container)
    {
        $container->setParameter('checkout_breadcrumb_builder_class', CheckoutBreadcrumbBuilder::class);
        $container->setParameter('shop_taxation_class', Taxation::class);
        $container->setParameter('shop_coupon_discount_class', CouponCodeDiscount::class);
        $container->setParameter('shop_order_amount_discount_class', OrderAmountDiscount::class);
        $container->setParameter('shop_login_form_subject_class', LoginSubject::class);
        $container->setParameter('shop_login_form_builder_class', LoginFormBuilder::class);

        $container->share('product_manager', [$this, 'getProductManagerService']);

        $container->share('shop_logger_message', [$this, 'getLoggerMessageService']);
        $container->share('shop_logger', [$this, 'getLoggerService']);

        $container->share('cart', [$this, 'getCartService']);
        $container->share('checkout_breadcrumb', [$this, 'getCheckoutBreadcrumbService']);
        $container->share('shop_checkout_options', [$this, 'getCheckoutOptionsService']);
        $container->share('shop_taxation', [$this, 'getTaxationService']);
        $container->share('shop_login_form_subject', [$this, 'getLoginFormSubjectService']);
        $container->share('shop_login_form_builder', [$this, 'getLoginFormBuilderService']);
        $container->share('shop_order_notification', [$this, 'getOrderNotificationService']);
        $container->share('showcase_node_type_converter', [$this, 'getNodeTypeConverterService']);

        $container->share('admin.orders.exporter', function (Container $container) {
            return new OrdersExporter(
                $container->get('connection'),
                $container->get('translator'),
                $container->get('language')
            );
        });

        // Sconti
        $container->share('shop_order_discounts', [$this, 'getOrderDiscounts']);
        $container->share('shop_order_discounts_facade', [$this, 'getOrderDiscountsFacade']);
        $container->share('shop_coupon_discount', [$this, 'getCouponDiscount']);
        $container->share('shop_order_amount_discount', [$this, 'getOrderAmountDiscount']);
        $container->share('shop_shopping_points_scheme', [$this, 'getShoppingPointsScheme']);
        $container->share('shop_shopping_points_discount', [$this, 'getShoppingPointsDiscount']);

        $container->share('shop_discount_calculator', function () {
            return new DiscountCalculator();
        });

        $container->extend('site', [$this, 'siteExtension']);
        $container->extend('main_site', [$this, 'siteExtension']);

        $container->extend(
            'translator',
            function(Translator $translator, Container $container) {
                $translator->addResource('pongho_app', 'Showcase', $container->get('current_user')->language->culture);
            }
        );

        $container->extend(
            'global_view_vars',
            function(array $vars, Container $container) {
                $vars['cart'] = $container->get('cart');
                $vars['shopping_points_scheme'] = $container->get('shop_shopping_points_scheme');
                $vars['order_discounts'] = $container->get('shop_order_discounts');

                return $vars;
            }
        );

        $container->extend(
            'search_searchable_fields',
            function(SearchableFields $fields) {
                $fields->addField('code');

                return $fields;
            }
        );

        /*
         * Admin / Orders
         */
        $container->share('shop.admin.orders.cancel_order_panel', function (Container $container) {
            return new CancelOrderPanel(
                $container->get('translator'),
                $container->get('pongho_url'),
                $container->get('nodetype_helper'),
                $container->get('shop.domain.orders.commands.cancel_order_factory')
            );
        });

        $container->share('shop.domain.orders.commands.cancel_order_factory', function (Container $container) {
            return new CancelOrderCommandFactory(
                $container->get('translator')
            );
        });

        $container->share('shop.admin.orders.confirm_order_payment_panel', function (Container $container) {
            return new ConfirmOrderPaymentPanel(
                $container->get('translator'),
                $container->get('pongho_url'),
                $container->get('nodetype_helper'),
                $container->get('shop.domain.orders.commands.confirm_order_payment_factory')
            );
        });

        $container->share('shop.domain.orders.commands.confirm_order_payment_factory', function (Container $container) {
            return new ConfirmOrderPaymentCommandFactory(
                $container->get('translator')
            );
        });

        $container->share('shop.admin.orders.process_order_panel', function (Container $container) {
            return new ProcessOrderPanel(
                $container->get('translator'),
                $container->get('pongho_url'),
                $container->get('nodetype_helper'),
                $container->get('shop.domain.orders.commands.process_order_factory')
            );
        });

        $container->share('shop.domain.orders.commands.process_order_factory', function (Container $container) {
            return new ProcessOrderCommandFactory(
                $container->get('shop_order_discounts'),
                $container->get('shop.domain.orders.commands.process_order_notifier'),
                $container->get('translator')
            );
        });

        $container->share('shop.domain.orders.commands.process_order_notifier', function (Container $container) {
            return new ProcessOrderNotifier(
                $container->get('mailer'),
                $container->get('site'),
                $container->get('shop.domain.orders.commands.process_order_notifier.view'),
                $container->get('language')
            );
        });

        $container->share('shop.domain.orders.commands.process_order_notifier.view', function (Container $container) {
            /** @var Site $site */
            $site = $container->get('site');

            $view = new View($site->getThemesPath('email/shop_order_processed.php'));
            $view->assignVars($container->get('global_view_vars'));

            return $view;
        });
    }

    /**
     * @return \Application\Showcase\Model\Manager\ProductManagerInterface
     */
    public function getProductManagerService()
    {
        return new ProductManager();
    }

    /**
     * @param Site $site
     */
    public function siteExtension(Site $site)
    {
        $site->setDefaultOptions(Config::getDefault($site));
    }

    /**
     * Restituisce il messaggio email per gli errori critici dei log
     *
     * @param Container $container
     * @return \Swift_Message
     */
    public function getLoggerMessageService(Container $container)
    {
        /** @var \Application\Core\Model\Site $site */
        $site = $container->get('site');

        $message = new \Swift_Message();

        $message
            ->addFrom($container->getParameter('logger.mail.from', 'noreply@metaline.it'))
            ->addTo($container->getParameter('logger.mail.to', 'log@metaline.it'))
            ->setSubject(sprintf('Log "pongho_shop" from site "%s"', $site->getName()))
            ->setContentType('text/html');

        return $message;
    }

    /**
     * Restituisce il gestore dei log
     *
     * @param Container $container
     * @return \Monolog\Logger
     */
    public function getLoggerService(Container $container)
    {
        $logger_name = 'pongho_shop';
        $logger = new Logger($logger_name);

        /** @var \Application\Core\Model\Site $site */
        $site = $container->get('site');

        // Salvo le informazioni di log su file
        $logger->pushHandler(
            new RotatingFileHandler(
                $site->getContentPath(
                    $container->getParameter('logger.path', '/log') . '/' . $logger_name . '.log'
                ),
                $container->getParameter('logger.maxfiles', 90),
                Logger::DEBUG
            )
        );

        $swift_mailer_handler = new SwiftMailerHandler(
            new \Swift_Mailer($container->get('mailer_transport')),
            $container->get('shop_logger_message'),
            Logger::DEBUG
        );

        $swift_mailer_handler->setFormatter(new HtmlFormatter());

        // Mail per gli errori critici
        $logger->pushHandler(
            new FingersCrossedHandler(
                new BufferHandler(
                    $swift_mailer_handler
                ), Logger::ERROR
            )
        );

        $logger->pushProcessor(function (array $record) use ($container) {
            /** @var \Application\Core\User $user */
            $user = $container->get('current_user');

            $record['extra'] = array_merge(
                $record['extra'],
                ['user_id' => $user->getAccount()->getId()]
            );

            return $record;
        });

        return $logger;
    }

    /**
     * Restituisce il carrello.
     *
     * @param Container $container
     * @return Order
     */
    public function getCartService(Container $container)
    {
        /**
         * @var Order $cart
         */

        // Permetto di sovrascrivere il carrello
        $event = new Event(null, 'shop.get_cart');
        $container->get('event_dispatcher')->notifyUntil($event);

        if ($event->isProcessed()) {
            $cart = $event->getReturnValue();
        } else {
            // Ricavo il carrello standard...
            $cart = Order::getCart(
                $container->get('current_user'),
                $container->get('session'),
                $container->get('site')
            );

            // ...e se non c’è lo creo.
            if ($cart == null) {
                $cart = Order::createCart(
                    $container->get('current_user'),
                    $container->get('session'),
                    $container->get('site'),
                    $container->get('request')
                );
            }

            /** @var \Application\Showcase\Utilities\CheckoutOptions $checkout_options */
            $checkout_options = $container->get('shop_checkout_options');

            $cart->setCheckoutOptions($checkout_options);

            // Filtro
            $event = new Event(null, 'shop.filter_cart');
            $dispatcher = $container->get('event_dispatcher');

            $cart = $dispatcher->filter($event, $cart)->getReturnValue();
        }

        /** @var \Application\Showcase\Discount\OrderDiscounts $orderDiscounts */
        $orderDiscounts = $container->get('shop_order_discounts');

        if ($orderDiscounts->isEnabled('order_amount')) {
            /** @var \Application\Showcase\Discount\OrderDiscountsFacade $orderDiscountsFacade */
            $orderDiscountsFacade = $container->get('shop_order_discounts_facade');
            $orderDiscountsFacade->updateOrderAmount($cart);
        }

        return $cart;
    }

    /**
     * getCheckoutBreadcrumbService
     *
     * @param Container $container
     * @return \Application\Showcase\Utilities\CheckoutBreadcrumbBuilder
     */
    public function getCheckoutBreadcrumbService(Container $container)
    {
        $class = $container->getParameter('checkout_breadcrumb_builder_class');

        return new $class(
            $container->get('language'),
            $container->get('request')
        );
    }

    /**
     * @return CheckoutOptions
     */
    public function getCheckoutOptionsService()
    {
        return new CheckoutOptions();
    }

    /**
     * getTaxationService
     *
     * @param Container $container
     * @return \Application\Showcase\Utilities\Taxation
     */
    public function getTaxationService(Container $container)
    {
        $class = $container->getParameter('shop_taxation_class');

        return new $class($container->get('site'));
    }

    /**
     * getLoginFormSubjectService
     *
     * @param Container $container
     * @return \Application\Showcase\Form\Subject\LoginSubject
     */
    public function getLoginFormSubjectService(Container $container)
    {
        $class = $container->getParameter('shop_login_form_subject_class');

        return new $class(
            $container->get('cart'),
            $container->get('current_user'),
            $container->get('language'),
            $container->get('mailer'),
            $container->get('site'),
            $container->get('application_manager'),
            $container->get('address_validator_factory'),
            $container->get('address_renderer_factory')
        );
    }

    /**
     * getLoginFormBuilderService
     *
     * @param Container $container
     * @return \Application\Showcase\Form\Builder\LoginFormBuilder
     */
    public function getLoginFormBuilderService(Container $container)
    {
        $class = $container->getParameter('shop_login_form_builder_class');

        return new $class(
            $container->get('shop_login_form_subject'),
            $container->get('language'),
            $container->get('site'),
            $container->get('application_manager')
        );
    }

    /**
     * getOrderNotificationService
     *
     * @param Container $container
     * @return OrderNotification
     */
    public function getOrderNotificationService(Container $container)
    {
        return new OrderNotification(
            $container->get('language'),
            $container->get('mailer'),
            $container->get('global_view_vars'),
            $container->get('shop_logger')
        );
    }

    /**
     * getNodeTypeConverterService
     *
     * @param Container $container
     * @return NodeTypeConverter
     */
    public function getNodeTypeConverterService(Container $container)
    {
        return new NodeTypeConverter(
            $container->get('taxonomy_builder')
        );
    }

    public function getOrderDiscounts(Container $container)
    {
        /** @var \Application\Core\Model\Site $site */
        $site = $container->get('site');

        return new OrderDiscounts(
            $site->getOption('order_discounts')
        );
    }

    /**
     * @param Container $container
     * @return OrderDiscountsFacade
     */
    public function getOrderDiscountsFacade(Container $container)
    {
        return new OrderDiscountsFacade(
            $container->get('shop_coupon_discount'),
            $container->get('shop_order_amount_discount'),
            $container->get('shop_shopping_points_discount'),
            $container->get('shop_order_discounts'),
            $container->get('language'),
            $container->get('logger')
        );
    }

    /**
     * @param Container $container
     * @return \Application\Showcase\Discount\Order\CouponCodeDiscount
     */
    public function getCouponDiscount(Container $container)
    {
        $class = $container->getParameter('shop_coupon_discount_class');

        return new $class(
            $container->get('language'),
            $container->get('current_user'),
            $container->get('node_type_manager')
        );
    }

    /**
     * @param Container $container
     * @return \Application\Showcase\Discount\Order\OrderAmountDiscount
     */
    public function getOrderAmountDiscount(Container $container)
    {
        $class = $container->getParameter('shop_order_amount_discount_class');

        return new $class(
            $container->get('language')
        );
    }

    /**
     * @param Container $container
     * @return ShoppingPoints\Scheme
     */
    public function getShoppingPointsScheme(Container $container)
    {
        /** @var \Application\Core\Model\Site $site */
        $site = $container->get('site');

        return new ShoppingPoints\Scheme(
            $site->getOption('shopping_point_cost'),
            $site->getOption('shopping_point_value'),
            $site->getOption('shopping_point_min_points'),
            $site->getOption('shopping_point_min_total_order'),
            $site->getOption('shopping_point_on_single_product'),
            $site->getOption('shopping_point_calculate_on_subtotal')
        );
    }

    /**
     * @param Container $container
     * @return ShoppingPointsDiscount
     */
    public function getShoppingPointsDiscount(Container $container)
    {
        return new ShoppingPointsDiscount(
            $container->get('language'),
            $container->get('shop_shopping_points_scheme')
        );
    }
}
