<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Controller\Admin;

use Application\Admin\Controller\SiteOptionsController;
use Application\Core\Model\Permit;
use Application\Core\Model\Role;
use Application\Showcase\Model\NodeType;
use Application\Showcase\Model\OrderDiscount;
use Pongho\Http\Exception\HttpUnauthorizedException;

/**
 * Controller per le opzioni del negozio.
 */
class ShopOptionsController extends SiteOptionsController
{
    /**
     * {@inheritdoc}
     */
    protected function boot()
    {
        parent::boot();

        // Potrei aver impostato la risposta con il redirect alla pagina di login.
        if ($this->response) {
            return;
        }

        // Permesso consentito?
        if (!$this->getHelper()->getUser()->isFounder()) {
            throw new HttpUnauthorizedException();
        }
    }

    /**
     * Azione `index`.
     *
     * @return \Pongho\Http\Response
     */
    public function indexAction()
    {
        $this->handleRequest([
            // Email ordini
            'orders_email'                         => 'string',

            // Export
            'enable_orders_export'                 => 'boolean',

            // Offerte
            'enable_offers'                        => 'boolean',
            'order_discounts'                      => 'discounts',

            // Raccolta punti
            'shopping_point_cost'                  => 'price',
            'shopping_point_value'                 => 'price',
            'shopping_point_min_points'            => 'integer',
            'shopping_point_min_total_order'       => 'price',
            'shopping_point_on_single_product'     => 'boolean',
            'shopping_point_calculate_on_subtotal' => 'boolean',

            // Tasse
            'enable_tax'                           => 'boolean',
            'tax_value'                            => 'integer',
            'tax_name'                             => 'string',

            // Checkout
            'enable_guest_checkout'                => 'boolean',
            'enable_order_note'                    => 'boolean',

            // Google Shopping
            'enable_google_shopping'               => 'boolean',
        ]);

        if ($this->getRequest()->getMethod() === 'POST') {
            $this->handleOrdersDiscounts();
            $this->handleGoogleShopping();

            return $this->getHelper()->redirectResponse($this->url('/shop-options/'));
        }

        $this->getHelper()->addJavascriptInline($this->getJSOptions());

        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/shop_options_main.php')
            ->assignVars([
                'action_form'     => $this->url('/' . $this->getParameter('path') . '/'),
                'order_discounts' => $this->getDiscountsList(),
            ]);

        return null;
    }

    /**
     * @return array
     */
    private function getDiscountsList()
    {
        $translator = $this->getHelper()->getTranslator();

        $discounts = [
            'coupon_code'     => $translator->trans('Discount with coupon'),
            'order_amount'    => $translator->trans('Discount on amount'),
            'shopping_points' => $translator->trans('Shopping points'),
        ];

        $discountsOrdered = [];
        foreach ($this->getOptionsContainer()->getOption('order_discounts') as $discountName) {
            $discountsOrdered[$discountName] = $discounts[$discountName];
        }

        return array_merge($discountsOrdered, $discounts);
    }

    /**
     * @param array  $post_options
     * @param string $field_name
     * @return array
     */
    protected function getFieldDiscounts(array $post_options, $field_name)
    {
        $rawDiscounts = isset($post_options[$field_name]) ? $post_options[$field_name] : [];
        $discounts = [];

        foreach ($rawDiscounts as $name => $discount) {
            $is_enabled = isset($discount['is_enabled']) ? (bool) $discount['is_enabled'] : false;
            $priority = isset($discount['priority']) ? (int) $discount['priority'] : 0;

            if ($is_enabled) {
                $discounts[$priority] = $name;
            }
        }

        ksort($discounts);

        return array_values($discounts);
    }

    /**
     * Gestisce gli sconti
     */
    private function handleOrdersDiscounts()
    {
        $discounts = $this->getOptionsContainer()->getOption('order_discounts');

        if (in_array('coupon_code', $discounts)) {
            $this->enableCouponCode();
        } else {
            $this->disableCouponCode();
        }

        if (in_array('order_amount', $discounts)) {
            $this->enableOrderAmount();
        } else {
            $this->disableOrderAmount();
        }

        if (in_array('shopping_points', $discounts)) {
            $this->enableShoppingPoints();
        } else {
            $this->disableShoppingPoints();
        }
    }

    /**
     * Abilita il permesso per la gestione dei coupon
     */
    private function enableCouponCode()
    {
        $permit = Permit::findOrCreate('shop.admin.coupons', true);

        if (!$permit->is_enabled) {
            $permit->is_enabled = true;
            $permit->save();
        }

        /*
         * Se attivo i buoni sconto, mi aspetto che gli admin e gli utenti
         * con il mio ruolo abbiano il permesso di gestirli.
         */
        $myRole = $this->getHelper()->getUser()->getAccount()->getRole();
        $myRole->addPermit($permit);

        /** @var Role $adminRole */
        $adminRole = Role::find(Role::ADMIN);
        $adminRole->addPermit($permit);
    }

    /**
     * Disabilita il permesso per la gestione dei coupon
     */
    private function disableCouponCode()
    {
        $permit = Permit::findByKey('shop.admin.coupons');

        if ($permit) {
            $permit->is_enabled = false;
            $permit->save();
        }
    }

    /**
     * Abilita il permesso per la gestione dei discount sul totale degli ordini
     */
    private function enableOrderAmount()
    {
        $permit = Permit::findOrCreate('shop.admin.orders_amount_discount', true);

        if (!$permit->is_enabled) {
            $permit->is_enabled = true;
            $permit->save();
        }
    }

    /**
     * Disabilita il permesso per la gestione dei discount sul totale dell'ordine
     */
    private function disableOrderAmount()
    {
        $permit = Permit::findByKey('shop.admin.orders_amount_discount');

        if ($permit) {
            $permit->is_enabled = false;
            $permit->save();
        }
    }

    /**
     * Abilita la raccolta punti.
     */
    private function enableShoppingPoints()
    {
        $discount = OrderDiscount::findByHandler('shopping_points');

        if (!$discount) {
            OrderDiscount::create([
                'handler_name' => 'shopping_points',
            ]);
        }
    }

    /**
     * Disabilita la raccolta punti.
     */
    private function disableShoppingPoints()
    {
        $discount = OrderDiscount::findByHandler('shopping_points');

        if ($discount) {
            $discount->delete();
        }
    }

    /**
     * Abilita il permesso per la gestione del feed di Google Shopping
     */
    private function handleGoogleShopping()
    {
        $is_enabled = $this->getHelper()->getSite()->getOption('enable_google_shopping');

        if ($is_enabled) {
            $permit = Permit::findOrCreate('shop.admin.google_shopping');

            if (!$permit->is_enabled) {
                $permit->is_enabled = true;
                $permit->save();
            }
        } else {
            $permit = Permit::findByKey('shop.admin.google_shopping');

            if ($permit) {
                $permit->is_enabled = false;
                $permit->save();
            }
        }
    }

    /**
     * @return \Application\Cms\Utilities\NodeTypeHelper
     */
    public function getHelper()
    {
        /** @var \Application\Cms\Utilities\NodeTypeHelper $helper */
        $helper = $this->container->get('nodetype_helper');

        $helper->setModuleSiteClass(NodeType::class);

        return $helper;
    }

    /**
     * @return string
     */
    protected function getJSOptions()
    {
        return <<<JS
(function ($) {
    "use strict";
    
    var shoppingPoints = $('#order-discount-shopping_points'),
        shoppingPointsSlave = $('#shopping-points'),
        coupons = $('#order-discount-coupon_code'),
        couponsSlave = $('#coupons');

    function toggle(master, slave) {
        slave.toggle(master.is(':checked'));
    }

    function toggleShoppingPoints() {
        toggle(shoppingPoints, shoppingPointsSlave);
    }
    
    function toggleCoupons() {
        toggle(coupons, couponsSlave);
    }

    shoppingPoints.on('change', toggleShoppingPoints);
    toggleShoppingPoints();

    coupons.on('change', toggleCoupons);
    toggleCoupons();
}(jQuery));
JS;
    }
}
