<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Controller\Admin;

use Application\Admin\Controller\CrudController;
use Application\Admin\Form\TextareaField;
use Application\Showcase\Model\NodeType;
use Application\Showcase\Model\Shipping;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\Exception\HttpUnauthorizedException;

/**
 * Controller per la gestione dei metodi di trasporto in admin.
 */
class ShippingsController extends CrudController
{
    /**
     * {@inheritdoc}
     */
    protected function boot()
    {
        parent::boot();

        // Potrei aver impostato la risposta con il redirect alla pagina di login.
        if ($this->response) {
            return;
        }

        // Permesso consentito?
        if (!$this->getHelper()->getUser()->hasPermit('shop.admin.shippings.manage')) {
            throw new HttpUnauthorizedException();
        }
    }

    /**
     * {@inheritdoc}
     */
    public function displayArchive($page = 1)
    {
        $path = $this->getParameter('path');

        $shippings = Shipping::all();

        foreach ($shippings as $shipping) {
            $actions = [
                ['edit', 'href' => $this->url("/{$path}/edit/$shipping->id/"), 'class' => 'edit'],
            ];

            if ($this->getHelper()->getUser()->hasPermit('shop.admin.shippings.delete')) {
                $actions[] = [
                    'delete',
                    'href'  => $this->url("/{$path}/delete/$shipping->id/"),
                    'class' => 'delete'
                ];
            }

            $shipping->actions = $this->parseActions($actions);
        }

        $this->getHelper()
            ->addCss(pongho_url(
                '/Application/Admin/Resources/views/css/style-legacy.css?v=' . filemtime(
                    __DIR__ . '/../../../Admin/Resources/views/css/style-legacy.css'
                )
            ));

        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/shippings_list.php')
            ->assignVars([
                'rows'    => $shippings,
                'add_url' => $this->url("/{$path}/add/"),
            ]);
    }

    /**
     * {@inheritdoc}
     */
    public function addAction()
    {
        if ($this->getHelper()->getUser()->hasPermit('shop.admin.shippings.add')) {
            return parent::addAction();
        }

        throw new HttpNotFoundException();
    }

    /**
     * {@inheritdoc}
     */
    public function deleteAction()
    {
        if ($this->getHelper()->getUser()->hasPermit('shop.admin.shippings.delete')) {
            return parent::deleteAction();
        }

        throw new HttpNotFoundException();
    }

    /**
     * {@inheritdoc}
     */
    protected function massDelete($row)
    {
        if ($this->getHelper()->getUser()->hasPermit('shop.admin.shippings.delete')) {
            return parent::massDelete($row);
        }

        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return Shipping::class;
    }

    /**
     * {@inheritdoc}
     */
    public function getTitle($model)
    {
        $translator = $this->getHelper()->getTranslator();

        if ($this->getAction() === 'add') {
            return $translator->trans('Add shipping method');
        }

        return $translator->trans('Edit the shipping method “%title%”', ['%title%' => $model->name]);
    }

    /**
     * {@inheritdoc}
     */
    public function getFields($node)
    {
        $fields = [
            ['name'],
            ['description', 'class' => TextareaField::class],
        ];

        return $this->getHelper()->filter($this, 'admin.shippings.filter_fields', $fields);
    }

    /**
     * Callback 'before_set_template'.
     *
     * @param \Application\Showcase\Model\Shipping $shipping
     */
    protected function beforeSetTemplateCallback(Shipping $shipping)
    {
        $this->displayConditions($shipping);
    }

    /**
     * Callback 'after_save'.
     *
     * @param \Application\Showcase\Model\Shipping $shipping
     */
    protected function afterSaveCallback(Shipping $shipping)
    {
        $this->saveConditions($shipping);
    }

    /**
     * Visualizza le condizioni per il calcolo del costo.
     *
     * @param \Application\Showcase\Model\Shipping $shipping
     */
    protected function displayConditions(Shipping $shipping)
    {
        $this->getHelper()->getBodyView()
            ->assignVar('conditions', $shipping->getShippingConditions());
    }

    /**
     * Salva le condizioni per il calcolo del costo.
     *
     * @param \Application\Showcase\Model\Shipping $shipping
     */
    protected function saveConditions(Shipping $shipping)
    {
        $conditions = isset($_POST['conditions']) && is_array($_POST['conditions']) ? $_POST['conditions'] : [];

        foreach ($conditions as $id => $condition) {
            if ((isset($condition['is_deletable']) && $condition['is_deletable']) || (trim($condition['cost']) == '' && trim($condition['condition_by_price']) == '')) {
                if (is_numeric($id)) {
                    $shipping->deleteShippingCondition($id);
                }

                continue;
            } else {
                if (is_numeric($id)) {
                    // aggiorno condizione
                    $shipping->updateShippingCondition($id, $condition);
                } else {
                    // nuova condizione
                    $shipping->addShippingCondition($condition);
                }
            }
        }
    }

    /**
     * @return \Application\Cms\Utilities\NodeTypeHelper
     */
    public function getHelper()
    {
        /** @var \Application\Cms\Utilities\NodeTypeHelper $helper */
        $helper = $this->container->get('nodetype_helper');

        $helper->setModuleSiteClass(NodeType::class);

        return $helper;
    }
}
