<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Controller\Admin;

use Application\Admin\Controller\CrudController;
use Application\Core\Model\Account;
use Application\Core\Utilities\Validator;
use Application\Showcase\Controller\Admin\Orders\OrderEditPanelCollection;
use Application\Showcase\Model\NodeType;
use Application\Showcase\Model\Order;
use Pongho\Core\TemplateResponse;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\Exception\HttpUnauthorizedException;
use Pongho\Menu\Item;
use Pongho\Menu\Menu;
use Pongho\Template\View;

/**
 * Controller per la gestione degli ordini.
 */
class OrdersController extends CrudController
{
    const FILTER_CUSTOMER = 'customer';
    const FILTER_PENDING = 'pending';
    const FILTER_PROCESSED = 'processed';
    const FILTER_CANCELED = 'canceled';

    /**
     * Elenco dei filtri possibili.
     *
     * @var array
     */
    protected static $FILTERS = [
        self::FILTER_PENDING,
        self::FILTER_PROCESSED,
        self::FILTER_CANCELED
    ];

    /**
     * {@inheritdoc}
     *
     * TODO: Valutare se questo metodo serve...
     */
    protected function boot()
    {
        parent::boot();

        // Potrei aver impostato la risposta con il redirect alla pagina di login.
        if ($this->response) {
            return;
        }

        // Permesso consentito?
        if (!$this->getHelper()->getUser()->hasPermit('shop.admin.orders')) {
            throw new HttpUnauthorizedException();
        }
    }

    /**
     * {@inheritdoc}
     */
    public function displayArchive($page = 1)
    {
        $pagination_query = [];
        $path = $this->getParameter('path');

        $customer_uid = isset($_GET['c']) ? $_GET['c'] : null;
        $filter = isset($_GET['filter']) && in_array($_GET['filter'],self::$FILTERS) ? $_GET['filter'] : ($customer_uid ? self::FILTER_CUSTOMER : self::FILTER_PENDING);
        $query = isset($_GET['q']) ? html_escape(trim($_GET['q'])) : '';

        // Opzioni base
        $options = [
            'conditions' => ['site_id = :site', 'site' => $this->getSiteId()],
            'order'      => 'ordered_at DESC',
            'include'    => ['customer', 'payment', 'shipping'],
        ];

        // Ricerca sul cliente
        if ($customer_uid) {
            if (is_numeric($customer_uid)) {
                $customer = Account::find($customer_uid);

                if ($customer) {
                    $options['conditions'][0] .= ' AND customer_id = :customer';
                    $options['conditions']['customer'] = $customer_uid;
                }
            } else {
                $customer = Account::first([
                    'conditions' => ['email = ?', $customer_uid],
                ]);

                if ($customer) {
                    $options['conditions'][0] .= ' AND customer_email = :customer';
                    $options['conditions']['customer'] = $customer_uid;
                }
            }

            if ($customer) {
                $pagination_query['c'] = $customer_uid;
            }
        } else {
            $customer = null;
        }

        // Filtri per lo stato
        $filter_status = $this->getFilterStatus($filter, $options, $customer_uid);
        $pagination_query['filter'] = $filter;

        // Aggiungo le condizioni per il filtro corrente
        $options = $this->addFilterConditions($options, $filter);

        // Query di ricerca
        if (!empty($query)) {
            $options['conditions'][0] .= ' AND id = :query';
            $options['conditions']['query'] = $query;

            $pagination_query['q'] = $query;
        }

        $options = $this->getHelper()->filter($this, 'admin.orders.filter_archive_options', $options, ['page' => $page]);

        $orders = Order::page($page, $this->records_per_page, $options);
        $total = Order::count($options);

        foreach ($orders as $order) {
            /** @var Order $order */
            $order = $this->getHelper()->filter($this, 'admin.orders.filter_order', $order);

            $actions = $this->getHelper()->filter(
                $this,
                'admin.orders.filter_node_actions',
                [
                    ['view', 'href' => $this->url("/{$path}/view/{$order->id}/")],
                ]
            );

            $order->actions = $this->parseActions($actions);

            $customer_uid = $order->customerIsGuest() ? $order->getCustomerEmail() : $order->customer->id;
            $order->customer_orders_url = $this->url("/{$path}/?c=" . $customer_uid);
        }

        // vista
        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/orders_list.php')
            ->assignVars([
                'search_url'      => $this->url("/{$path}/"),
                'export_url'      => $this->url("/{$path}/export/"),
                'query'           => $query,
                'customer'        => $customer,
                'list_orders_url' => $this->url("/{$path}/"),
                'filter_status'   => $filter_status,
                'filter'          => $filter,
                'orders'          => $orders,
            ]);

        // Paginazione
        $pagination_query['site'] = $this->getSiteId();
        $this->displayPagination($page, $total, $this->records_per_page, $pagination_query);
    }

    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return Order::class;
    }

    /**
     * {@inheritdoc}
     */
    public function getTitle($model)
    {
        $translator = $this->getHelper()->getTranslator();

        if ($this->getAction() === 'add') {
            return $translator->trans('Add order');
        }

        return $translator->trans('Edit the order “%title%”', ['%title%' => $model->id]);
    }

    /**
     * {@inheritdoc}
     */
    public function getFields($model)
    {
        return [];
    }

    /**
     * @return bool
     */
    protected function isExportEnabled()
    {
        return (bool) $this->getHelper()->getSite()->getOption('enable_orders_export');
    }

    /**
     * @return \Pongho\Http\Response
     * @throws HttpNotFoundException If the export feature is not enabled.
     */
    protected function exportAction()
    {
        if (!$this->isExportEnabled()) {
            throw new HttpNotFoundException();
        }

        /**
         * @var \Application\Core\Export\DataExporterInterface $exporter
         * @var \Application\Core\Export\SerializerInterface $serializer
         */
        $exporter = $this->getContainer()->get('admin.orders.exporter');
        $serializer = $this->getContainer()->get('admin.export_serializer');

        $response = $serializer->serialize($exporter);

        $response->getHeaders()
            ->makeDownloadable('orders.' . $serializer->getFileExtension())
            ->makeNoCacheable();

        return $response;
    }

    /**
     * Azione `view`.
     *
     * Permette di visualizzare un ordine.
     *
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     */
    public function viewAction()
    {
        /** @var \Application\Showcase\Model\Order $order */
        $order = $this->getModel();

        if ($order === null) {
            throw new HttpNotFoundException();
        }

        $path = $this->getParameter('path');

        $request = $this->getRequest();
        if ($request->getMethod() === 'POST') {
            $orderEditPanelCollection = $this->getOrderEditPanelCollection($order);
            $action = $request->post->get('order[action]', null, true);

            $orderEditPanelCollection->get($action)->handleRequest($order, $request);
        }

        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/orders_view.php')
            ->assignVars([
                'title'                    => $this->getHelper()->getTranslator()->trans('Order no. %title%', ['%title%' => $order->id]),
                'order'                    => $order,
                'edit_customer_url'        => $order->customerIsGuest() ? false : $this->url('/users/edit/' . $order->customer_id . '/'),
                'action_url'               => $this->url("/{$path}/view/{$order->id}/"),
                'edit_cc_action_url'       => $this->url("/{$path}/editcc/{$order->id}/"),
                'check_status_url'         => $order->getPaymentHandler()->supportGatewayDetails() ? $this->url("/{$path}/checkstatus/{$order->id}/") : false,
                'orderEditPanelCollection' => $this->getOrderEditPanelCollection($order),
            ]);

        $this->getHelper()
            ->addCss(pongho_url(
                '/Application/Showcase/Resources/views/css/orders.css?v=' . filemtime(
                    __DIR__ . '/../../Resources/views/css/orders.css'
                )
            ))
            ->addJavascriptInline(
                <<<JS
$(document).ready(function () {
    "use strict";
    
    $('#order-cc-link').click(function (event) {
        event.preventDefault();

        $(this).colorbox({
            inline: true,
            width: 400,
            height: 250
        });
    });

    $("#order-check-status").colorbox({
        height: 500,
        width: 500
    });
});
JS
            );
    }

    /**
     * @param Order $order
     * @return OrderEditPanelCollection
     */
    private function getOrderEditPanelCollection(Order $order)
    {
        $collection = new OrderEditPanelCollection();

        if (!$order->isProcessed() && !$order->isCanceled()) {
            if (!$order->isPaid()) {
                $collection->push(
                    $this->getContainer()->get('shop.admin.orders.confirm_order_payment_panel')
                );
            }

            $collection->push(
                $this->getContainer()->get('shop.admin.orders.process_order_panel')
            );

            $collection->push(
                $this->getContainer()->get('shop.admin.orders.cancel_order_panel')
            );
        }

        return $this->getHelper()->filter(
            $this,
            'admin.orders.filter_order_edit_panel_collection',
            $collection
        );
    }

    /**
     * Azione `editcc`.
     *
     * Permette di modificare il codice cliente (Partita IVA, Codice Fiscale o equivalenti).
     *
     * @return \Pongho\Http\Response
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     */
    public function editccAction()
    {
        if (($order = $this->getModel()) === null) {
            throw new HttpNotFoundException();
        }

        $path = $this->getParameter('path');

        if ($this->getRequest()->getMethod() === 'POST') {
            $customer_code_name = isset($_POST['customer_code_name']) ? $_POST['customer_code_name'] : '';
            $customer_code_value = isset($_POST['customer_code_value']) ? trim($_POST['customer_code_value']) : '';

            $is_valid = !empty($customer_code_value);

            // @todo Internazionalizzazione dei nomi.
            if (!in_array($customer_code_name, ['partita_iva', 'codice_fiscale'])) {
                $is_valid = false;
            }

            if ($is_valid && method_exists('Application\\Core\\Utilities\\Validator', $customer_code_name)) {
                if (!Validator::$customer_code_name($customer_code_value)) {
                    $is_valid = false;
                }
            }

            if ($is_valid) {
                $order->customer_code_name = $customer_code_name;
                $order->customer_code_value = $customer_code_value;

                $order->save();
            }
        }

        return $this->getHelper()->redirectResponse($this->url("/{$path}/view/{$order->id}/"));
    }

    /**
     * Azione `checkstatus`.
     *
     * Permette di controllare lo stato dell’ordine chiedendolo al gateway del pagamento, se questo lo supporta.
     *
     * @return \Pongho\Http\Response
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     */
    public function checkstatusAction()
    {
        /** @var \Application\Showcase\Model\Order $order */
        if (($order = $this->getModel()) === null) {
            throw new HttpNotFoundException();
        }

        $handler = $order->getPaymentHandler();

        if (!$handler->supportGatewayDetails()) {
            throw new HttpNotFoundException();
        }

        $view = new View(__DIR__ . '/../../Resources/views/order_details.php');
        $view
            ->assignVars($this->getHelper()->getViewGlobalVariables())
            ->assignVars([
                'order'   => $order,
                'details' => $handler->getOrderDetails(),
            ]);

        return new TemplateResponse($view);
    }

    /**
     * {@inheritdoc}
     */
    public function addAction()
    {
        throw new HttpNotFoundException();
    }

    /**
     * {@inheritdoc}
     */
    public function editAction()
    {
        throw new HttpNotFoundException();
    }

    /**
     * {@inheritdoc}
     */
    public function deleteAction()
    {
        throw new HttpNotFoundException();
    }

    /**
     * {@inheritdoc}
     */
    protected function getMassActions()
    {
        return [];
    }

    /**
     * Aggiunge le condizioni di ricerca per il filtro.
     *
     * @param array   $options Elenco delle opzioni base.
     * @param integer $filter  Il filtro di ricerca: deve essere una delle costanti FILTER_* di questa classe.
     * @return array
     */
    protected function addFilterConditions(array $options, $filter)
    {
        switch ($filter) {
            case self::FILTER_PROCESSED:

                $options['conditions'][0] .= " AND status IN ('" . implode("', '", Order::$PROCESSED_STATUSES) . "')";

                break;

            case self::FILTER_CANCELED:

                $options['conditions'][0] .= " AND status IN ('" . implode("', '", Order::$CANCELED_STATUSES) . "')";

                break;

            case self::FILTER_CUSTOMER:

                $options['conditions'][0] .= " AND status NOT IN ('" . Order::STATUS_CART . "', '" . Order::STATUS_PAYMENT . "', '" . Order::STATUS_PAYMENT_LOCKED . "', '" . Order::STATUS_TRANSACTION . "')";

                break;

            default:

                $options['conditions'][0] .= " AND status IN ('" . implode("', '", Order::$ORDER_STATUSES) . "')";

                break;
        }

        return $options;
    }

    /**
     * Restituisce il codice HTML per i filtri degli ordini.
     *
     * @param integer $filter       Il filtro di ricerca: deve essere una delle costanti FILTER_* di questa classe.
     * @param array   $options      Elenco delle opzioni base.
     * @param integer $customer_uid ID o email del cliente se si sta filtrando sul cliente.
     * @return string
     */
    protected function getFilterStatus($filter, array $options, $customer_uid = null)
    {
        $translator = $this->getHelper()->getTranslator();

        /** @var Order $model_class */
        $model_class = $this->getModelClass();
        $path = $this->getParameter('path');
        $label_tpl = '%s <span class="count">(%d)</span>';
        $url_tpl = "/{$path}/?filter=%s" . ($customer_uid === null ? '' : "&c={$customer_uid}");

        $filter_status = new Menu('filter-status');

        if ($customer_uid) {
            $title = $translator->trans('All');
            $total = $model_class::count($this->addFilterConditions($options, self::FILTER_CUSTOMER));
            $label = sprintf($label_tpl, $title, $total);

            $all = new Item($label, $this->url(sprintf($url_tpl, self::FILTER_CUSTOMER)), ['title' => $title]);

            if ($filter === self::FILTER_CUSTOMER) {
                $all->setActive();
            }

            $filter_status->add($all);
        }

        foreach (self::$FILTERS as $_filter) {
            $total = $model_class::count($this->addFilterConditions($options, $_filter));

            if ($total) {
                // TODO: Spostare in qualche modo queste traduzioni nel Translator
                $title = $this->getHelper()->getLocalization()->get('filter_orders_' . $_filter);
                $label = sprintf($label_tpl, $title, $total);

                $item = new Item($label, $this->url(sprintf($url_tpl, $_filter)), ['title' => $title]);

                if ($_filter === $filter) {
                    $item->setActive();
                }

                $filter_status->add($item);
            }
        }

        return $filter_status;
    }

    /**
     * @return \Application\Cms\Utilities\NodeTypeHelper
     */
    public function getHelper()
    {
        /** @var \Application\Cms\Utilities\NodeTypeHelper $helper */
        $helper = $this->container->get('nodetype_helper');

        $helper->setModuleSiteClass(NodeType::class);

        return $helper;
    }
}
