<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Controller\Admin;

use Application\Admin\Controller\AdminController;
use Application\Admin\Utilities\TablesManager;
use Application\Core\Utilities\Pagination;
use Application\Showcase\GoogleShopping\Product;
use Application\Showcase\Model\Size;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Utilities\Inflector;

/**
 * GoogleShoppingController
 */
class GoogleShoppingController extends AdminController
{
    /**
     * @var int
     */
    protected $records_per_page = 50;

    /**
     * Visualizza la pagina iniziale dell’archivio.
     */
    public function indexAction()
    {
        return $this->displayArchive();
    }

    /**
     * Gestisce le pagine dell’archivio.
     */
    public function pageAction()
    {
        return $this->displayArchive($this->getParameter('page', 1, true));
    }

    /**
     * Permette di fare una ricerca nei prodotti.
     *
     * @return \Pongho\Http\JsonResponse
     */
    public function searchAction()
    {
        // TODO
    }

    /**
     * Rimuove il prodotto dal feed.
     *
     * @return \Pongho\Http\RedirectResponse
     */
    public function removeAction()
    {
        $id = (int) $this->getParameter('id');

        /** @var Size $size */
        $size = Size::find($id);
        $size->removeFromGoogleShoppingFeed()->save();

        $translator = $this->getHelper()->getTranslator();

        if ($this->getRequest()->isAjax()) {
            return $this->getHelper()->displayMessage(
                $translator->trans('The product has been removed from the feed.')
            );
        }

        return $this->getHelper()->redirectResponse(
            $this->url('/shop-google-shopping/')
        );
    }

    /**
     * @param int $page
     * @return \Pongho\Http\Response|null
     */
    protected function displayArchive($page = 1)
    {
        $site = $this->getHelper()->getSite();

        if (!$site->getOption('enable_google_shopping')) {
            throw new HttpNotFoundException();
        }

        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/google-shopping.php')
            ->assignVars([
                'table'      => $this->getTableManager($page),
                'pagination' => $this->createPagination($page),
            ]);

        $this->getHelper()
            ->addCss(pongho_url('/Application/Showcase/Resources/views/css/google-shopping.css'))
            ->addJavascript(pongho_url('/Application/Showcase/Resources/views/js/google-shopping.js'));

        return null;
    }

    /**
     * @param int $page
     * @return TablesManager
     */
    protected function getTableManager($page)
    {
        return new TablesManager(
            $this->getTableManagerConfig(),
            $this->getRequest(),
            $this->getRecords($page)
        );
    }

    /**
     * @return array
     */
    protected function getTableManagerConfig()
    {
        /**
         * @var \Pongho\Sdk\GoogleShopping\ProductValidator $validator
         */
        $validator = $this->getContainer()->get('google_shopping_product_validator');
        $translator = $this->getHelper()->getTranslator();

        return [
            'columns'         => [
                'check'          => [
                    'label'  => $translator->trans('Valid'),
                    'class'  => 'google-shopping__check',
                    'render' => function (Product $product) use ($validator) {
                        if ($validator->isValid($product)) {
                            return '';
                        }

                        $code = '';
                        foreach ($validator->getErrors($product) as $error) {
                            $code .= '<li>' . $error . '</li>';
                        }

                        return '<div style="display: none;"><ul>' . $code . '</ul></div>';
                    }
                ],
                'title'          => [
                    'name'      => 'title',
                    'label'     => $translator->trans('Product'),
                    'class'     => 'main',
                    'orderable' => true,
                    'render'    => function (Product $product) use ($translator) {
                        $path = Inflector::pluralize($product->getNodeType());
                        $product_id = $product->getProductId();
                        $item_id = $product->getId()->getValue();

                        // TODO: permessi
                        $actions = [
                            [
                                $translator->trans('Edit product'),
                                'href'  => $this->url("/{$path}/edit/{$product_id}/"),
                                'class' => 'edit'
                            ],
                            [
                                $translator->trans('Remove from feed'),
                                'href'                 => $this->url("/shop-google-shopping/remove/{$item_id}/"),
                                'class'                => 'delete',
                                'data-confirm-message' => $translator->trans(
                                    'Are you sure you want to remove the product from the feed?'
                                ),
                            ],
                        ];

                        $actions = $this->parseActions($actions);

                        return '<strong>' . $product->getTitle()->getValue()
                        . '</strong><div class="actions"><span>' . $actions . '</span></div>';
                    }
                ],
                'product_type'   => [
                    'name'      => 'category',
                    'label'     => $translator->trans('Category'),
                    'orderable' => true,
                    'render'    => function (Product $product) {
                        return $product->getProductType()->getValue();
                    }
                ],
                'brand'          => [
                    'name'      => 'brand',
                    'label'     => $translator->trans('Brand'),
                    'orderable' => true,
                    'render'    => function (Product $product) {
                        return $product->getBrand()->getValue();
                    }
                ],
                'size'           => [
                    'name'   => 'size',
                    'label'  => $translator->trans('Size'),
                    'render' => function (Product $product) {
                        return $product->getSize()->getValue();
                    }
                ],
                'availabilities' => [
                    'name'      => 'availabilities',
                    'label'     => $translator->trans('Availabilities'),
                    'orderable' => true,
                    'render'    => function (Product $product) {
                        return $product->getAvailabilities();
                    }
                ],
                'price'          => [
                    'name'   => 'price',
                    'label'  => $translator->trans('Price'),
                    'render' => function (Product $product) {
                        return $product->getPrice()->getValue();
                    }
                ],
            ],
            'row_attributes'  => function (Product $product) use ($validator) {
                return [
                    'class' => $validator->isValid($product) ? 'google-shopping__valid' : 'google-shopping__warning',
                ];
            },
            'show_ckb'        => false,
            'show_add'        => false,
            'empty_recordset' => $translator->trans('No products to show yet'),
        ];
    }

    /**
     * @param int $page
     * @return \Application\Showcase\GoogleShopping\Product[]
     */
    protected function getRecords($page)
    {
        /** @var \Application\Showcase\Model\Manager\GoogleShoppingProductManager $manager */
        $manager = $this->getContainer()->get('google_shopping_product_manager');

        $order = $this->buildQueryOrder();
        $limit = $this->records_per_page;
        $offset = ($page - 1) * $limit;

        $products = $manager->findAll($order, $limit, $offset);
        $manager->populateNodeType($products);

        return $products;
    }

    /**
     * @return array
     */
    protected function buildQueryOrder()
    {
        $request = $this->getRequest();

        if ($request->query->has('o')) {
            return (array) $request->query->get('o');
        }

        return null;
    }

    /**
     * @param array $actions
     * @return string
     */
    protected function parseActions(array $actions)
    {
        /** @var \Pongho\Template\Html $html */
        $html = $this->getContainer()->get('template_html');

        $code = [];
        foreach ($actions as $action) {
            $label = array_shift($action);
            $attributes = $action;

            $code[] = $html->generateTagBlock('a', $label, $attributes);
        }

        return implode(' | ', $code);
    }

    /**
     * @param int $page
     * @return Pagination
     */
    protected function createPagination($page)
    {
        /** @var \Application\Showcase\Model\Manager\GoogleShoppingProductManager $manager */
        $manager = $this->getContainer()->get('google_shopping_product_manager');

        $url = $this->url('/shop-google-shopping/', '&');
        $url_info = explode('?', $url);

        $qs = [];

        if (isset($url_info[1])) {
            parse_str($url_info[1], $qs);
        }

        return new Pagination(
            $url_info[0],
            $manager->countAll(),
            $page,
            $this->records_per_page,
            $qs
        );
    }
}
