<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Controller\Admin;

use Application\Admin\Controller\CrudFormController;
use Application\Admin\Form\FormConfig;
use Application\Core\Form\Field\DiscountField;
use Application\Core\Form\Field\PriceField;
use Application\Core\Model\Role;
use Application\Newsletter\Model\NewsletterEmail;
use Application\Showcase\Controller\NodeController;
use Application\Showcase\Model\OrderDiscount;
use Application\Showcase\Model\OrderDiscountCondition;
use Pongho\Core\TemplateResponse;
use Pongho\Form\Field\DateField;
use Pongho\Form\Field\RadioField;
use Pongho\Form\Field\SelectField;
use Pongho\Form\Field\TextareaField;
use Pongho\Form\Field\TextField;
use Pongho\Form\Fieldset;
use Pongho\Form\Filter\EmailFilter;
use Pongho\Form\Subject\ModelSubject;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\Exception\HttpUnauthorizedException;
use Pongho\Http\JsonResponse;
use Pongho\Http\Response;
use Pongho\Template\View;
use Pongho\Template\ViewClosure;
use Pongho\Utilities\DateTime;

/**
 * Controller per la gestione dei buoni sconto.
 */
class CouponsController extends CrudFormController
{
    /**
     * {@inheritdoc}
     *
     * TODO: Valutare se questo metodo serve...
     */
    protected function boot()
    {
        parent::boot();

        // Potrei aver impostato la risposta con il redirect alla pagina di login.
        if ($this->response) {
            return;
        }

        // Permesso consentito?
        if (!$this->getHelper()->getUser()->hasPermit('shop.admin.coupons')) {
            throw new HttpUnauthorizedException();
        }
    }

    protected function beforeSetTemplateCallback()
    {
        $this->enableWysiwyg();

        $this->getHelper()->loadSelect2();

        $this->getHelper()->addJavascriptInline(<<<JS
(function () {
    "use strict";

    var validFrom = $('#field-coupon-valid_from'), validTo = $('#field-coupon-expire_on');

    validFrom.on('change', function () {
        var dateFrom = validFrom.val();

        if (new Date(dateFrom) > new Date(validTo.val())) {
            validTo.val(dateFrom);
        }
    });

    validTo.on('change', function () {
        var dateEnd = validTo.val();

        if (new Date(dateEnd) < new Date(validFrom.val())) {
            validFrom.val(dateEnd);
        }
    });
}());

(function () {
    "use strict";

    function init() {
        $('.js-select2').select2();

        $('.js-select2-ajax').each(function () {
            var el = $(this);

            el.select2({
                ajax: {
                    url: el.data('source'),
                    dataType: 'json',
                    delay: 250,
                    processResults: function (data) {
                        return {
                            results: data
                        };
                    }
                },
                minimumInputLength: 3,
                escapeMarkup: function (markup) { return markup; },
                templateResult: function (item) {
                    if (item.loading) {
                        return item.text;
                    }

                    var markup;

                    if (el.data('type') === 'user') {
                        markup = '<div class="select2-result">' +
                            '<strong>' + item.fullname + '</strong><br>' +
                            '<em>' + item.email + '</em>' +
                            '</div>';
                    } else {
                        markup = '<div class="select2-result"><strong>' + item.name + '</strong>';

                        if (item.code) {
                            markup += '<br><em>Cod. ' + item.code + '</em>';
                        }

                        markup += '</div>';
                    }

                    return markup;
                },
                templateSelection: function (item) {
                    if (el.data('type') === 'user') {
                        return item.fullname || item.text;
                    } else {
                        return item.name || item.text;
                    }
                }
            });
        });

        $('.js-coupon-relations').each(function () {
            var section = $(this),
                optionsSelector = section.data('option-selector'),
                masterField = section.find('input[type="radio"]'),
                slaves = section.find('.js-coupon-relations-options').find('.control-group');

            masterField.on('change', function () {
                onRelChange(optionsSelector, masterField, slaves);
            });

            onRelChange(optionsSelector, masterField, slaves);
        });
    }

    function onRelChange(optionsSelector, masterField, slaves) {
        slaves.hide();

        var fieldGroup = $(optionsSelector + '_' + masterField.filter(':checked').val()).closest('.control-group');

        fieldGroup.show();
    }

    $('#form-coupon').closest('.form-wrapper').on('after-update', init);
    init();
}());

(function () {
    "use strict";

    $('body')
        .on('click', '#button-coupon-preview', function (e) {
            e.preventDefault();

            var button = $(e.currentTarget),
                body = $('body'),
                bodyOverflow = body.css('overflow');

            $.colorbox({
                href: button.attr('href'),
                innerWidth: 700,
                innerHeight: '80%',
                onComplete: function () {
                    body.css({overflow: 'hidden'});
                    
                    
                },
                onClosed: function () {
                    body.css({overflow: bodyOverflow});
                }
            });
        })
        .on('click', '.js-wysiwyg-code', function (e) {
            var button = $(e.currentTarget),
                form = $(button.data('form')),
                code = button.data('code');

            form.tinymce().execCommand('mceInsertContent', false, code);
        })
    ;
}());
JS
        );
    }

    /**
     * {@inheritdoc}
     */
    public function getNamespace()
    {
        return 'shop.coupon';
    }

    /**
     * {@inheritdoc}
     */
    public function getEventNamespace($event_name)
    {
        return 'admin.shop.coupon.' . $event_name;
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveTitle()
    {
        return $this->getHelper()->getTranslator()->trans('Discount coupons settings');
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddButtonText()
    {
        return $this->getHelper()->getTranslator()->trans('Add coupon');
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddFirstText()
    {
        return $this->getHelper()->getTranslator()->trans('Add the first coupon');
    }

    /**
     * {@inheritdoc}
     */
    public function getAddEditTitle($model)
    {
        $translator = $this->getHelper()->getTranslator();

        if ($model->is_new_record()) {
            return $translator->trans('Add coupon');
        }

        return $translator->trans('Edit the coupon “%title%”', ['%title%' => $model->code]);
    }

    /**
     * {@inheritdoc}
     */
    protected function hasAddButton()
    {
        return true;
    }

    /**
     * {@inheritdoc}
     */
    protected function getTableColumns()
    {
        $translator = $this->getHelper()->getTranslator();

        return [
            [
                'name'   => 'code',
                'label'  => $translator->trans('Coupon'),
                'class'  => 'main',
                'render' => function (OrderDiscount $row) {
                    $code = '<strong>' . $row->getCode() . '</strong>';

                    if ($row->getDescription()) {
                        $code .= '<br>' . $row->getDescription();
                    }

                    $code .= '<div class="actions"><span>' . $row->actions . '</span></div>';

                    return $code;
                }
            ],
            [
                'name'   => 'discount',
                'label'  => $translator->trans('Discount'),
                'render' => function (OrderDiscount $row) {
                    return $row->condition->formatValue();
                }
            ],
            [
                'name'   => 'condition',
                'label'  => $translator->trans('Min. amount'),
                'render' => function (OrderDiscount $row) {
                    return $row->condition->formatCondition();
                }
            ],
            [
                'name'   => 'valid_from',
                'label'  => $translator->trans('Valid from'),
                'render' => function (OrderDiscount $row) {
                    if ($row->getValidFrom()) {
                        return $row->getValidFrom()->format('d/m/Y');
                    } else {
                        return '-';
                    }
                }
            ],
            [
                'name'   => 'expire_on',
                'label'  => $translator->trans('Expire on'),
                'render' => function (OrderDiscount $row) {
                    return $row->getExpireOn()->format('d/m/Y');
                }
            ],
        ];
    }

    /**
     * {@inheritdoc}
     *
     * @param OrderDiscount $row
     */
    public function parseArchiveRow($row)
    {
        $path = $this->getParameter('path');
        $translator = $this->getHelper()->getTranslator();

        $actions = [
            [
                $translator->trans('Edit'),
                'href'  => $this->url("/{$path}/edit/{$row->id}/"),
                'class' => 'edit',
            ],
        ];

        if ($row->isDeletable()) {
            $actions[] = [
                $translator->trans('Delete'),
                'href'  => $this->url("/{$path}/delete/{$row->id}/"),
                'class' => 'delete',
            ];
        }

        $row->actions = $this->parseActions($actions);
    }

    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return OrderDiscount::class;
    }

    /**
     * {@inheritdoc}
     *
     * @return OrderDiscount
     */
    protected function getModel()
    {
        /** @var OrderDiscount $model */
        $model = parent::getModel();

        if ($this->getAction() === 'add') {
            $model->handler_name = 'coupon_code';
            $model->code = strtoupper(random(10));
            $model->valid_from = new DateTime();
            $model->expire_on = new DateTime();
        }

        return $model;
    }

    /**
     * {@inheritdoc}
     */
    protected function getModelOptions()
    {
        return [
            'conditions' => [
                'handler_name = :handler',
                'handler' => 'coupon_code',
            ],
            'include'    => 'condition',
        ];
    }

    /**
     * {@inheritdoc}
     */
    protected function getFormsOptions()
    {
        return $this->getHelper()->filter(
            $this,
            $this->getEventNamespace('filter_form_options'),
            $this->getDefaultFormsOptions(),
            [
                'action' => $this->getAction(),
            ]
        );
    }

    /**
     * @return FormConfig
     */
    protected function getDefaultFormsOptions()
    {
        $translator = $this->getHelper()->getTranslator();

        /** @var OrderDiscount $discountModel */
        $discountModel = $this->getModel();
        $discountSubject = new ModelSubject($discountModel);

        /** @var OrderDiscountCondition $conditionsModel */
        $conditionsModel = OrderDiscountCondition::first([
            'conditions' => ['discount_id = ?', $discountModel->id],
        ]);

        if ($conditionsModel === null) {
            $conditionsModel = new OrderDiscountCondition([
                'discount'  => '0,00',
                'condition' => 0,
            ]);
        }

        $conditionsSubject = new ModelSubject($conditionsModel);

        $discountSubject->attachGetter('conditions', function () use ($conditionsSubject) {
            return $conditionsSubject;
        });

        $conditionsSubject->setBeforeSaveCallback(function (ModelSubject $subject) use ($discountModel) {
            $subject->set('discount_id', $discountModel->id);
        });

        $config = new FormConfig('coupon', $discountSubject, $this->getHelper()->getLocalization());

        // Struttura
        $config->addBaseStructure($this->getActiveTab(), $this->currentUrl());
        $config->addTab('content/main', $translator->trans('Main'));
        $config->addPanel('content/main/coupon');

        $codeFieldAttributes = [];
        if ($this->getAction() !== 'add') {
            $codeFieldAttributes['readonly'] = 'readonly';
        }

        $config->addFields(
            'content/main/coupon',
            [
                'code'        => [
                    'class'      => TextField::class,
                    'label'      => $translator->trans('Coupon code'),
                    'attributes' => $codeFieldAttributes,
                ],
                'description' => [
                    'class'      => TextareaField::class,
                    'label'      => $translator->trans('Description'),
                    'attributes' => [
                        'maxlength' => 200,
                        'rows'      => 5,
                    ],
                ],
                'conditions'  => [
                    'class'    => Fieldset::class,
                    'settings' => [
                        'is_rel'     => true,
                        'hide_label' => true,
                    ],
                ],
                'valid_from'  => [
                    'class'       => DateField::class,
                    'label'       => $translator->trans('Valid from'),
                    'description' => $translator->trans('At 00:00:00'),
                ],
                'expire_on'   => [
                    'class'       => DateField::class,
                    'label'       => $translator->trans('Expire on'),
                    'description' => $translator->trans('At 23:59:59'),
                ],
            ]
        );

        $config->addFields(
            'content/main/coupon/conditions',
            [
                'value'     => [
                    'class'       => DiscountField::class,
                    'label'       => $translator->trans('Discount'),
                    'description' => $translator->trans(
                        'Discount (format: 20% or %format%)',
                        [
                            '%format%' => format_price(123.45, false)
                        ]
                    ),
                ],
                'condition' => [
                    'class' => PriceField::class,
                    'label' => $translator->trans('Min. amount'),
                ],
            ]
        );

        $this->addUsersRelConfig($config, $discountSubject);
        $this->addProductsRelConfig($config, $discountSubject);
        $this->addNotificationTabConfig($config, $discountSubject);

        return $config;
    }

    /**
     * @param FormConfig   $config
     * @param ModelSubject $discountSubject
     */
    protected function addUsersRelConfig(FormConfig $config, ModelSubject $discountSubject)
    {
        $translator = $this->getHelper()->getTranslator();
        $user = $this->getHelper()->getUser();

        $config->addField(
            'content/main/users',
            [
                'class'      => Fieldset::class,
                'label'      => $translator->trans('Users'),
                'attributes' => [
                    'class'                => 'panel js-coupon-relations',
                    'data-option-selector' => '#field-coupon-user',
                ],
            ]
        );

        $relModesOptions = [];
        $relModesOptions['all'] = $translator->trans('All');
        $relModesOptions['user'] = $translator->trans('Single user');

        // FIXME: improve permissions on groups
        if ($user->hasPermit('users.handle_groups')) {
            $relModesOptions['group'] = $translator->trans('Group');
        }

        // FIXME: improve permissions on roles
        $relModesOptions['role'] = $translator->trans('Role');

        $config->addField(
            'content/main/users/user_rel_mode',
            [
                'class'   => RadioField::class,
                'label'   => $translator->trans('Association type'),
                'options' => $relModesOptions,
            ]
        );

        $config->addField(
            'content/main/users/rels',
            [
                'class'      => Fieldset::class,
                'settings'   => [
                    'hide_label' => true,
                ],
                'attributes' => [
                    'class' => 'js-coupon-relations-options',
                ],
            ]
        );

        $config->addField(
            'content/main/users/rels/user_user',
            [
                'class'      => SelectField::class,
                'label'      => $translator->trans('Find a user'),
                'attributes' => [
                    'class'       => 'js-select2-ajax',
                    'multiple'    => 'multiple', // FIXME: move on field settings
                    'data-source' => $this->url('/shop-coupons/suggest_users/'),
                    'data-type'   => 'user',
                ],
                'options'    => function () use ($config) {
                    /** @var OrderDiscount $discountModel */
                    $discountModel = $config->getFormSubject()->getModel();
                    $userIds = $discountModel->getSetting('user_user', []);

                    if (!is_array($userIds)) {
                        $userIds = [];
                    }

                    /** @var \Application\Core\Model\Manager\AccountManagerInterface $accountManager */
                    $accountManager = $this->getContainer()->get('account_manager');

                    $options = [];
                    foreach ($accountManager->findAllByIds($userIds) as $account) {
                        $options[$account->getId()] = $account->name();
                    }

                    return $options;
                }
            ]
        );

        // FIXME: improve permissions on groups
        if ($user->hasPermit('users.handle_groups')) {
            $config->addField(
                'content/main/users/rels/user_group',
                [
                    'class'      => SelectField::class,
                    'label'      => $translator->trans('Group'),
                    'attributes' => [
                        'class'    => 'js-select2',
                        'multiple' => 'multiple', // FIXME: move on field settings
                    ],
                    'options'    => function () {
                        /** @var \Application\Core\Model\Manager\GroupManagerInterface $manager */
                        $manager = $this->getContainer()->get('group_manager');

                        $options = [];
                        foreach ($manager->findAll() as $group) {
                            $options[$group->getId()] = $group->getName();
                        }

                        return $options;
                    },
                ]
            );
        }

        $config->addField(
            'content/main/users/rels/user_role',
            [
                'class'      => SelectField::class,
                'label'      => $translator->trans('Role'),
                'attributes' => [
                    'class'    => 'js-select2',
                    'multiple' => 'multiple', // FIXME: move on field settings
                ],
                'options'    => function () {
                    /** @var \Application\Core\Model\Manager\RoleManagerInterface $manager */
                    $manager = $this->getContainer()->get('role_manager');

                    $options = [];
                    foreach ($manager->findAll() as $role) {
                        $options[$role->getId()] = $role->getName();
                    }

                    unset($options[Role::USER_NOT_LOGGED]);

                    return $options;
                },
            ]
        );

        $this->attachGetterSetter($discountSubject, 'user_rel_mode', 'all', array_keys($relModesOptions));
        $this->attachGetterSetter($discountSubject, 'user_user');
        $this->attachGetterSetter($discountSubject, 'user_group', []);
        $this->attachGetterSetter($discountSubject, 'user_role', []);
    }

    /**
     * @return JsonResponse
     */
    public function suggest_usersAction()
    {
        $request = $this->getRequest();
        $term = trim($request->query->get('term', ''));

        $result = [];

        if ($term) {
            /** @var \Application\Core\Model\Manager\AccountManagerInterface $accountManager */
            $accountManager = $this->getContainer()->get('account_manager');

            foreach ($accountManager->search($term) as $account) {
                $result[] = [
                    'id'       => $account->getId(),
                    'fullname' => $account->name(),
                    'email'    => $account->getEmail(),
                ];
            }
        }

        return new JsonResponse($result);
    }

    /**
     * @param FormConfig   $config
     * @param ModelSubject $discountSubject
     */
    protected function addProductsRelConfig(FormConfig $config, ModelSubject $discountSubject)
    {
        $translator = $this->getHelper()->getTranslator();

        /** @var \Application\Core\Model\Account $account */
        $account = $this->getHelper()->getUser()->getAccount();
        $languageId = $account->getLanguageId();

        /** @var \Application\Cms\Model\Manager\NodeTypeManager $nodeTypeManager */
        $nodeTypeManager = $this->getContainer()->get('node_type_manager');

        /** @var \Application\Cms\Model\Manager\TaxonomyManagerInterface $taxonomyManager */
        $taxonomyManager = $this->getContainer()->get('taxonomy_manager');

        foreach ($nodeTypeManager->findAllNodeTypesByController(NodeController::class, $languageId) as $nodeType) {
            $config->addField(
                'content/main/node_' . $nodeType->node_type,
                [
                    'class'      => Fieldset::class,
                    'label'      => $translator->trans('Products'),
                    'attributes' => [
                        'class'                => 'panel js-coupon-relations',
                        'data-option-selector' => '#field-coupon-node_' . $nodeType->node_type,
                    ],
                ]
            );

            $taxonomies = $taxonomyManager->findAllByModuleSite($nodeType->getId());

            /*
             * Tipologia di associazione
             */
            $relModeOptions = [
                'all'  => $translator->trans('All'),
                'node' => $translator->trans('Single product'),
            ];

            foreach ($taxonomies as $taxonomy) {
                $relModeOptions['tax_' . $taxonomy->name] = $nodeType->trans('taxonomy_' . $taxonomy->name, $languageId);
            }

            $config->addField(
                'content/main/node_' . $nodeType->node_type . '/node_' . $nodeType->node_type . '_rel_mode',
                [
                    'class'   => RadioField::class,
                    'label'   => $translator->trans('Association type'),
                    'options' => $relModeOptions,
                ]
            );

            $this->attachGetterSetter(
                $discountSubject,
                'node_' . $nodeType->node_type . '_rel_mode',
                'all',
                array_keys($relModeOptions)
            );

            /*
             * Associazione con uno o più prodotti
             */
            $config->addField(
                'content/main/node_' . $nodeType->node_type . '/rels',
                [
                    'class'      => Fieldset::class,
                    'settings'   => [
                        'hide_label' => true,
                    ],
                    'attributes' => [
                        'class' => 'js-coupon-relations-options',
                    ],
                ]
            );

            $config->addField(
                'content/main/node_' . $nodeType->node_type . '/rels/node_' . $nodeType->node_type . '_node',
                [
                    'class'      => SelectField::class,
                    'label'      => $translator->trans('Find a product'),
                    'attributes' => [
                        'class'       => 'js-select2-ajax',
                        'multiple'    => 'multiple', // FIXME: move on field settings
                        'data-source' => $this->url('/shop-coupons/suggest_nodes/?type=' . $nodeType->node_type),
                        'data-type'   => 'node_' . $nodeType->node_type,
                    ],
                    'options'    => function () use ($config, $nodeType, $languageId) {
                        /** @var OrderDiscount $discountModel */
                        $discountModel = $config->getFormSubject()->getModel();
                        $nodeIds = $discountModel->getSetting('node_' . $nodeType->node_type . '_node', []);

                        if (!is_array($nodeIds)) {
                            $nodeIds = [];
                        }

                        /** @var \Application\Cms\Model\Manager\NodeManagerInterface $nodeManager */
                        $nodeManager = $this->getContainer()->get('node_manager');

                        $options = [];
                        foreach ($nodeManager->findAllByIds($nodeIds, $languageId) as $node) {
                            $options[$node->getId()] = $node->title();
                        }

                        return $options;
                    }
                ]
            );

            $this->attachGetterSetter(
                $discountSubject,
                'node_' . $nodeType->node_type . '_node'
            );

            /*
             * Associazione con le tassonomie
             */
            foreach ($taxonomies as $taxonomy) {
                $fieldName = 'node_' . $nodeType->node_type . '_tax_' . $taxonomy->name;

                $config->addField(
                    'content/main/node_' . $nodeType->node_type . '/rels/' . $fieldName,
                    [
                        'class'      => SelectField::class,
                        'label'      => $nodeType->trans('taxonomy_' . $taxonomy->name, $languageId),
                        'attributes' => [
                            'class'    => 'js-select2',
                            'multiple' => 'multiple', // FIXME: move on field settings
                        ],
                        'options'    => function () use ($taxonomy, $languageId) {
                            $options = [];
                            if ($taxonomy->mode === 'hierarchy') {
                                foreach ($taxonomy->getTerms($languageId) as $term) {
                                    $options[$term->parent_id][$term->getId()] = $term->title();
                                }
                            } else {
                                foreach ($taxonomy->getTerms($languageId) as $term) {
                                    $options[$term->getId()] = $term->title();
                                }
                            }

                            return $options;
                        },
                        'settings'   => [
                            'hierarchical' => $taxonomy->mode === 'hierarchy',
                        ],
                    ]
                );

                $this->attachGetterSetter($discountSubject, $fieldName, []);
            }
        }
    }

    /**
     * @return JsonResponse
     */
    public function suggest_nodesAction()
    {
        $request = $this->getRequest();
        $term = trim($request->query->get('term', ''));
        $type = trim($request->query->get('type', ''));

        /** @var \Application\Core\Model\Account $account */
        $account = $this->getHelper()->getUser()->getAccount();
        $languageId = $account->getLanguageId();

        $result = [];

        if ($type) {
            // TODO: check node type
        }

        if ($term) {
            /** @var \Application\Showcase\Model\Manager\ProductManagerInterface $productManager */
            $productManager = $this->getContainer()->get('product_manager');

            foreach ($productManager->search($term, $languageId, ['node_type' => $type]) as $node) {
                $result[] = [
                    'id'   => $node->getId(),
                    'name' => $node->title(),
                    'code' => $node->code(),
                ];
            }
        }

        return new JsonResponse($result);
    }

    private function attachGetterSetter(ModelSubject $subject, $fieldName, $default = null, array $validOptions = null)
    {
        $subject->attachGetter(
            $fieldName,
            function (ModelSubject $subject, $fieldName) use ($default) {
                /** @var OrderDiscount $discountModel */
                $discountModel = $subject->getModel();

                return $discountModel->getSetting($fieldName, $default);
            }
        );

        $subject->attachSetter(
            $fieldName,
            function (ModelSubject $subject, $value, $fieldName) use ($default, $validOptions) {
                if ($validOptions && !in_array($value, $validOptions)) {
                    $value = $default;
                }

                /** @var OrderDiscount $discountModel */
                $discountModel = $subject->getModel();

                $discountModel->setSetting($fieldName, $value);
            }
        );
    }

    /**
     * @param FormConfig   $config
     * @param ModelSubject $discountSubject
     */
    protected function addNotificationTabConfig(FormConfig $config, ModelSubject $discountSubject)
    {
        if ($this->getAction() === 'add') {
            return;
        }

        /** @var \Application\Core\Model\Manager\ApplicationManagerInterface $applicationManager */
        $applicationManager = $this->getContainer()->get('application_manager');

        if (!$applicationManager->isEnabled('Newsletter', $this->getSiteId())) {
            return;
        }

        /** @var OrderDiscount $discountModel */
        $discountModel = $discountSubject->getModel();
        $translator = $this->getHelper()->getTranslator();

        $config->addTab('content/notification', $translator->trans('Coupon notification'));

        $config->addField('content/notification/settings', [
            'class'       => Fieldset::class,
            'attributes'  => [
                'class' => 'panel',
            ],
            'settings'    => [
                'hide_label' => true,
            ],
            'description' => $translator->trans('Send an email notification to users associated with the coupon.'),
        ]);

        $config->addField('content/notification/settings/email_sender_name', [
            'class' => TextField::class,
            'label' => $translator->trans('Sender name'),
        ]);

        $config->addField('content/notification/settings/email_sender_email', [
            'class'  => TextField::class,
            'label'  => $translator->trans('Sender email'),
            'filter' => EmailFilter::class,
        ]);

        $config->addField('content/notification/settings/email_subject', [
            'class'      => TextField::class,
            'label'      => $translator->trans('Subject'),
            'attributes' => [
                'style' => 'width:100%'
            ],
        ]);

        $config->addField('content/notification/settings/email_body', [
            'class'      => TextareaField::class,
            'label'      => $translator->trans('Message body'),
            'attributes' => [
                'class' => 'tinymce-newsletter',
                'cols'  => 30,
                'rows'  => 40,
            ],
            'settings'   => [
                'view' => new ViewClosure(function (TextareaField $field) use ($translator, $discountModel) {
                    $view = new View(__DIR__ . '/../../Resources/views/coupon-email-body-field.php');

                    $view->assignVars([
                        'field'       => $field,
                        'translator'  => $translator,
                        'discount'    => $discountModel,
                        'preview_url' => $this->url('/shop-coupons/notification_preview/' . $discountModel->getId() . '/'),
                    ]);

                    return $view->render();
                }),
            ],
        ]);
    }

    /**
     * @return Response
     */
    public function notification_previewAction()
    {
        $this->checkNewsletterApp();

        $discount = $this->getDiscount();
        $translator = $this->getHelper()->getTranslator();

        if (!$discount->isReadyToSendNotification()) {
            return $this->getHelper()->displayError(
                $this->getHelper()->getTranslator()->trans(
                    'Before send the notification, you must populate the email fields.'
                )
            );
        }

        $view = new View(__DIR__ . '/../../Resources/views/coupon-notification-preview.php');

        $view->assignVars([
            'discount'    => $discount,
            'content_url' => $this->url('/shop-coupons/notification_content/' . $discount->getId() . '/'),
            'send_url'    => $this->url('/shop-coupons/send_emails/' . $discount->getId() . '/'),
            'translator'  => $translator,
        ]);

        return new TemplateResponse($view);
    }

    /**
     * @return Response
     */
    public function notification_contentAction()
    {
        $this->checkNewsletterApp();

        return new Response(
            $this->prepareNotificationBody(
                $this->getDiscount()
            )
        );
    }

    /**
     * @throws HttpNotFoundException If the Newsletter app is not enabled.
     * @throws HttpNotFoundException If the discount does not exist.
     */
    public function send_emailsAction()
    {
        $this->checkNewsletterApp();

        $discount = $this->getDiscount();

        if (!$discount->isReadyToSendNotification()) {
            return $this->getHelper()->displayError(
                $this->getHelper()->getTranslator()->trans(
                    'Before send the notification, you must populate the email fields.'
                )
            );
        }

        $newsletter = new NewsletterEmail();
        $newsletter->setSubject($discount->getEmailSubject());
        $newsletter->setBodyRaw($this->prepareNotificationBody($discount));
        $newsletter->save();

        /** @var \Application\Newsletter\Utilities\SchedulingFacade $schedulingFacade */
        $schedulingFacade = $this->getContainer()->get('newsletter_scheduling_facade');

        $scheduling = $schedulingFacade->prepareScheduling(
            $newsletter,
            $this->buildNotificationRecipientsFilter($discount),
            null,
            $discount->getEmailSenderName(),
            $discount->getEmailSenderEmail()
        );

        $scheduling->save();
        $discount->markAsNotified()->save();

        return $this->getHelper()->displayMessage(
            $this->getHelper()->getTranslator()->trans('The notification will be sent in the next minutes…')
        );
    }

    /**
     * @throws HttpNotFoundException If the newsletter app is not installed or enabled.
     */
    private function checkNewsletterApp()
    {
        /** @var \Application\Core\Model\Manager\ApplicationManagerInterface $applicationManager */
        $applicationManager = $this->getContainer()->get('application_manager');

        if (!$applicationManager->isEnabled('Newsletter', $this->getSiteId())) {
            throw new HttpNotFoundException();
        }
    }

    /**
     * @return OrderDiscount
     * @throws HttpNotFoundException If the discount does not exist.
     *
     * @todo DiscountManager
     */
    private function getDiscount()
    {
        $discountId = (int) $this->getParameter('id');

        if (!$discountId) {
            throw new HttpNotFoundException();
        }

        /** @var OrderDiscount $discount */
        $discount = OrderDiscount::find($discountId);

        if (!$discount) {
            throw new HttpNotFoundException();
        }

        return $discount;
    }

    /**
     * @param OrderDiscount $discount
     * @return \Application\Newsletter\Filter\FilterRequest
     */
    private function buildNotificationRecipientsFilter(OrderDiscount $discount)
    {
        /** @var \Application\Showcase\Discount\Order\CouponCodeDiscount $couponDiscount */
        $couponDiscount = $this->getContainer()->get('shop_coupon_discount');

        return $couponDiscount->buildRecipientsFilter($discount);
    }

    /**
     * @param OrderDiscount $discount
     * @return string
     */
    private function prepareNotificationBody(OrderDiscount $discount)
    {
        return str_replace('%COUPON_CODE%', $discount->getCode(), $discount->getEmailBody());
    }

    /**
     * {@inheritdoc}
     *
     * @todo: Si potrebbe spostare direttamente in AdminController...
     */
    protected function getActionNotFoundEventName()
    {
        return $this->getEventNamespace('action_not_found');
    }
}
