<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Utilities;

use Application\Core\Mailer\Helper;
use Application\Showcase\Model\Order;
use Pongho\Core\LocalizationInterface;
use Pongho\Template\View;
use Psr\Log\LoggerInterface;
use Symfony\Component\Mailer\Exception\TransportExceptionInterface;
use Symfony\Component\Mailer\Mailer;
use Symfony\Component\Mime\Address;
use Symfony\Component\Mime\Email;

class OrderNotification
{
    /**
     * @var LocalizationInterface
     */
    protected $lang;

    /**
     * @var Mailer
     */
    protected $mailer;

    /**
     * @var Helper
     */
    private $mailerHelper;

    /**
     * Elenco delle variabili da passare al template.
     *
     * Di fatto, il servizio "global_view_vars".
     *
     * @var array
     */
    protected $view_vars;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    public function __construct(
        LocalizationInterface $lang,
        Mailer $mailer,
        Helper $mailerHelper,
        array $view_vars,
        LoggerInterface $logger
    ) {
        $this->lang = $lang;
        $this->mailer = $mailer;
        $this->mailerHelper = $mailerHelper;
        $this->view_vars = $view_vars;
        $this->logger = $logger;
    }

    public function send(Order $order)
    {
        $site = $order->site;

        $userMessageView = new View($site->getThemesPath('email/shop_order_to_user.php'));
        $userMessageView
            ->assignVars($this->view_vars)
            ->assignVars([
                'order'          => $order,
                'order_user_url' => $site->getPermalink() . 'shop/order/' . $order->id . '/',
            ]);

        $adminMessageView = new View($site->getThemesPath('email/shop_order_to_admin.php'));
        $adminMessageView
            ->assignVars($this->view_vars)
            ->assignVars([
                'order'           => $order,
                'order_admin_url' => pongho_url('/shop-orders/view/' . $order->id . '/?site=' . $site->id),
                'user_admin_url'  => pongho_url('/users/edit/' . $order->customer_id . '/?site=' . $site->id),
            ]);

        $customerAddress = new Address(
            $order->getCustomerEmail(),
            $order->getCustomerName()
        );

        $adminAddress = new Address(
            $site->getOption('orders_email', $site->getOption('company_email')),
            $site->getOption('company_name')
        );

        try {
            // Mail al cliente
            $email = (new Email())
                ->from($adminAddress)
                ->to($customerAddress)
                ->subject(
                    sprintf($this->lang->get('email_shop_order_to_user_subject'), $site->getName(), $order->id)
                )
                ->html(
                    $this->mailerHelper->replaceVars(
                        file_get_contents($site->getThemesPath('email/base.php')),
                        [
                            'CONTENT' => $userMessageView->render(),
                        ]
                    )
                );

            $this->mailer->send($this->mailerHelper->prepareEmail($email));

            $this->logger->debug(
                '[SEND ORDER EMAIL] Sent email to user.',
                [
                    'order' => $order->getId(),
                    'from'  => $adminAddress,
                    'to'    => $customerAddress,
                ]
            );
        } catch (TransportExceptionInterface $transportException) {
            $this->logger->error(
                sprintf(
                    '[SEND ORDER EMAIL] Could not be send the order confirmation email to user: "%s"',
                    $transportException->getMessage()
                ),
                [
                    'order'     => $order->getId(),
                    'from'      => $adminAddress,
                    'to'        => $customerAddress,
                    'exception' => $transportException,
                ]
            );
        }

        try {
            // Mail all'amministratore
            $email = (new Email())
                ->from($adminAddress)
                ->to($adminAddress)
                ->replyTo($customerAddress)
                ->subject(
                    sprintf($this->lang->get('email_shop_order_to_admin_subject'), $site->getName(), $order->id)
                )
                ->html(
                    $this->mailerHelper->replaceVars(
                        file_get_contents($site->getThemesPath('email/base.php')),
                        [
                            'CONTENT' => $adminMessageView->render(),
                        ]
                    )
                );

            $this->mailer->send($this->mailerHelper->prepareEmail($email));

            $this->logger->debug(
                '[SEND ORDER EMAIL] Sent email to admin.',
                [
                    'order' => $order->getId(),
                    'from'  => $adminAddress,
                    'to'    => $adminAddress,
                ]
            );
        } catch (TransportExceptionInterface $transportException) {
            $this->logger->error(
                sprintf(
                    '[SEND ORDER EMAIL] Could not be send the order confirmation email to admin: "%s"',
                    $transportException->getMessage()
                ),
                [
                    'order'     => $order->getId(),
                    'from'      => $adminAddress,
                    'to'        => $adminAddress,
                    'exception' => $transportException,
                ]
            );
        }
    }
}
