<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Updates;

use Application\Core\Model\Translation;
use Application\Showcase\Model\OrderDiscount;
use Application\Showcase\Model\OrderDiscountCondition;
use Application\Showcase\Updates\Model\Coupon;
use Application\Showcase\Updates\Model\Order;
use Updates\BaseUpdate;

/**
 * Update52.
 *
 * Sconti sugli ordini, fase 2: Sposto i dati.
 */
class Update52 extends BaseUpdate
{
    /**
     * {@inheritdoc}
     */
    public function commit()
    {
        $map = [];

        /** @var Coupon $coupon */
        foreach (Coupon::all() as $coupon) {
            /** @var OrderDiscount $discount */
            $discount = OrderDiscount::create([
                'handler_name' => 'shop_coupon_discount',
                'code'         => $coupon->code,
                'expire_on'    => $coupon->expire_on,
            ]);

            $map[$coupon->id] = $discount->id;

            OrderDiscountCondition::create([
                'discount_id' => $discount->id,
                'value'       => $coupon->discount,
                'value_type'  => $coupon->discount_type,
                'condition'   => $coupon->min_price,
            ]);
        }

        $options = [
            'conditions' => ['coupon_id IS NOT NULL'],
        ];

        /** @var Order $order */
        foreach (Order::all($options) as $order) {
            $order->discount_id = $map[$order->coupon_id];
            $order->discount_name = $this->getDiscountName($order);
            $order->discount_value = $order->coupon_discount_value;

            $order->save();
        }
    }

    /**
     * {@inheritdoc}
     */
    public function rollback(\Exception $prev)
    {
        foreach (OrderDiscountCondition::all() as $condition) {
            $condition->delete();
        }

        $options = [
            'conditions' => ['coupon_id IS NOT NULL'],
        ];

        /** @var Order $order */
        foreach (Order::all($options) as $order) {
            $order->discount_id = null;
            $order->discount_name = '';
            $order->discount_value = 0;

            $order->save();
        }

        foreach (OrderDiscount::all() as $discount) {
            $discount->delete();
        }
    }

    private function getDiscountName(Order $order)
    {
        static $coupon_label_percent, $coupon_label_fixed;

        if ($coupon_label_percent === null || $coupon_label_fixed === null) {
            $coupon_label_percent = Translation::findByKeyAndSiteAndLanguage('coupon_label_percent', $order->site_id, $order->customer->language_id)->value;
            $coupon_label_fixed = Translation::findByKeyAndSiteAndLanguage('coupon_label_fixed', $order->site_id, $order->customer->language_id)->value;
        }

        if ($order->coupon_discount_type === '%') {
            return sprintf($coupon_label_percent, $order->coupon_discount);
        } else {
            return $coupon_label_fixed;
        }
    }
}
