<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Updates;

use Application\Core\Installer\Helper\TranslationsHelper;
use Updates\BaseUpdate;

/**
 * Update209:
 * - Sconto sulle taglie
 * - Accettazione delle condizioni alla conferma dell’ordine
 */
class Update209 extends BaseUpdate
{
    public function commit()
    {
        $this->fixSizesOfferColumn();
        $this->fixOrdersOfferColumn();
        $this->fixTranslations();
        $this->fixPaymentsTable();

        $this->addMessage('BC BREAK! Devi aggiungere la spunta per accettare le condizioni nella conferma ordine');
        $this->addMessage('Verifica il corretto funzionamento dei metodi di pagamento');
    }

    private function fixSizesOfferColumn()
    {
        $platformName = $this->connection->getDatabasePlatform()->getName();

        $sql = match ($platformName) {
            'mysql'      => <<<SQL
                UPDATE pongho_nodes_sizes AS sizes
                  JOIN pongho_nodes AS products ON products.id = sizes.node_id
                   SET sizes.offer = CASE WHEN products.discount_type = '%'
                       THEN (sizes.price * (100 - products.discount) / 100)
                       ELSE sizes.price - products.discount
                       END
                SQL,
            'postgresql' => <<<SQL
                UPDATE pongho_nodes_sizes AS sizes
                   SET offer = CASE WHEN products.discount_type = '%'
                       THEN (sizes.price * (100 - products.discount) / 100)
                       ELSE sizes.price - products.discount
                       END
                  FROM pongho_nodes AS products 
                 WHERE products.id = sizes.node_id
                SQL,
            default      => throw new \Exception("Database '{$platformName}' is not supported."),
        };

        $this->connection->executeStatement($sql);
    }

    private function fixOrdersOfferColumn()
    {
        $platform = $this->connection->getDatabasePlatform()->getName();
        $schema = $this->connection->createSchemaManager()->introspectSchema();

        if ($schema->getTable('pongho_orders_rows')->hasColumn('realprice')) {
            match ($platform) {
                'mysql' => $this->connection->executeStatement(
                    "ALTER TABLE pongho_orders_rows CHANGE realprice offer DECIMAL(12,4) UNSIGNED NOT NULL DEFAULT '0.0000'",
                ),
                'postgresql' => $this->connection->executeStatement(
                    'ALTER TABLE pongho_orders_rows RENAME COLUMN realprice TO offer',
                ),
                default => throw new \RuntimeException(
                    sprintf('Database platform "%s" is not supported.', $platform),
                ),
            };
        }
    }

    private function fixTranslations()
    {
        (new TranslationsHelper($this->connection))
            ->setCulture('it_IT')
            ->addTranslation('shop_order_conditions', 'Accetto i %stermini e le condizioni di vendita%s.', 'showcase')
            ->addTranslation('shop_order_conditions_error', 'Devi accettare le condizioni per proseguire.', 'showcase');

        (new TranslationsHelper($this->connection))
            ->setCulture('en_US')
            ->addTranslation('shop_order_conditions', 'I accept the %sterms and conditions of sale%s.', 'showcase')
            ->addTranslation('shop_order_conditions_error', 'You must accept the conditions to continue.', 'showcase');
    }

    private function fixPaymentsTable()
    {
        $this->connection->executeQuery('UPDATE pongho_payments SET is_enabled = :enabled', ['enabled' => true]);
    }
}
