<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Updates;

use Application\Core\Installer\Helper\PermissionsHelper;
use Application\Core\Installer\Helper\TranslationsHelper;
use Doctrine\DBAL\ParameterType;
use Updates\BaseUpdate;

/**
 * Update189
 *
 * Nuovi permessi "shop.buy" e "shop.view_orders" e rinominato tutti gli altri aggiungendo il prefisso "shop.admin."
 */
class Update189 extends BaseUpdate
{
    public function commit()
    {
        $this->fixOldPermissions();
        $this->addPermissions();
        $this->addTranslations();
    }

    private function fixOldPermissions()
    {
        $helper = new PermissionsHelper($this->connection);
        $helper->removePermission('shop.admin');

        $rename = [
            'shop.orders'                 => 'shop.admin.orders',
            'shop.payments'               => 'shop.admin.payments.manage',
            'shop.payments.add'           => 'shop.admin.payments.add',
            'shop.payments.delete'        => 'shop.admin.payments.delete',
            'shop.shippings'              => 'shop.admin.shippings.manage',
            'shop.shippings.add'          => 'shop.admin.shippings.add',
            'shop.shippings.delete'       => 'shop.admin.shippings.delete',
            'shop.coupons'                => 'shop.admin.coupons',
            'shop.orders_amount_discount' => 'shop.admin.orders_amount_discount',
            'shop.google_shopping'        => 'shop.admin.google_shopping',
            'shop.sizes'                  => 'shop.admin.sizes',
            'shop.stats'                  => 'shop.admin.stats',
            'shop.stats.sold'             => 'shop.admin.stats.sold',
            'shop.stats.ordered'          => 'shop.admin.stats.ordered',
        ];

        foreach ($rename as $old => $new) {
            $helper->renamePermission($old, $new);
        }
    }

    private function addPermissions()
    {
        $key_field_name = $this->connection->quoteIdentifier('key');
        $roles = [];
        $permissions = ['shop.buy', 'shop.view_orders'];

        foreach ($this->connection->fetchFirstColumn('SELECT * FROM pongho_roles') as $role_id) {
            $roles[] = $role_id;
        }

        foreach ($permissions as $key) {
            try {
                $this->connection->insert(
                    'pongho_permits',
                    [
                        $key_field_name => $key,
                        'is_enabled'    => true,
                    ],
                    [
                        $key_field_name => ParameterType::STRING,
                        'is_enabled'    => ParameterType::BOOLEAN,
                    ]
                );
            } catch (\Exception) {
                // Permesso già inserito, passo oltre...
                continue;
            }

            $permission_id = dbal_last_insert_id($this->connection, 'pongho_permits_id_seq');

            foreach ($roles as $role_id) {
                $this->connection->insert(
                    'pongho_permits_roles',
                    [
                        'permit_id' => $permission_id,
                        'role_id'   => $role_id,
                    ],
                    [
                        'permit_id' => ParameterType::INTEGER,
                        'role_id'   => ParameterType::INTEGER,
                    ]
                );
            }
        }

        // Aggiungo dei permessi che potrebbero non essere stati ancora creati,
        // e proprio per questo evito di associarli ai ruoli.
        (new PermissionsHelper($this->connection))
            ->addPermissions([
                'shop.admin.payments.add',
                'shop.admin.payments.delete',
                'shop.admin.shippings.add',
                'shop.admin.shippings.delete',
            ]);
    }

    private function addTranslations()
    {
        (new TranslationsHelper($this->connection))
            ->changeKey('permit_shop.orders', 'permit_shop.admin.orders')
            ->changeKey('permit_shop.payments', 'permit_shop.admin.payments.manage')
            ->changeKey('permit_shop.payments.add', 'permit_shop.admin.payments.add')
            ->changeKey('permit_shop.payments.delete', 'permit_shop.admin.payments.delete')
            ->changeKey('permit_shop.shippings', 'permit_shop.admin.shippings.manage')
            ->changeKey('permit_shop.shippings.add', 'permit_shop.admin.shippings.add')
            ->changeKey('permit_shop.shippings.delete', 'permit_shop.admin.shippings.delete')
            ->changeKey('permit_shop.coupons', 'permit_shop.admin.coupons')
            ->changeKey('permit_shop.orders_amount_discount', 'permit_shop.admin.orders_amount_discount')
            ->changeKey('permit_shop.google_shopping', 'permit_shop.admin.google_shopping')
            ->changeKey('permit_shop.stats', 'permit_shop.admin.stats')
            ->setCulture('it_IT')
            ->addTranslation('permit_shop.buy', 'Può acquistare dal sito', 'showcase')
            ->addTranslation('permit_shop.view_orders', 'Può visualizzare i propri ordini sul sito', 'showcase')
            ->addTranslation('permit_ns_shop.admin', 'Amministrazione negozio', 'showcase')
            ->addTranslation('permit_shop.admin.orders', 'Può gestire gli ordini', 'showcase')
            ->addTranslation('permit_ns_shop.admin.payments', 'Metodi di pagamento', 'showcase')
            ->addTranslation('permit_shop.admin.payments.manage', 'Gestione', 'showcase')
            ->addTranslation('permit_shop.admin.payments.add', 'Aggiunta', 'showcase')
            ->addTranslation('permit_shop.admin.payments.delete', 'Eliminazione', 'showcase')
            ->addTranslation('permit_ns_shop.admin.shippings', 'Metodi di spedizione', 'showcase')
            ->addTranslation('permit_shop.admin.shippings.manage', 'Gestione', 'showcase')
            ->addTranslation('permit_shop.admin.shippings.add', 'Aggiunta', 'showcase')
            ->addTranslation('permit_shop.admin.shippings.delete', 'Eliminazione', 'showcase')
            ->addTranslation('permit_shop.admin.coupons', 'Può gestire i buoni sconto', 'showcase')
            ->addTranslation('permit_shop.admin.orders_amount_discount', 'Può gestire gli sconti in base all’importo degli ordini', 'showcase')
            ->addTranslation('permit_shop.admin.google_shopping', 'Può gestire il feed di Google Shopping', 'showcase')
            ->addTranslation('permit_shop.admin.sizes', 'Può gestire i tipi di taglia', 'showcase')
            ->addTranslation('permit_shop.admin.stats', 'Può visualizzare le statistiche', 'showcase')
        ;
    }
}
