<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Updates;

use Doctrine\DBAL\ParameterType;
use Pongho\Utilities\DateTime;
use Updates\BaseUpdate;

/**
 * Update109.
 *
 * Popola le nuove colonne per tener traccia dei prodotti legati ai carrelli, ai pagamenti e agli ordini da evadere.
 */
class Update109 extends BaseUpdate
{
    private $products = [];

    private $sizes = [];

    /**
     * {@inheritdoc}
     */
    public function commit()
    {
        $this->populate('product');
        $this->populate('size');
        $this->populateProductAvailabilities();

        $types = [
            'availabilities'              => ParameterType::INTEGER,
            'bound_in_cart'               => ParameterType::INTEGER,
            'bound_in_cart_updated_at'    => 'datetime',
            'bound_in_payment'            => ParameterType::INTEGER,
            'bound_in_payment_updated_at' => 'datetime',
            'bound_in_order'              => ParameterType::INTEGER,
        ];

        foreach ($this->products as $id => $data) {
            if (isset($data['bound_in_cart_updated_at'])) {
                $data['bound_in_cart_updated_at'] = new \DateTime($data['bound_in_cart_updated_at']);
            }

            if (isset($data['bound_in_payment_updated_at'])) {
                $data['bound_in_payment_updated_at'] = new \DateTime($data['bound_in_payment_updated_at']);
            }

            $this->connection->update('pongho_nodes', $data, ['id' => $id], $types);
        }

        foreach ($this->sizes as $id => $data) {
            if (isset($data['bound_in_cart_updated_at'])) {
                $data['bound_in_cart_updated_at'] = new \DateTime($data['bound_in_cart_updated_at']);
            }

            if (isset($data['bound_in_payment_updated_at'])) {
                $data['bound_in_payment_updated_at'] = new \DateTime($data['bound_in_payment_updated_at']);
            }

            $this->connection->update('pongho_nodes_sizes', $data, ['id' => $id], $types);
        }
    }

    /**
     * @param string $type
     */
    private function populate($type)
    {
        $method = 'populate' . ucfirst($type);
        $field = $type . '_id';

        $query = <<<SQL
SELECT r.{$field}, SUM(r.quantity) AS bound_in_cart, MAX(r.updated_at) AS bound_in_cart_updated_at
FROM pongho_orders_rows AS r
  INNER JOIN pongho_orders AS o ON o.id = r.order_id
WHERE o.status = 'cart' AND r.updated_at > ?
GROUP BY r.{$field}
SQL;

        $statement = $this->connection->prepare($query);
        $statement->bindValue(1, new DateTime('1 hour ago'), 'datetime');

        foreach ($statement->executeQuery()->fetchAllAssociative() as $row) {
            $this->$method($row->$field, 'bound_in_cart', $row['bound_in_cart']);
            $this->$method($row->$field, 'bound_in_cart_updated_at', $row['bound_in_cart_updated_at']);
        }

        $query = <<<SQL
SELECT r.{$field}, SUM(r.quantity) AS bound_in_payment, MAX(r.updated_at) AS bound_in_payment_updated_at
FROM pongho_orders_rows AS r
  INNER JOIN pongho_orders AS o ON o.id = r.order_id
WHERE o.status IN ('payment', 'transaction') AND r.updated_at > ?
GROUP BY r.{$field}
SQL;

        $statement = $this->connection->prepare($query);
        $statement->bindValue(1, new DateTime('2 hour ago'), 'datetime');

        foreach ($statement->executeQuery()->fetchAllAssociative() as $row) {
            $this->$method($row->$field, 'bound_in_payment', $row['bound_in_payment']);
            $this->$method($row->$field, 'bound_in_payment_updated_at', $row['bound_in_payment_updated_at']);
        }

        $query = <<<SQL
SELECT r.{$field}, SUM(r.quantity) AS bound_in_order
FROM pongho_orders_rows AS r
  INNER JOIN pongho_orders AS o ON o.id = r.order_id
WHERE o.status IN ('order', 'pending', 'payment_authorized')
GROUP BY r.{$field}
SQL;

        foreach ($this->connection->fetchAllAssociative($query) as $row) {
            $this->$method($row->$field, 'bound_in_order', $row['bound_in_order']);
        }
    }

    private function populateProductAvailabilities()
    {
        // Ricavo i tipi di nodo legati allo showcase
        $query_node_types = <<<SQL
SELECT ms.id
FROM pongho_modules_sites AS ms
  INNER JOIN pongho_modules AS m ON m.id = ms.module_id
WHERE m.controller = 'Application\\\\Showcase\\\\Controller\\\\NodeController'
SQL;

        $sth_node_types = $this->connection->executeQuery($query_node_types);

        $node_types = [];
        foreach ($sth_node_types->fetchFirstColumn() as $id) {
            $node_types[] = (int) $id;
        }

        // Ricavo i prodotti
        if ($node_types !== []) {
            $placeholders = rtrim(str_repeat('?, ', count($node_types)), ', ');
            $sth_products = $this->connection->executeQuery(
                "SELECT id, size_type_id FROM pongho_nodes WHERE module_site_id IN ($placeholders)",
                $node_types
            );

            // Per ogni prodotto sommo le disponibilità delle taglie
            foreach ($sth_products->fetchAllAssociative() as $product) {
                $size_type_id = (int) $product['bound_in_order'];

                if ($size_type_id) {
                    $query = <<<SQL
SELECT SUM(s.availabilities)
FROM pongho_nodes_sizes AS s
  INNER JOIN pongho_nodes_sizes_names AS sn ON sn.id = s.size_name_id
WHERE s.node_id = ? AND sn.size_type_id = ?
SQL;
                    $params = [$product['bound_in_order'], $size_type_id];
                } else {
                    $query = 'SELECT SUM(availabilities) FROM pongho_nodes_sizes WHERE node_id = ?';
                    $params = [$product['id']];
                }

                $sth = $this->connection->executeQuery($query, $params);

                foreach ($sth->fetchFirstColumn() as $availabilities) {
                    $this->populateProduct($product['id'], 'availabilities', $availabilities);
                }
            }
        }
    }

    /**
     * @param int    $product_id
     * @param string $field
     * @param mixed  $value
     */
    private function populateProduct($product_id, $field, $value)
    {
        if (!isset($this->products[$product_id])) {
            $this->products[$product_id] = [];
        }

        $this->products[$product_id][$field] = $value;
    }

    /**
     * @param int    $size_id
     * @param string $field
     * @param mixed  $value
     */
    private function populateSize($size_id, $field, $value)
    {
        if (!isset($this->sizes[$size_id])) {
            $this->sizes[$size_id] = [];
        }

        $this->sizes[$size_id][$field] = $value;
    }
}
