<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Tests\WebTracking\GoogleAnalytics;

use Application\Showcase\Entity\OrderInterface;
use Application\Showcase\Entity\OrderRowInterface;
use Application\Showcase\WebTracking\GoogleAnalytics\Ecommerce;
use PHPUnit\Framework\MockObject\MockObject;

class EcommerceTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @var Ecommerce
     */
    protected $ga;

    protected function setUp(): void
    {
        $this->ga = new Ecommerce();
    }

    public function testRender(): void
    {
        $expected = <<<JS
(function(){ga('require','ecommerce');ga('ecommerce:addTransaction',{"id":null,"revenue":0,"shipping":0,"tax":0});ga('ecommerce:addItem',{"id":123,"name":"White milk","sku":"SKU123","price":12.34,"quantity":2,"category":"Milk"});ga('ecommerce:addItem',{"id":123,"name":"Pink milk","sku":"FOO456","price":72.23,"quantity":1});ga('ecommerce:addItem',{"id":123,"name":"Black milk","sku":"BAR789","price":56,"quantity":1,"category":"Milk"});ga('ecommerce:send');})();
JS;

        $this->assertEquals($expected, $this->ga->render($this->createOrderMock()));
    }

    public function testTaxonomyDifferentFromCategorySupport(): void
    {
        $expected = <<<JS
(function(){ga('require','ecommerce');ga('ecommerce:addTransaction',{"id":null,"revenue":0,"shipping":0,"tax":0});ga('ecommerce:addItem',{"id":123,"name":"White milk","sku":"SKU123","price":12.34,"quantity":2,"category":"Milk"});ga('ecommerce:addItem',{"id":123,"name":"Pink milk","sku":"FOO456","price":72.23,"quantity":1});ga('ecommerce:addItem',{"id":123,"name":"Black milk","sku":"BAR789","price":56,"quantity":1,"category":"Milk"});ga('ecommerce:send');})();
JS;

        $this->assertEquals($expected, $this->ga->render($this->createOrderMock('brand'), 'brand'));
    }

    public function testAffiliationSupport(): void
    {
        $expected = <<<JS
(function(){ga('require','ecommerce');ga('ecommerce:addTransaction',{"id":null,"revenue":0,"shipping":0,"tax":0,"affiliation":"myAffiliation"});ga('ecommerce:addItem',{"id":123,"name":"White milk","sku":"SKU123","price":12.34,"quantity":2,"category":"Milk"});ga('ecommerce:addItem',{"id":123,"name":"Pink milk","sku":"FOO456","price":72.23,"quantity":1});ga('ecommerce:addItem',{"id":123,"name":"Black milk","sku":"BAR789","price":56,"quantity":1,"category":"Milk"});ga('ecommerce:send');})();
JS;

        $this->assertEquals($expected, $this->ga->render($this->createOrderMock(), 'category', 'myAffiliation'));
    }

    public function testMultipleTrackerSupport(): void
    {
        $expected = <<<JS
(function(){ga('myTracker.require','ecommerce');ga('myTracker.ecommerce:addTransaction',{"id":null,"revenue":0,"shipping":0,"tax":0});ga('myTracker.ecommerce:addItem',{"id":123,"name":"White milk","sku":"SKU123","price":12.34,"quantity":2,"category":"Milk"});ga('myTracker.ecommerce:addItem',{"id":123,"name":"Pink milk","sku":"FOO456","price":72.23,"quantity":1});ga('myTracker.ecommerce:addItem',{"id":123,"name":"Black milk","sku":"BAR789","price":56,"quantity":1,"category":"Milk"});ga('myTracker.ecommerce:send');})();
JS;

        $this->assertEquals($expected, $this->ga->render($this->createOrderMock(), 'category', null, 'myTracker'));
    }

    /**
     * @param string $taxonomy_name
     */
    private function createOrderMock($taxonomy_name = 'category'): OrderInterface|MockObject
    {
        $orderId = 123;

        $rows = [
            $this->createRowMock($orderId, 'SKU123', 'White milk', 12.34, 2, 'Milk', $taxonomy_name),
            $this->createRowMock($orderId, 'FOO456', 'Pink milk', 72.23, 1, '', $taxonomy_name),
            $this->createRowMock($orderId, 'BAR789', 'Black milk', 56, 1, 'Milk', $taxonomy_name),
        ];

        $order = $this->createMock(OrderInterface::class);
        $order->expects($this->any())->method('getRows')->willReturn($rows);

        return $order;
    }

    /**
     * @param int    $orderId
     * @param string $sku
     * @param string $title
     * @param float  $price
     * @param int    $quantity
     * @param string $category_name
     * @param string $taxonomy_name
     */
    private function createRowMock($orderId, $sku, $title, $price, $quantity, $category_name = null, $taxonomy_name = 'category'): OrderRowInterface|MockObject
    {
        $row = $this->createMock(OrderRowInterface::class);

        $row->expects($this->any())->method('getOrderId')->willReturn($orderId);
        $row->expects($this->any())->method('sku')->willReturn($sku);
        $row->expects($this->any())->method('title')->willReturn($title);
        $row->expects($this->any())->method('price')->willReturn($price);
        $row->expects($this->any())->method('quantity')->willReturn($quantity);

        $category = $this->getMockBuilder(\Application\Cms\Model\Term::class)
            ->disableOriginalConstructor()
            ->setMethods(['title'])
            ->getMock();

        $product = $this->getMockBuilder(\Application\Showcase\Model\Node::class)
            ->disableOriginalConstructor()
            ->setMethods(['getTerm'])
            ->getMock();

        if ($category_name) {
            $category->expects($this->any())->method('title')->willReturn($category_name);
            $product->expects($this->any())->method('getTerm')->with($taxonomy_name)->willReturn($category);
        } else {
            $product->expects($this->any())->method('getTerm')->with($taxonomy_name)->willReturn(null);
        }

        $row->expects($this->any())->method('getProduct')->willReturn($product);

        return $row;
    }
}
