<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Stats\Report;

use Application\Admin\Widgets\Stats\ReportInterface;
use Application\Cms\Model\NodeTranslation;
use Application\Showcase\Model\Order;
use Application\Showcase\Model\OrderRow;
use Pongho\Utilities\DateTime;

/**
 * TopProductsReport
 */
class TopSoldProductsReport implements ReportInterface
{
    public const TYPE_SOLD = 'sold';

    public const TYPE_ORDERED = 'ordered';

    /**
     * @var int
     */
    private $language_id;

    /**
     * @var DateTime
     */
    protected $start_date;

    /**
     * @var DateTime
     */
    protected $end_date;

    /**
     * @var string
     */
    protected $type;

    /**
     * @var string
     */
    protected $sort_order = 'DESC';

    /**
     * @param int      $language_id
     * @param string   $type
     * @param string   $sort_order
     */
    public function __construct($language_id, DateTime $start_date, DateTime $end_date, $type = self::TYPE_SOLD, $sort_order = 'DESC')
    {
        if (!in_array($type, [self::TYPE_SOLD, self::TYPE_ORDERED])) {
            throw new \InvalidArgumentException(
                sprintf(
                    'Type "%s" is not supported. You must use only these: %s',
                    $type,
                    implode(', ', [self::TYPE_SOLD, self::TYPE_ORDERED])
                )
            );
        }

        if (!in_array($sort_order, ['ASC', 'DESC'])) {
            throw new \InvalidArgumentException(sprintf('The "%s" sort order is not valid.', $sort_order));
        }

        $this->language_id = $language_id;
        $this->start_date = $start_date;
        $this->end_date = $end_date;
        $this->type = $type;
        $this->sort_order = $sort_order;
    }

    /**
     * {@inheritdoc}
     */
    public function getStartDate()
    {
        return $this->start_date;
    }

    /**
     * {@inheritdoc}
     */
    public function getEndDate()
    {
        return $this->end_date;
    }

    /**
     * @return string
     */
    public function getSortOrder()
    {
        return $this->sort_order;
    }

    /**
     * {@inheritdoc}
     */
    public function getData()
    {
        $data = [];

        if ($this->type === self::TYPE_ORDERED) {
            $date_field = 'ordered_at';
            $status = Order::$ORDER_STATUSES;
        } else {
            $date_field = 'paid_at';
            $status = [Order::STATUS_PROCESSED];
        }

        $options = [
            'select'     => '`from`.product_id, pt.title, SUM(`from`.quantity) AS products_count, pt.permalink',
            'joins'      => 'INNER JOIN ' . Order::tableName() . ' AS o ON o.id = `from`.order_id'
                         . ' INNER JOIN ' . NodeTranslation::tableName() . ' AS pt ON pt.node_id = `from`.product_id',
            'conditions' => [
                "pt.language_id = :language AND (o.{$date_field} BETWEEN :start AND :end) AND status IN ('" . implode("', '", $status) . "')",
                'language' => $this->language_id,
                'start'    => $this->start_date,
                'end'      => $this->end_date,
            ],
            'group'      => '`from`.product_id, pt.title, pt.permalink',
            'order'      => 'SUM(`from`.quantity) ' . $this->sort_order,
            'limit'      => 10,
        ];

        foreach (OrderRow::all($options) as $stat) {
            $data[] = [
                'x' => [(string) $stat->title],
                'y' => [(int) $stat->products_count],
            ];
        }

        return $data;
    }
}
