<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Payment\VirtualPay;

use Application\Admin\Form\FormConfig;
use Application\Core\I18n\Translation\Translator;
use Application\Showcase\Model\Order;
use Application\Showcase\Payment\BasePayment;
use Application\Showcase\Payment\ListenerResponse;
use Application\Showcase\Payment\PaymentFailResponse;
use Application\Showcase\Payment\PaymentOptions;
use Application\Showcase\Payment\PaymentResponse;
use Application\Showcase\Payment\TicketResponse;
use Pongho\Form\Field\CheckboxField;
use Pongho\Form\Field\TextField;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\Exception\HttpUnauthorizedException;
use Pongho\Http\RedirectResponse;
use Pongho\Http\Request;
use Pongho\Http\Response;
use Pongho\Utilities\DateTime;

/**
 * Gestisce il pagamento tramite Virtual Pay.
 */
class Payment extends BasePayment
{
    /**
     * {@inheritdoc}
     */
    public function preparePayment()
    {
        $newStatus = Order::STATUS_PAYMENT;

        $this->getLogger()->debug(
            '[VirtualPay] Prepare order.',
            [
                'order'       => $this->order->getId(),
                'prev_status' => $this->order->status,
                'new_status'  => $newStatus,
            ]
        );

        $this->order->convertCartToOrder($newStatus);

        // Tentativo
        $attempt = $this->getOrderPaymentDetail('attempt', 0);
        $attempt++;
        $this->setOrderPaymentDetail('attempt', $attempt);

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function handlePayment(PaymentOptions $options)
    {
        return new PaymentResponse(
            $this->buildPaymentResponse($options)
        );
    }

    /**
     * {@inheritdoc}
     */
    public function handleCancel($redirect_url)
    {
        $this->order->status = Order::STATUS_CART;

        $response = new PaymentFailResponse(new RedirectResponse($redirect_url));

        $options = $this->order->payment->getHandlerOptions();
        if ($options['notify_on_fail']) {
            $response->setSendNotifyToAdmin(true);
        }

        return $response;
    }

    /**
     * {@inheritdoc}
     */
    public function handleListener(Request $request, PaymentOptions $options)
    {
        $logger = $this->getLogger();

        $logger->info('[VirtualPay] Listener request', ['query' => $request->query->all()]);

        $success = (bool) $request->query->get('success');
        $transactionId = $request->query->get('TRANSACTION_ID');
        $merchantId = $request->query->get('MERCHANT_ID');
        $paymentOrderId = $request->query->get('ORDER_ID');
        $authCode = $request->query->get('COD_AUT');
        $orderAmount = $request->query->get('IMPORTO');
        $currency = $request->query->get('DIVISA');
        $MAC = $request->query->get('MAC');

        $orderIdSegments = explode('-', (string) $paymentOrderId);
        if (isset($orderIdSegments[0]) && $orderIdSegments[0] === 'PONGHO') {
            $orderId = $orderIdSegments[1] ?? null;
        } else {
            $orderId = $orderIdSegments[0] ?? null;
        }

        if (!$orderId) {
            $logger->error("[VirtualPay] The order '{$orderId}' is not valid.");

            throw new HttpNotFoundException();
        }

        $this->order = Order::find($orderId);

        if (!$this->order) {
            $logger->error("[VirtualPay] The order '{$orderId}' does not exist.");

            throw new HttpNotFoundException();
        }

        $orderOptions = $options->get($this->order);
        $expectedMAC = strtoupper(
            hash(
                'sha256',
                $transactionId . $merchantId . $paymentOrderId . $authCode . $orderAmount . $currency . $orderOptions['MAC_key']
            )
        );

        if ($expectedMAC !== $MAC) {
            $logger->error(
                "[VirtualPay] The MAC of the request '{$MAC}' does not match the expected MAC '{$expectedMAC}'."
            );

            throw new HttpUnauthorizedException();
        }

        if ($success) {
            $logger->info("[VirtualPay] The payment of the order '{$orderId}' was successful.");

            $this->order->status = Order::STATUS_ORDER;
            $this->order->paid_at = new DateTime();

            return new ListenerResponse(true);
        } else {
            $logger->info("[VirtualPay] The payment of the order '{$orderId}' has failed.");

            $this->order->status = Order::STATUS_PAYMENT_FAILED;

            $response = new ListenerResponse(false);

            $options = $this->order->payment->getHandlerOptions();
            if ($options['notify_on_fail']) {
                $response->setSendFailedNotification(true);
            }

            return $response;
        }
    }

    /**
     * {@inheritdoc}
     */
    public function handleTicket()
    {
        return new TicketResponse();
    }

    /**
     * {@inheritdoc}
     */
    public static function getFormFieldsConfig(FormConfig $config, Translator $translator, array $options = [])
    {
        return [
            'merchant_id'    => [
                'class' => TextField::class,
                'label' => 'MERCHANT ID',
            ],
            'ABI'            => [
                'class' => TextField::class,
                'label' => 'ABI',
            ],
            'MAC_key'        => [
                'class' => TextField::class,
                'label' => 'Chiave MAC',
            ],
            'notify_on_fail' => [
                'class'       => CheckboxField::class,
                'label'       => $translator->trans('Payment failed'),
                'description' => $translator->trans('Send a notification when the payment fails'),
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public static function getDefaultOptions()
    {
        return [
            'merchant_id'    => '',
            'ABI'            => '',
            'MAC_key'        => '',
            'notify_on_fail' => false,
        ];
    }

    /**
     * @return \Psr\Log\LoggerInterface
     */
    private function getLogger()
    {
        return $this->getContainer()->get('logger');
    }

    /**
     * @return Response
     */
    private function buildPaymentResponse(PaymentOptions $options)
    {
        $orderOptions = $options->get($this->order);

        $attempt = $this->getOrderPaymentDetail('attempt', 0);

        $orderId = ($this->isDebug() ? 'PONGHO-' . $this->order->getId() : $this->order->getId()) . '-' . $attempt;
        $orderAmount = number_format($this->order->total(), 2, ',', '');

        $items = '';
        foreach ($this->order->rows() as $row) {
            $product = $row->getProduct();
            $productName = str_replace('"', '', $product->title);
            $items .= $product->getId() . '^' . $productName . '^' . $row->quantity() . '^' . number_format($row->total(), 2, ',', '') . '^EUR;';
        }

        $MAC = strtoupper(
            hash(
                'sha256',
                $orderOptions['merchant_id'] . $orderId . $orderAmount . 'EUR' . $orderOptions['ABI'] . $items . $orderOptions['MAC_key']
            )
        );

        $content = <<<HTML
<!DOCTYPE html>
<html lang="en">
<head>
	<meta charset="UTF-8">
	<title>Virtual Pay</title>
</head>
<body>

<div>
	<h1>Reindirizzamento alla banca</h1>
	<p>Stai per essere reindirizzato al sito della banca per completare il pagamento.</p>
	<div>
		LOADING...
	</div>
</div>

<form action="https://www.servizipos.it/ems-elk-web/pages/ems_elink_payment.xhtml" method="post" name="form" style="margin-top: 1rem;">
    <div style="display: none;">
        <input type="text" name="MERCHANT_ID" value="{$orderOptions['merchant_id']}">
        <input type="text" name="ORDER_ID" value="{$orderId}">
        <input type="text" name="IMPORTO" value="{$orderAmount}">
        <input type="text" name="DIVISA" value="EUR">
        <input type="text" name="ABI" value="{$orderOptions['ABI']}">
        <input type="text" name="SEPARATORI" value=";^">
        <input type="text" name="ITEMS" value="{$items}">
        <input type="text" name="EMAIL" value="{$this->order->getCustomerEmail()}">
        <input type="text" name="LINGUA" value="ita">
        <input type="text" name="URLOK" value="{$orderOptions['ticket_url']}">
        <input type="text" name="URLKO" value="{$orderOptions['cancel_url']}">
        <input type="text" name="URLACK" value="{$orderOptions['listener_url']}&amp;success=1">
        <input type="text" name="URLNACK" value="{$orderOptions['listener_url']}&amp;success=0">
        <input type="text" name="MAC" value="{$MAC}">
    </div>

    <button type="submit">Procedi sul sito della banca</button>
</form>

<script>document.form.submit();</script>

</body>
</html>
HTML;

        return new Response($content);
    }

    private function getOrderPaymentDetail($key, $default = null)
    {
        $orderPaymentDetails = $this->getOrderPaymentDetails();

        if (array_key_exists($key, $orderPaymentDetails)) {
            return $orderPaymentDetails[$key];
        }

        return $default;
    }

    private function setOrderPaymentDetail($key, $value)
    {
        $orderPaymentDetails = $this->getOrderPaymentDetails();

        $orderPaymentDetails[$key] = $value;

        $this->order->payment_details = serialize($orderPaymentDetails);
    }

    private function getOrderPaymentDetails()
    {
        $orderPaymentDetails = unserialize($this->order->payment_details);

        if (!$orderPaymentDetails) {
            $orderPaymentDetails = [];
        }

        return $orderPaymentDetails;
    }
}
