<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Payment\Unicredit;

use Application\Admin\Form\FormConfig;
use Application\Core\I18n\Translation\Translator;
use Application\Core\Localization;
use Application\Showcase\Model\Order;
use Application\Showcase\Payment\BasePayment;
use Application\Showcase\Payment\PaymentFailResponse;
use Application\Showcase\Payment\PaymentOptions;
use Application\Showcase\Payment\PaymentResponse;
use Application\Showcase\Payment\TicketResponse;
use Guzzle\Http\Client;
use Pongho\Form\Field\RadioField;
use Pongho\Form\Field\TextField;
use Pongho\Http\RedirectResponse;
use Pongho\Http\Request;

/**
 * Gestisce il pagamento tramite il gateway di Unicredit (PagOnline).
 */
class Payment extends BasePayment
{
    /**
     * {@inheritdoc}
     */
    public function preparePayment()
    {
        $this->order->convertCartToOrder(Order::STATUS_PAYMENT);

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function handlePayment(PaymentOptions $options)
    {
        $options = $options->get($this->order);

        return new PaymentResponse(new RedirectResponse($this->getPaymentUrl($options)));
    }

    /**
     * {@inheritdoc}
     */
    public function handleCancel($redirect_url)
    {
        $this->order->status = Order::STATUS_CART;
        $this->order->payment_details = serialize(['flagRiciclaOrdine' => 'Y']);

        return new PaymentFailResponse(new RedirectResponse($redirect_url));
    }

    /**
     * {@inheritdoc}
     */
    public function handleListener(Request $request, PaymentOptions $options)
    {
        /** @var Order $order */
        if (!isset($_GET['numeroOrdine']) || ($order = Order::find($_GET['numeroOrdine'])) === null) {
            throw new \RuntimeException('Order not exists!');
        }

        if ($order->payment->handler_class !== self::class) {
            throw new \RuntimeException(sprintf('Order not found with %s handler.', self::class));
        }

        $this->setOrder($order);
        $listener = new Listener($order, $this->isDebug());

        return $listener->handle($request, $options);
    }

    /**
     * {@inheritdoc}
     */
    public function handleTicket()
    {
        return new TicketResponse();
    }

    /**
     * {@inheritdoc}
     */
    public static function getFormFieldsConfig(FormConfig $config, Translator $translator, array $options = [])
    {
        return [
            'numeroCommerciante' => [
                'class' => TextField::class,
                'label' => 'Merchant ID',
            ],
            'stabilimento'       => [
                'class' => TextField::class,
                'label' => 'Stabilimento',
            ],
            'userID'             => [
                'class' => TextField::class,
                'label' => $translator->trans('User'),
            ],
            'password'           => [
                'class' => TextField::class,
                'label' => $translator->trans('Password'),
            ],
            'secret_string'      => [
                'class' => TextField::class,
                'label' => 'Secret String',
            ],
            'flagDeposito'       => [
                'class'   => RadioField::class,
                'label'   => $translator->trans('Transaction mode'),
                'options' => [
                    'Y' => $translator->trans('Instant purchase'),
                    'N' => $translator->trans('Authorized purchase'),
                ],
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public static function getDefaultOptions()
    {
        return [
            'numeroCommerciante' => '',
            'stabilimento'       => '',
            'userID'             => '',
            'password'           => '',
            'secret_string'      => '',
            'flagDeposito'       => 'Y',
        ];
    }

    /**
     * Restituisce l’indirizzo per il pagamento dell’ordine.
     *
     * @throws \OutOfBoundsException
     * @throws \InvalidArgumentException
     * @return string
     *
     * @todo   Traduzione del titolo dell’ordine.
     * @todo   Internazionalizzare il parametro `langCompratore`.
     */
    protected function getPaymentUrl(array $options = [])
    {
        if (!array_key_exists('ticket_url', $options)) {
            throw new \InvalidArgumentException('The "ticket_url" option not exists!');
        }

        if (!array_key_exists('cancel_url', $options)) {
            throw new \InvalidArgumentException('The "cancel_url" option not exists!');
        }

        $order = $this->getOrder();

        if ($options['numeroCommerciante'] == '') {
            throw new \OutOfBoundsException('You must set "Merchant ID" option for proceed to UniCredit payment!');
        }

        if ($options['stabilimento'] == '') {
            throw new \OutOfBoundsException('You must set "Stabilimento" option for proceed to UniCredit payment!');
        }

        if ($options['userID'] == '') {
            throw new \OutOfBoundsException('You must set "Utente" option for proceed to UniCredit payment!');
        }

        if ($options['password'] == '') {
            throw new \OutOfBoundsException('You must set "Password" option for proceed to UniCredit payment!');
        }

        if ($options['secret_string'] == '') {
            throw new \OutOfBoundsException('You must set "Secret String" option for proceed to UniCredit payment!');
        }

        $details = unserialize($this->order->payment_details);
        $flagRiciclaOrdine = is_array(
            $details
        ) && isset($details['flagRiciclaOrdine']) ? $details['flagRiciclaOrdine'] : '';

        $parameters = [
            'numeroCommerciante' => $options['numeroCommerciante'],
            'userID'             => $options['userID'],
            'password'           => $options['password'],
            'numeroOrdine'       => $order->id,
            'totaleOrdine'       => Utilities::formatPrice($order->total()),
            'valuta'             => 978,
            'flagDeposito'       => $options['flagDeposito'],
            'urlOk'              => $options['ticket_url'],
            'urlKo'              => $options['cancel_url'],
            'tipoRispostaApv'    => 'click',
            'flagRiciclaOrdine'  => $flagRiciclaOrdine ?: 'N',
            'stabilimento'       => $options['stabilimento'],
        ];

        $parameters['mac'] = Mac::get($parameters, $options['secret_string']);
        $parameters['password'] = '';
        $parameters['tipoPagamento'] = 'CartCred';
        $parameters['causalePagamento'] = sprintf('Ordine n° %d', $order->id);
        $parameters['emailCompratore'] = $order->getCustomerEmail();
        $parameters['langCompratore'] = 'it';

        return 'http://pagamenti.unicredito.it/initInsert.do?' . http_build_query($parameters, '', '&');
    }

    /**
     * {@inheritdoc}
     */
    public function supportGatewayDetails()
    {
        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function getOrderStatus()
    {
        $details = $this->getOrderDetails();

        /**
         * ON: Ordine inserito nuovo
         * RO: Ordine inserito richiesto
         * IN: Ordine inserito in attesa di input del cliente
         * AR: Ordine in attesa di risposta dal circuito autorizzativo
         * OK: Ordine autorizzato e in attesa di conferma
         * KO: Ordine non autorizzato
         * EX: Ordine scaduto
         * ST: Ordine stornato prima della contabilizzazione
         * IP: Ordine in rimborso parziale
         * IT: Ordine in rimborso totale
         * CP: Ordine rimborsato parzialmente
         * CT: Ordine rimborsato totalmente
         * RC: Rimborso cancellato
         * IC: Ordine confermato
         * CO: Ordine contabilizzato
         * AB: Ordine abbandonato
         */

        return match ($details['status_code']) {
            'ON', 'RO', 'IN', 'AR'             => Order::STATUS_PAYMENT,
            'OK'                               => Order::STATUS_PAYMENT_AUTHORIZED,
            'KO', 'EX'                         => Order::STATUS_PAYMENT_FAILED,
            'ST', 'IP', 'IT', 'CP', 'CT', 'RC' => Order::STATUS_INVOICE_CANCELED,
            'IC', 'CO'                         => Order::STATUS_ORDER,
            'AB'                               => Order::STATUS_CART,
            default                            => throw new \OutOfRangeException(
                sprintf('Lo stato %s non è supportato!', $details['status']),
            ),
        };
    }

    /**
     * {@inheritdoc}
     */
    public function getOrderDetails()
    {
        $client = new Client('https://pagamenti.unicredito.it/backoffice/servizi/execute_remote_command.do');

        $options = $this->order->payment->getHandlerOptions();

        $parameters = [
            'numeroCommerciante' => $options['numeroCommerciante'],
            'stabilimento'       => $options['stabilimento'],
            'userID'             => $options['userID'],
            'password'           => $options['password'],
            'tipoComando'        => 'ORDER_STATE',
            'formatoRisposta'    => 'plaintext',
            'numeroOrdine'       => $this->order->id,
        ];

        $parameters['mac'] = Mac::get($parameters, $options['secret_string']);
        $parameters['password'] = '';

        $request = $client->get(null, null, ['query' => $parameters]);
        $body = trim((string) $request->send()->getBody());

        $data = [];
        parse_str($body, $data);

        $response = [
            'status_code' => $data['STATE'],
        ];

        $messages = [
            'ON' => 'Ordine inserito nuovo',
            'RO' => 'Ordine inserito richiesto',
            'IN' => 'Ordine inserito in attesa di input del cliente',
            'AR' => 'Ordine in attesa di risposta dal circuito autorizzativo',
            'OK' => 'Ordine autorizzato e in attesa di conferma',
            'KO' => 'Ordine non autorizzato',
            'IC' => 'Ordine confermato',
            'CO' => 'Ordine contabilizzato',
            'AB' => 'Ordine abbandonato',
            'ST' => 'Ordine stornato prima della contabilizzazione',
            'EX' => 'Ordine scaduto',
            'IP' => 'Ordine in rimborso parziale',
            'IT' => 'Ordine in rimborso totale',
            'CP' => 'Ordine rimborsato parzialmente',
            'CT' => 'Ordine rimborsato totalmente',
            'RC' => 'Rimborso cancellato',
        ];

        if (array_key_exists($data['STATE'], $messages)) {
            $response['status'] = sprintf('%s (%s)', $data['STATE'], $messages[$data['STATE']]);
        }

        $response['PAYMENTTYPE'] = $data['PAYMENTTYPE'];
        $response['AMOUNT'] = format_price($data['AMOUNT'] / 100);
        $response['DEPOSITAMOUNT'] = format_price($data['DEPOSITAMOUNT'] / 100);
        $response['APPROVEAMOUNT'] = format_price($data['APPROVEAMOUNT'] / 100);
        $response['TIMECREATED'] = $data['TIMECREATED'];
        $response['TIMEMODIFIED'] = $data['TIMEMODIFIED'];

        return $response;
    }
}
