<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Payment\Soisy;

use Application\Admin\Form\FormConfig;
use Application\Core\I18n\Translation\Translator;
use Application\Core\Model\Province;
use Application\Showcase\Model\Order;
use Application\Showcase\Payment\BasePayment;
use Application\Showcase\Payment\ListenerResponse;
use Application\Showcase\Payment\PaymentFailResponse;
use Application\Showcase\Payment\PaymentOptions;
use Application\Showcase\Payment\PaymentResponse;
use Application\Showcase\Payment\TicketResponse;
use GuzzleHttp\Client;
use OutOfBoundsException;
use Pongho\Form\Field\CheckboxField;
use Pongho\Form\Field\TextField;
use Pongho\Http\Exception\HttpException;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\RedirectResponse;
use Pongho\Http\Request;
use Pongho\Utilities\DateTime;
use RuntimeException;

class Payment extends BasePayment
{
    public function preparePayment()
    {
        $this->order->convertCartToOrder(Order::STATUS_PAYMENT_LOCKED);

        return $this;
    }

    public function handlePayment(PaymentOptions $options)
    {
        $options = $options->get($this->order);

        return new PaymentResponse(new RedirectResponse($this->getPaymentUrl($options)));
    }

    public function handleCancel($redirect_url)
    {
        if ($this->order->isPaid()) {
            throw new HttpNotFoundException();
        }

        $this->order->status = Order::STATUS_CART;

        return new PaymentFailResponse(new RedirectResponse($redirect_url));
    }

    public function handleListener(Request $request, PaymentOptions $options)
    {
        /*
         * @link https://doc.soisy.it/it/Integrazione/Callback.html
         */

        $logger = $this->getContainer()->get('logger');

        if ($request->getMethod() !== 'POST') {
            $logger->warning(
                "[Soisy] Listener: HTTP Method '{$request->getMethod()}' not allowed."
            );

            throw new HttpException(405); // method not allowed
        }

        /** @var Order $order */
        $orderId = (int) $request->query->get('o');
        $order = Order::find($orderId);

        if (!$order) {
            $logger->warning(
                "[Soisy] Listener: Order '{$orderId}' not found!"
            );

            throw new HttpNotFoundException();
        }

        $this->setOrder($order);

        $orderReference = $request->post->get('orderReference');

        if ($order->getId() !== (int) $orderReference) {
            $logger->warning(
                "[Soisy] Listener: Parameter 'orderReference' ({$orderReference}) must be equal to '{$orderId}'."
            );

            throw new HttpNotFoundException();
        }

        $orderDetails = json_decode($order->payment_details, true);

        if (empty($orderDetails['token'])) {
            $logger->warning(
                "[Soisy] Listener: The order {$orderId} was never handled."
            );

            throw new HttpNotFoundException();
        }

        $orderToken = $request->post->get('orderToken');

        if ($orderToken !== $orderDetails['token']) {
            $logger->warning(
                "[Soisy] Listener: Parameter 'orderToken' ({$orderToken}) must be equal to '{$orderDetails['token']}'."
            );

            throw new HttpNotFoundException();
        }

        $eventId = (string) $request->post->get('eventId');
        $eventMessage = (string) $request->post->get('eventMessage');
        $eventDate = (string) $request->post->get('eventDate');

        if (!isset($orderDetails['events'])) {
            $orderDetails['events'] = [];
        }

        $orderDetails['events'][] = [
            'eventId'      => $eventId,
            'eventMessage' => $eventMessage,
            'eventDate'    => $eventDate,
        ];

        $logger->info(
            "[Soisy] Listener: Event '{$eventId}' for order '{$orderId}'",
            [
                'eventId'      => $eventId,
                'eventMessage' => $eventMessage,
                'eventDate'    => $eventDate,
            ]
        );

        $response = new ListenerResponse();

        switch ($eventId) {
            case 'LoanWasApproved': // Richiesta approvata
            case 'RequestCompleted': // Richiesta completata
                // nothing to do

                break;

            case 'LoanWasVerified': // In attesa di finanziamento
            case 'LoanWasDisbursed': // Finanziato
                if ($order->status === Order::STATUS_PAYMENT || $order->status === Order::STATUS_PAYMENT_LOCKED) {
                    $order->status = Order::STATUS_ORDER;

                    if (!$order->isPaid()) {
                        $order->paid_at = new DateTime();

                        $response->confirmOrder(true);
                    }
                }

                break;

            case 'UserWasRejected': // Annullato
                $order->status = Order::STATUS_PAYMENT_FAILED;

                $options = $order->payment->getHandlerOptions();
                if ($options['notify_on_fail']) {
                    $response->setSendFailedNotification(true);
                }

                break;

            default:
                $logger->error("[Soisy] Listener error: Unknown event '{$eventId}'");
        }

        return $response;
    }

    public function handleTicket()
    {
        return new TicketResponse();
    }

    /**
     * {@inheritdoc}
     */
    public static function getFormFieldsConfig(FormConfig $config, Translator $translator, array $options = [])
    {
        return [
            'shop_id'        => [
                'class' => TextField::class,
                'label' => $translator->trans('Shop ID'),
            ],
            'auth_token'     => [
                'class' => TextField::class,
                'label' => $translator->trans('Authentication Token'),
            ],
//            'instalments'    => [
//                'class'      => NumberField::class,
//                'label'      => $translator->trans('Instalments'),
//                'attributes' => ['min' => 3, 'max' => 60],
//            ],
            'notify_on_fail' => [
                'class'       => CheckboxField::class,
                'label'       => $translator->trans('Payment failed'),
                'description' => $translator->trans('Send a notification when the payment fails'),
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public static function getDefaultOptions()
    {
        return [
            'shop_id'        => '',
            'auth_token'     => '',
            'instalments'    => 5,
            'notify_on_fail' => false,
        ];
    }

    /**
     * @return string
     */
    private function getPaymentUrl(array $options = [])
    {
        /*
         * @link https://doc.soisy.it/it/Integrazione/Ordine/Crea_un_ordine.html
         */

        $order = $this->getOrder();

        if ($options['shop_id'] == '') {
            throw new OutOfBoundsException("[Soisy] Parameter 'shop_id' is required.");
        }

        if ($options['auth_token'] == '') {
            throw new OutOfBoundsException("[Soisy] Parameter 'auth_token' is required.");
        }

        if ($this->isDebug()) {
            $url = "https://api.sandbox.soisy.it/api/shops/{$options['shop_id']}/orders";
        } else {
            $url = "https://api.soisy.it/api/shops/{$options['shop_id']}/orders";
        }

        $instalments = 10;
        if ($order->total() > 1000) {
            $instalments = 15;
        }

        $headers = [
            'Accept'       => 'application/json, */*',
            'Content-Type' => 'application/json',
            'X-Auth-Token' => $options['auth_token'],
        ];

        $body = [
            'email'            => $order->getCustomerEmail(),
            'firstname'        => $order->getCustomerFirstName(),
            'lastname'         => $order->getCustomerLastName(),
            'amount'           => $this->castAmount($order->total()),
            'instalments'      => $instalments,
            'vatId'            => '',
            'vatCountry'       => '',
            'fiscalCode'       => '',
            'mobilePhone'      => '',
            'city'             => $order->invoice_city,
            'province'         => $this->getProvinceCode($order->invoice_province_id),
            'address'          => $order->invoice_address1,
            // 'civicNumber'      => '',
            'postalCode'       => $order->invoice_postcode,
            'zeroInterestRate' => true,
            'orderReference'   => $order->getId(),
        ];

        if (array_key_exists('ticket_url', $options)) {
            $body['successUrl'] = $options['ticket_url'];
        }

        if (array_key_exists('cancel_url', $options)) {
            $body['errorUrl'] = $options['cancel_url'];
        }

        if (array_key_exists('listener_url', $options)) {
            $body['callbackUrl'] = $options['listener_url'];
        }

        $client = new Client();

        $response = $client->post($url, [
            'headers'     => $headers,
            'http_errors' => false,
            'json'        => $body,
        ]);

        if ($response->getStatusCode() < 200 && $response->getStatusCode() > 299) {
            throw new RuntimeException(
                "[Soisy] Create a new order request error: Unexpected status code '{$response->getStatusCode()}'."
            );
        }

        $result = json_decode($response->getBody()->getContents(), true);

        if (!isset($result['token']) || !isset($result['redirectUrl'])) {
            throw new RuntimeException(
                "[Soisy] Create a new order request error: No parameters 'token' and 'redirectUrl' found."
            );
        }

        $order->payment_details = json_encode([
            'token' => $result['token'],
        ]);

        $order->save();

        return $result['redirectUrl'];
    }

    /**
     * @param float $amount
     * @return int
     */
    private function castAmount($amount)
    {
        return round($amount * 100);
    }

    /**
     * @param null|int $provinceId
     * @return string
     */
    private function getProvinceCode($provinceId)
    {
        if ($provinceId) {
            /** @var Province $province */
            $province = Province::find($provinceId);

            if ($province) {
                return $province->plate;
            }
        }

        return '';
    }
}
