<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Payment\PayPal;

use Application\Showcase\Model\Order;
use Application\Showcase\Payment\BaseListener;
use Application\Showcase\Payment\ListenerResponse;
use Application\Showcase\Payment\PaymentOptions;
use Pongho\Http\Request;
use Pongho\Sdk\PayPal\Utilities;
use Pongho\Utilities\DateTime;

/**
 * Gestisce le notifiche immediate di pagamento (IPN) provenienti da PayPal.
 */
class Listener extends BaseListener
{
    /**
     * {@inheritdoc}
     */
    public function handle(Request $request, PaymentOptions $options)
    {
        if ($this->isValidTransaction()) {
            $this->order->payment_details = serialize(['transaction' => $_POST['txn_id']]);
            $this->order->status = Order::STATUS_ORDER;
            $this->order->paid_at = new DateTime();

            return new ListenerResponse(true);
        }

        throw new \RuntimeException('This request is not a valid PayPal transaction!');
    }

    /**
     * Verifica la validità della transazione.
     *
     * @return bool
     */
    protected function isValidTransaction()
    {
        return ($this->isVerifiedIPN() && $this->isPrimaryPayPalEmail() && $this->isNotProcessed() && $this->isVerifiedAmount() && $this->isCompleted());
    }

    /**
     * Verifica che la richiesta provenga effettivamente da PayPal.
     *
     * @return bool
     * @throws \RuntimeException
     */
    protected function isVerifiedIPN()
    {
        if ($this->isDebug()) {
            $paypal_url = 'https://www.sandbox.paypal.com/cgi-bin/webscr';
        } else {
            $paypal_url = 'https://www.paypal.com/cgi-bin/webscr';
        }

        $handle = curl_init($paypal_url);

        if ($handle === false) {
            throw new \RuntimeException('Can’t connect to PayPal to validate IPN message.');
        }

        $req = 'cmd=_notify-validate&' . http_build_query($_POST, '', '&');

        curl_setopt($handle, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1);
        curl_setopt($handle, CURLOPT_POST, 1);
        curl_setopt($handle, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($handle, CURLOPT_POSTFIELDS, $req);
        curl_setopt($handle, CURLOPT_SSL_VERIFYPEER, 1);
        curl_setopt($handle, CURLOPT_SSL_VERIFYHOST, 2);
        curl_setopt($handle, CURLOPT_FORBID_REUSE, 1);

        // Set TCP timeout to 30 seconds
        curl_setopt($handle, CURLOPT_CONNECTTIMEOUT, 30);
        curl_setopt($handle, CURLOPT_HTTPHEADER, ['Connection: Close']);

        $response = curl_exec($handle);

        if (curl_errno($handle) != 0) {
            // cURL error
            $error = 'Can’t connect to PayPal to validate IPN message: ' . curl_error($handle);

            curl_close($handle);

            throw new \RuntimeException($error);
        }

        curl_close($handle);

        if (strcmp($response, 'VERIFIED') == 0) {
            return true;
        }

        if (strcmp($response, 'INVALID') == 0) {
            return false;
        }

        throw new \RuntimeException('Invalid HTTP response from PayPal');
    }

    /**
     * Verifica la corrispondenza dell’indirizzo email.
     *
     * @return bool
     * @throws \RuntimeException
     */
    protected function isPrimaryPayPalEmail()
    {
        $options = $this->order->payment->getHandlerOptions();

        if (isset($_POST['receiver_email']) && trim((string) $_POST['receiver_email']) === $options['email']) {
            return true;
        }

        throw new \RuntimeException(sprintf('Fail %s.', __METHOD__));
    }

    /**
     * Controlla se l’ordine è già stato processato.
     *
     * @return bool
     * @throws \RuntimeException
     */
    protected function isNotProcessed()
    {
        $details = unserialize($this->order->payment_details);
        $transaction = is_array($details) && isset($details['transaction']) ? $details['transaction'] : '';

        if (isset($_POST['txn_id']) && trim((string) $_POST['txn_id']) !== $transaction) {
            return true;
        }

        throw new \RuntimeException(sprintf('Fail %s.', __METHOD__));
    }

    /**
     * Verifica la corrispondenza dell’importo dell’ordine.
     *
     * @return bool
     * @throws \RuntimeException
     */
    protected function isVerifiedAmount()
    {
        if (isset($_POST['mc_gross']) && trim((string) $_POST['mc_gross']) === Utilities::formatPrice($this->order->total)) {
            return true;
        }

        throw new \RuntimeException(sprintf('Fail %s.', __METHOD__));
    }

    /**
     * Verifica che lo stato del pagameto sia `Completed`.
     *
     * @return bool
     * @throws \RuntimeException
     */
    protected function isCompleted()
    {
        if (isset($_POST['payment_status']) && trim((string) $_POST['payment_status']) === 'Completed') {
            return true;
        }

        throw new \RuntimeException(sprintf('Fail %s.', __METHOD__));
    }
}
