<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Payment\PagOnline;

use Application\Admin\Form\FormConfig;
use Application\Core\I18n\Translation\Translator;
use Application\Showcase\Model\Order;
use Application\Showcase\Payment\BasePayment;
use Application\Showcase\Payment\PagOnline\Exception\InitException;
use Application\Showcase\Payment\PagOnline\Exception\VerifyException;
use Application\Showcase\Payment\PaymentOptions;
use Application\Showcase\Payment\PaymentResponse;
use Application\Showcase\Payment\TicketResponse;
use Pongho\Core\TemplateResponse;
use Pongho\Form\Field\RadioField;
use Pongho\Form\Field\TextField;
use Pongho\Http\RedirectResponse;
use Pongho\Http\Request;
use Pongho\Http\Response;
use Pongho\Sdk\PagOnline\PaymentInitRequest;
use Psr\Log\LogLevel;

/**
 * Gestisce il pagamento tramite PagOnline di UniCredit.
 */
class Payment extends BasePayment
{
    /**
     * {@inheritdoc}
     */
    public function preparePayment()
    {
        $this->getLogger()->debug(
            '[PagOnline] Prepare order.',
            [
                'order'       => $this->order->getId(),
                'prev_status' => $this->order->status,
                'new_status'  => Order::STATUS_PAYMENT_LOCKED,
            ]
        );

        $this->order->convertCartToOrder(Order::STATUS_PAYMENT_LOCKED);

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function handlePayment(PaymentOptions $options)
    {
        return new PaymentResponse(new RedirectResponse($this->getPaymentUrl($options)));
    }

    /**
     * {@inheritdoc}
     */
    public function handleCancel($redirect_url)
    {
        $this->getLogger()
            ->error('Caso non trattato per insufficienza della documentazione della banca.');

        return $this->createErrorResponse();
    }

    /**
     * {@inheritdoc}
     *
     * @todo Provare il parametro callbackURL quando la banca avrà cuore di implementarlo.
     */
    public function handleListener(Request $request, PaymentOptions $options)
    {
        return null;
    }

    /**
     * {@inheritdoc}
     */
    public function handleTicket()
    {
        $order = $this->getOrder();
        $options = $order->payment ? $order->payment->getHandlerOptions() : [];

        try {
            $verify = $this->verify($options);

            $this->getLogger()->debug(
                '[PagOnline] Verify: ' . $verify->rc,
                [
                    'order'   => $order->getId(),
                    'options' => $options,
                    'verify'  => [
                        'Error'      => $verify->error,
                        'Rc'         => $verify->rc,
                        'ErrorDesc'  => $verify->errorDesc,
                        'TranID'     => $verify->tranID,
                        'AuthCode'   => $verify->authCode,
                        'EnrStatus'  => $verify->enrStatus,
                        'AuthStatus' => $verify->authStatus,
                        'Brand'      => $verify->brand,
                    ],
                ]
            );
        } catch (VerifyException $verifyException) {
            $log_message = '';
            $log_error = false;
            $http_response = null;

            $verify = $verifyException->getVerify();

            // Gestisco i casi noti
            switch ($verify->rc) {
                case 'IGFS_20026': // SHOP ID DUPLICATO
                    /**
                     * Si verifica quando il campo 'shopID', che rappresenta l’ID della transazione, è già stato usato.
                     * Non mi è chiaro come questo accada, eppure è successo.
                     *
                     * Se il cliente ritorna al carrello e lo ritrova com’era prima, la banca continuerebbe a segnalare
                     * l’errore: quindi tanto vale lasciare il carrello nel limbo del pagamento in corso anche se
                     * l’utente sarà costretto a ricreare il carrello.
                     */
                    $log_message = 'Transazione duplicata, segnalo l’errore al cliente e NON annullo il pagamento in corso, altrimenti l’errore si ripresenterebbe ancora.';

                    break;

                case 'IGFS_20090': // Transazione annullata dall’utente
                    $order->cancelPaymentTransaction();

                    $log_message = 'Transazione annullata dall’utente, si torna al carrello.';
                    $http_response = new RedirectResponse('/shop/cart/');

                    break;

                default:
                    $messagesToDebug = [
                        'IGFS_002', // CARTA SCADUTA
                        'IGFS_104', // CARTA SOGGETTA A RESTRIZIONI
                        'IGFS_007', // TIMEOUT
                        'IGFS_008', // AUTORIZZAZIONE NEGATA
                        'IGFS_020', // CARTA INVALIDA
                        'IGFS_030', // FONDI INSUFFICIENTI
                        'IGFS_104', // CARTA SOGGETTA A RESTRIZIONI
                        'IGFS_118', // CONTO NON TROVATO O NON ABILITATO
                        'IGFS_902', // TRANSAZIONE NON VALIDA
                        'IGFS_1921', // 3DS:UNABLE TO AUTENTICATE
                        'IGFS_1922', // 3DS:AUTENTICATION DENIED
                        'IGFS_1923', // 3DS:UNABLE TO VERIFY (VERES=U)
                    ];

                    /**
                     * Altri casi noti, ma che non ho idea di come trattare,
                     * quindi continuo a fare il log con il livello di ERROR
                     *
                     * - IGFS_005: ERRORE DI FORMATO
                     * - IGFS_086: MALFUNZIONAMENTO SISTEMA
                     * - IGFS_089: PAYMENT INSTRUMENT ERROR
                     * - IGFS_800: TERMINALE NON ABILITATO
                     * - IGFS_814: TRANSAZIONE IN CORSO
                     * - IGFS_20007: STATO ORDINE NON VALIDO
                     * - IGFS_20092: BUYERMOBILE ERRATO
                     */

                    $order->cancelPaymentTransaction();
                    $log_error = !in_array($verify->rc, $messagesToDebug);
                    break;
            }

            if ($log_error) {
                $this->getLogger()->error(
                    '[PagOnline] Caso non trattato per insufficienza della documentazione della banca.',
                    [
                        'order'     => $order->getId(),
                        'options'   => $options,
                        'verify'    => [
                            'Error'      => $verify->error,
                            'Rc'         => $verify->rc,
                            'ErrorDesc'  => $verify->errorDesc,
                            'TranID'     => $verify->tranID,
                            'AuthCode'   => $verify->authCode,
                            'EnrStatus'  => $verify->enrStatus,
                            'AuthStatus' => $verify->authStatus,
                            'Brand'      => $verify->brand,
                        ],
                        'exception' => $verifyException,
                    ]
                );
            } else {
                $this->getLogger()->debug(
                    '[PagOnline] ' . $log_message,
                    [
                        'order'   => $order->getId(),
                        'options' => $options,
                        'verify'  => [
                            'Error'      => $verify->error,
                            'Rc'         => $verify->rc,
                            'ErrorDesc'  => $verify->errorDesc,
                            'TranID'     => $verify->tranID,
                            'AuthCode'   => $verify->authCode,
                            'EnrStatus'  => $verify->enrStatus,
                            'AuthStatus' => $verify->authStatus,
                            'Brand'      => $verify->brand,
                        ],
                    ]
                );
            }

            if (!$http_response instanceof Response) {
                $http_response = $this->createErrorResponse(
                    $verifyException->getMessage()
                );
            }

            return new TicketResponse(false, $http_response);
        }

        if ($options['transaction'] === PaymentInitRequest::TRANSACTION_AUTH) {
            $order->status = Order::STATUS_PAYMENT_AUTHORIZED;
        } else {
            $order->status = Order::STATUS_ORDER;
        }

        $order->save();

        return new TicketResponse(true);
    }

    /**
     * {@inheritdoc}
     */
    public static function getFormFieldsConfig(FormConfig $config, Translator $translator, array $options = [])
    {
        return [
            'terminal_id' => [
                'class' => TextField::class,
                'label' => 'Terminal ID',
            ],
            'api_key'     => [
                'class' => TextField::class,
                'label' => 'API Key (kSig)',
            ],
            'user_id'     => [
                'class' => TextField::class,
                'label' => 'User ID',
            ],
            'transaction' => [
                'class'   => RadioField::class,
                'label'   => $translator->trans('Transaction mode'),
                'options' => [
                    'PURCHASE' => $translator->trans('Instant purchase'),
                    'AUTH'     => $translator->trans('Authorized purchase'),
                ],
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public static function getDefaultOptions()
    {
        return [
            'terminal_id' => '',
            'api_key'     => '',
            'user_id'     => '',
            'transaction' => PaymentInitRequest::TRANSACTION_AUTH,
        ];
    }

    /**
     * @return \Psr\Log\LoggerInterface
     */
    protected function getLogger()
    {
        return $this->getContainer()->get('logger');
    }

    /**
     * @return string
     */
    protected function getPaymentUrl(PaymentOptions $options)
    {
        $order = $this->getOrder();
        $options = $options->get($order);
        $init = $this->init($options);

        $order->payment_details = json_encode(
            [
                'payment_id' => $init->paymentID,
            ]
        );

        $order->save();

        return $init->redirectURL;
    }

    /**
     * @return \IgfsCgInit
     * @throws InitException
     */
    protected function init(array $options)
    {
        $order = $this->getOrder();

        require ROOT_PATH . '/pongho/src/Pongho/Sdk/PagOnline/IGFS_CG_API/init/IgfsCgInit.php';

        $init = new \IgfsCgInit();

        // Configurazione della libreria
        $init->installPath = ROOT_PATH . '/pongho/src/Pongho/Sdk/PagOnline';

        if ($this->isDebug()) {
            $init->disableCheckSSLCert();
        }

        // Preparo la richiesta
        $init->serverURL = $this->getServerUrl();
        $init->timeout = 15000;
        $init->tid = $options['terminal_id'];
        $init->kSig = $options['api_key'];

        if ($this->isDebug()) {
            $init->shopID = 'metaline-' . $order->getId() . '-' . $order->created_at->format('YmdHis');
        } else {
            $init->shopID = $order->getId();
        }

        $init->shopUserRef = $order->getCustomerEmail();
        $init->shopUserName = $this->compileShopUserName(
            $order->getCustomerFirstName(),
            $order->getCustomerLastName()
        );

        $init->amount = round($order->total() * 100);
        $init->trType = $options['transaction'];
        $init->currencyCode = 'EUR';
        $init->langID = 'IT';

        $init->notifyURL = $options['ticket_url'];
        $init->errorURL = $options['cancel_url'];
        //$init->callbackURL = $options['listener_url'];

        $result = $init->execute();

        $this->getLogger()->log(
            $result ? LogLevel::DEBUG : LogLevel::ERROR,
            '[PagOnline] Payment URL: ' . $init->redirectURL,
            [
                'order'   => $order->getId(),
                'options' => $options,
                'init'    => [
                    'serverURL'    => $init->serverURL,
                    'timeout'      => $init->timeout,
                    'tid'          => $init->tid,
                    'kSig'         => $init->kSig,
                    'shopID'       => $init->shopID,
                    'shopUserRef'  => $init->shopUserRef,
                    'shopUserName' => $init->shopUserName,
                    'amount'       => $init->amount,
                    'trType'       => $init->trType,
                    'currencyCode' => $init->currencyCode,
                    'langID'       => $init->langID,
                    'notifyURL'    => $init->notifyURL,
                    'Error'        => $init->error,
                    'Rc'           => $init->rc,
                    'ErrorDesc'    => $init->errorDesc,
                    'PaymentID'    => $init->paymentID,
                    'RedirectURL'  => $init->redirectURL,
                ],
            ]
        );

        if ($result) {
            return $init;
        }

        $e = new InitException($init);

        $this->getLogger()->warning($e->getMessage());

        throw $e;
    }

    /**
     * @return \IgfsCgVerify
     * @throws VerifyException
     */
    protected function verify(array $options)
    {
        $order = $this->getOrder();
        $payment_details = json_decode($order->payment_details, true);

        require ROOT_PATH . '/pongho/src/Pongho/Sdk/PagOnline/IGFS_CG_API/init/IgfsCgVerify.php';

        $verify = new \IgfsCgVerify();

        // Configurazione della libreria
        $verify->installPath = ROOT_PATH . '/pongho/src/Pongho/Sdk/PagOnline';

        if ($this->isDebug()) {
            $verify->disableCheckSSLCert();
        }

        // Preparo la richiesta
        $verify->serverURL = $this->getServerUrl();
        $verify->timeout = 15000;
        $verify->tid = $options['terminal_id'];
        $verify->kSig = $options['api_key'];

        if ($this->isDebug()) {
            $verify->shopID = 'metaline-' . $order->getId() . '-' . $order->created_at->format('YmdHis');
        } else {
            $verify->shopID = $order->getId();
        }

        $verify->paymentID = $payment_details['payment_id'];

        if ($verify->execute()) {
            return $verify;
        }

        throw new VerifyException($verify);
    }

    /**
     * @return string
     */
    protected function getServerUrl()
    {
        if ($this->isDebug()) {
            return 'https://testuni.netsw.it/UNI_CG_SERVICES/services/PaymentInitGatewayPort?wsdl';
        }

        return 'https://pagamenti.unicredit.it/UNI_CG_SERVICES/services/PaymentInitGatewayPort?wsdl';
    }

    /**
     * @param string $error
     * @return TemplateResponse
     */
    protected function createErrorResponse($error = null)
    {
        $message = 'Si è verificato un errore durante il pagamento.';

        if ($error) {
            $message .= '<br><br><span style="color: red;">' . $error . '</span>';
        }

        $message .= '<br><br><a href="' . url('/shop/cart/') . '" class="btn">Torna al carrello</a>';

        $view = $this->getContainer()->get('theme_view');
        $view
            ->setTemplate('message.php')
            ->assignVars(
                [
                    'message_type' => 'alert',
                    'title'        => 'Attenzione',
                    'message'      => $message,
                ]
            );

        return new TemplateResponse($view);
    }

    /**
     * @param string $firstName
     * @param string $lastName
     * @return string
     */
    private function compileShopUserName($firstName, $lastName)
    {
        return $this->filterForShopUserNameField($firstName) . ',' . $this->filterForShopUserNameField($lastName);
    }

    /**
     * @param string $string
     * @return string
     */
    private function filterForShopUserNameField($string)
    {
        $map = [
            'à' => 'a',
            'è' => 'e',
            'é' => 'e',
            'ì' => 'i',
            'ò' => 'o',
            'ù' => 'u',
        ];

        $string = str_replace(array_keys($map), array_values($map), $string);

        return preg_replace('/[^a-zA-Z]+/', '', $string);
    }
}
